use Utf8Checker;
use Db::Connection;
use Storage::Storage;
use Logging;
use MigrationDumpStatus;
use CompatArgParse;
use Encoding;
use Storage::FileStorage;
#==============================================================
#
# common subroutines
#

sub printToLog {
  Logging::info(shift);
}

sub printToError {
  Logging::error(shift);
}

#BEGINNOTINRAQ2

sub makeFileReader
{
	my ($filename, $doNotFail) = @_;
	local (*INPUT);

	unless (-T $filename) {

		if (defined($doNotFail)) {
			&printToLog("Error: file '$filename' is not a text file");
		} else {
			&printToError("Error: file '$filename' is not a text file");
		}

		return undef;
	}

	unless (open(INPUT,"<$filename")) {

		binmode INPUT;
		if (defined($doNotFail)) {
			&printToLog("Error: unable to open file '$filename' to read");
		} else {
			&printToError("Error: unable to open file '$filename' to read");
		}

		return undef;
	}

	my $input = *INPUT;

	my $this = {
		'READ' => sub {
				my $a = <$input>;
				return $a;
			},

		'CLOSE' => sub { close($input); }

	};

	return $this;
}

sub makeArrayReader {
  my (@arr) = @_;
  my $size= 0;
  my $count = scalar(@arr);

  my $this={'READ'=>sub {
			  if($count == $size) {
				return undef;
			  }

			  return $arr[$count++];
			},
			'CLOSE'=>sub {
			}
		   };
  return $this;
}

sub makeParser {
  my $reader;
  my ($this,$ptrRows,$keySep,$valSep,$valKeySep,$comment,
      $typeRows,$maxIndex,$rowIndex);
  {
	$reader = shift;
	unless(defined($reader)) {
	  return undef;
	}

    my %args=(@_);
    $keySep = $args{'KEYSEPARATOR'};
    $valSep = $args{'VALUESEPARATOR'};
    $valKeySep = $args{'KEYVALUESEPARATOR'};
  }
  $this={'READER'=>sub {
		   if(@_){
			 $reader=shift;
		   }
		   return $reader;
		 },
		 'FILE'=>sub {
		   if(@_){
			 $reader = makeFileReader(@_);
		   }
		   return undef;
		 },
		 'KEYSEPARATOR'=>sub{
		   if(@_){
			 $keySep=shift;
		   }
		   return $keySep;
		 },
		 'VALUESEPARATOR'=>sub{
		   if(@_){
			 $valSep=shift;
		   }
		   return $valSep;
		 },
		 'KEYVALUESEPARATOR'=>sub{
		   if(@_){
			 $valKeySep=shift;
		   }
		   return $valKeySep;
		 },
		 'COMMENT'=>sub{
		   if(@_){
			 $comment=shift;
		   }
		   return $comment;
		 },
		 'PARSE'=>sub{
		   my %args=(@_);

		   $keySep=$args{'KEYSEPARATOR'} if exists $args{'KEYSEPARATOR'};
		   $valSep=$args{'VALUESEPARATOR'} if exists $args{'VALUESEPARATOR'};
		   $valKeySep=$args{'KEYVALUESEPARATOR'} if exists $args{'KEYVALUESEPARATOR'};
		   $comment=$args{'COMMENT'} if exists $args{'COMMENT'};

		   $ptrRows = &parseFile('reader'=>$reader,'keysep'=>$keySep,
								 'valsep'=>$valSep,'keyvalsep'=>$valKeySep,
								 'comment'=>$comment);
		   $typeRows = ref($ptrRows);
		   if($typeRows=~/ARRAY/){
			 $rowIndex=-1;
			 $maxIndex=scalar(@{$ptrRows})-1;
		   }
		   return $ptrRows;
		 },
		 'ROW'=>sub{
		   if($typeRows=~/HASH/){
			 my $key=shift;
			 return $ptrRows->{$key};
		   }elsif($typeRows=~/ARRAY/){
			 if($rowIndex<$maxIndex){
			   $rowIndex++;
			   return $ptrRows->[$rowIndex];
			 }else{
			   return undef;
			 }
		   }else{
			 return undef;
		   }
		 },
		};

  return $this;
}

sub makeFileParser {
  my $filename = shift;
  return makeParser(makeFileReader($filename), @_);
}

sub makeSafeFileParser {
  my $filename = shift;
  return makeParser(makeFileReader($filename, 'doNotFail'), @_);
}

sub makeStringParser {
  my $content = shift;
  return makeParser(makeArrayReader($content), @_);
}

sub parseFile {
  my %arg = (@_);
  my $reader = $arg{'reader'};
  my $keySep = $arg{'keysep'};
  my $valSep = $arg{'valsep'};
  my $valKeySep = $arg{'keyvalsep'};
  my $comment = $arg{'comment'};

  my ($ret,$key,$value,$reKey,$reValue,@values,$keyBlank,$valBlank,
     $vkey,$vvalue,$reKeyValue,$reComment);

  if($keySep){
    $ret = {};
    if(ref($keySep)=~/regexp/i || ($keySep eq ' ')){
      $reKey = $keySep;
    }else{
      $reKey = qr/$keySep/;
    }
  }else{
    $ret = [];
  }
  if($valSep){
    if(ref($valSep)=~/regexp/i || ($valSep eq ' ')){
      $reValue = $valSep;
    }else{
      $reValue = qr/$valSep/;
    }
  }
  if($valKeySep){
    if(ref($valKeySep)=~/regexp/i || ($valKeySep eq ' ')){
      $reKeyValue = $valKeySep;
    }else{
      $reKeyValue = qr/$valKeySep/;
    }
  }
  if($comment){
    if(ref($comment)=~/regexp/i){
      $reComment = $comment;
    }else{
      $reComment = qr/$comment/;
    }
  }

  while($_ = $reader->{'READ'}->()){
    chomp;
    next unless $_;
    if($comment && /$reComment/){
	  next;
    }
    if ($keySep){
      ($key,$value)=split($reKey,$_,2);
      if($key){
		$key=~s/^\s+//;
		$key=~s/\s+$//;
		if($valSep){

		  if ($valKeySep){
			$ret->{$key}={};
			foreach $value (split($reValue,$value)){
			  ($vkey,$vvalue)=split($reKeyValue,$value);
			  $vkey=~s/^\s+//;
			  $vkey=~s/\s+$//;
			  $ret->{$key}->{$value}=$vvalue;
			}
		  }else{
			push @{$ret->{$key}},split($reValue,$value);
		  }
		}else{
		  $value=~s/^\s+//;
		  $value=~s/\s+$//;
		  $ret->{$key}=$value;
		}
      }
    }else{
      if($valSep){
		push @{$ret},[split($valSep,$_)];
      }else{
		push @{$ret},$_;
      }
    }
  }

  $reader->{'CLOSE'}->();

  return $ret;
}

#ENDNOTINRAQ2

sub makePasswordNode {
  my ($password, $type) = @_;

  if (!defined $password) {
        printToError("'undef' password passed to makePasswordNode. Set to empty");
        $password = '';
  }

  if (!$password) {
	$type = 'plain';
  } else {
	if ($type ne 'plain') {
	  $type = 'encrypted';
	}
  }

  return XmlNode->new('password',
				 'content' => $password,
				 'attributes' => {'type' => $type});
}

sub mangleName {
  my $name = shift;

  if($name =~ /^\d/) {
    $name = 'user' . $name;
  }
  $name =~ s/\./_/g;

  return $name;
}

sub makeSysUserNode {
  my ($name, $password, $type) = @_;

  unless(defined($name) && defined($password) && defined($type)) {
	printToError("makeSysUserNode: Username, password or password type is not set.");
  }

  my $sysUserNode = XmlNode->new('sysuser');
  $sysUserNode->{'ATTRIBUTE'}->('name', mangleName($name));
  $sysUserNode->{'ADDCHILD'}->(makePasswordNode($password, $type));

  return $sysUserNode;
}

# Returns home directory for the system account $user
sub getHomeDir {
	my ($user) = @_;
	my $home;
	my @pw = getpwnam($user);
	if (-1 != $#pw) {
		$home = $pw[7];
	}
	unless ($home) {
		# how it is by default
		$home = '/home/$account';
	}
	return $home;
}

# Returns password for system account $user
sub getSystemPassword {
	my ($user) = @_;
	my $password;
	my @pw = getpwnam($user);
	if (-1 != $#pw) {
		$password = $pw[1];
	}
	unless ($password) {
		$password = '';
	}
	return $password;
}

sub createUserNameFromDomain {
  my ($name) = @_;

  if ($name =~ /^\d/) {
	$name = 'u' . $name;
  }

  $name =~ s/\./_/g;
  $name =~ s/^www\.//g;
  $name =~ s/[aoeui]//g if length($name) > 16;
  $name = substr($name, 0, 16) if length($name) > 16;

  return $name;
}

my $compress = 1;

sub setCompress {
  ($compress) = @_;
}

#FIXME: HACK
my $workDir;

sub initStorage {
  my ($splitsize, $outputFilename) = @_;   # new param $outputFilename

  my $compress = !(exists $ptrArgs->{'no-compress'});

  $workDir = `pwd`;
  chomp $workDir;
  $workDir .= "/";

  if (! $outputFilename) {
	if (exists $ptrArgs->{'output'}) {
		$storage = Storage::Storage::createMimeStorage($compress, $ptrArgs->{'output'}, $splitsize);
	} else {
		$storage = Storage::Storage::createFileStorage($compress, $workDir, $splitsize);
	}
  } else {
	if (exists $ptrArgs->{'output'}) {
                $storage = Storage::Storage::createMimeStorage($compress, $ptrArgs->{'output'}."/".$outputFilename, $splitsize);
	} else {
                $storage = Storage::Storage::createFileStorage($compress, $workDir."/".$outputFilename, $splitsize);
        }
  }
}

#
# Compatibility wrapper
#

sub makeDumpDb {
  my ($dbName, $dbType, $dbUser, $dbPassword, $dbHost, $ptrDomainBin, $variables, $domainSocket, $plesk_7) = @_;

  #??
  return if exists $ptrArgs->{'no-content'} or exists $ptrArgs->{'configuration-only'};

  my %params;

  if ($dbType eq 'Pg') { $dbType = 'postgresql'; }

  $params{'name'} = $dbName;
  $params{'user'} = $dbUser;
  $params{'type'} = $dbType;
  $params{'password'} = $dbPassword;
  $params{'host'} = $dbHost if $dbHost;
  $params{'socket'} = $domainSocket if $domainSocket;
  $params{'preload_dirs'} = $variables if $variables;

  my %testparams = %params;

  if ($dbType eq 'Pg') {
    $testparams{'name'} = 'template1';
    # flag to use pg_manage utility for adding users to db group in plesk 7, bug 86423
    $params{'plesk_7'} = $plesk_7 if $plesk_7;
  }

  # check db existence, bug 83616
  my $connection = Db::Connection::getConnection(%testparams);
  my $db_exist = $connection->execute_rownum($dbType eq 'postgresql' ? "SELECT 1 FROM pg_database WHERE datname = '$dbName'" : 'SELECT 1');
  $connection->finish();
  $connection->disconnect();

  return $storage->addDb("$dbName.$dbType.sql", %params) if $db_exist;
  
  Logging::error("Database $dbName is not accessible for dumping");

  return undef;
}

sub addDbUsers {
  my ($dbNode, $dbName, $wrapUserMysql) = @_;

  printToLog("Dupming $dbName database users");

  $dbName =~ s/_/\\_/g;

  my (@users, $dbUser, $ptrRow);
  my $sql = "SELECT DISTINCT User FROM db WHERE Db = '$dbName'";

  if ($wrapUserMysql->{'EXECUTE'}->($sql) == 0) { # bug 94641
	my $sql = "SELECT DISTINCT User FROM db WHERE Db LIKE '$dbName'";
	$wrapUserMysql->{'FINISH'}->();
	$wrapUserMysql->{'EXECUTE'}->($sql);
  }

  while ($ptrRow = $wrapUserMysql->{'FETCHROW'}->()) {
		push @users, (@{$ptrRow})[0];
  }
  $wrapUserMysql->{'FINISH'}->();

  foreach $dbUser (@users) {
		addDbUser($dbNode, $dbUser, $wrapUserMysql);
  }
}

sub addDbUser {
  my ($dbNode, $user, $wrapUserMysql, $passwordNode, $accessHosts) = @_;

  &printToLog("Dumping mysql user '$user'");

	my ( $item );

  if ( ref( $wrapUserMysql ) =~ /HASH/ &&
			 ref( $wrapUserMysql->{'EXECUTE'} ) =~ /CODE/ ) {

    my ($sql,$ptrRow,@accesshosts,$accesshost);

    $sql = "SELECT password,host FROM user WHERE user='$user'";

    if ( $wrapUserMysql->{'EXECUTE'}->( $sql ) ) {

			$item = XmlNode->new('dbuser', 'attributes' => {'name' => $user} );

      while ( $ptrRow = $wrapUserMysql->{'FETCHROW'}->() ) {
				unless ( ref($passwordNode) =~ /HASH/ ) {
					$passwordNode = makePasswordNode( $ptrRow->[0] );
				}
				push @accesshosts, $ptrRow->[1];
      }

			if ( $passwordNode ) {
				$item->{'ADDCHILD'}->( $passwordNode );
			}

			if (defined($accessHosts) && ref($accessHosts) =~ /ARRAY/) {
			  @accesshosts = @{$accessHosts};
			}

			foreach $accesshost ( @accesshosts ) {
				$item->{'ADDCHILD'}->( XmlNode->new( 'accesshost', 'content' => $accesshost ) );
			}

			$dbNode->{'ADDCHILD'}->( $item );

    }

    $wrapUserMysql->{'FINISH'}->();
  }

	return $item;
}

my $lastDumpId = 0;

sub generateDumpId {
  return $$ . "-" . $lastDumpId++;
}

#########################################
#
#	Dumps array given by reference into
#	temporary file.
#
#########################################
sub dumpArrayToTemporaryFile( $ )
{
	my ($arrayRef) = @_;

	use POSIX;

	my $list_filename;
	for (;;) {
		$list_filename = tmpnam();
		sysopen(TMP, $list_filename, O_RDWR | O_CREAT | O_EXCL) && last;
	}

	foreach my $line (@{$arrayRef}) {
		print TMP $line . "\n";
	}
	close TMP;

	return $list_filename;
}

#################################################
#
#	Check that given directory may be archived
#
#################################################
sub checkDirForArchive( $ )
{
	my ($srcDir) = @_;

	if (!-d $srcDir) {
		return;
	}

	# check that directory is not empty
	if (!opendir(SRCDIR, $srcDir)) {
		return;
	}

	my $filename;

	while (defined ($filename = readdir SRCDIR)) {
		next if $filename =~ /^\.\.?$/;
		# directory is not empty
		closedir(SRCDIR);
		return 1;
	}

	# directory is empty
	closedir(SRCDIR);
	return;
}

#
# FOR BACKWARD COMPATIBILITY ONLY!
# Use Storage.pm for new code.
#
# conversions:
# $ptrNames: ['.'], '.', '' -> undef, 'foobar' -> ['foobar']
# $ptrExcludes: [], '' -> undef
#

#
# sub makeDumpFile - make a dump of dir tree (use 'tar')
#
# Arguments:
#          $destFile - name of destination file.
#                      If the file has not '.tgz' or '.tar.gz' extension
#                      '.tgz' will be added
#          $scrDir   - path to source dir
#
#          $ptrNames - pointer to an array with names of files and dirs to
#                      include into dump
#                      can be a single name
#                      by default '.' - include all entries
#
#          $ptrExcludes - pointer to an array with names of files and dirs to
#                      exclude from dump
#                      can be a single name
#
#          $extraParams - extra params to run tar
#                         ('-h' - don't dump symlinks; dump the files they point to)
#

sub makeDumpFile {
  my ($destFile, $srcDir, $ptrNames, $ptrExclude, $extraParams, $user, $noRecursion) = @_;

  if (substr($destFile, 0, length($workDir)) eq $workDir) {
    $destFile = substr($destFile, length($workDir));
  }

  if (substr($destFile, 0, length("pma/")) eq "pma/") {
    $destFile = substr($destFile, length("pma/"));
  }

  return if exists $ptrArgs->{'no-content'} or exists $ptrArgs->{'configuration-only'};

  my $followSymlinks = 0;
  if (ref($extraParams) eq "ARRAY" and @{$extraParams} > 0 and $extraParams->[0] eq '-h') {
    $followSymlinks = 1;
  }

  my $include;
  if (ref($ptrNames) eq "ARRAY") {
    if (@{$ptrNames} > 0 and !(@{$ptrNames} == 1 and $ptrNames->[0] eq '.')) {
      $include = $ptrNames;
    }
  } elsif (defined $ptrNames) {
    if ($ptrNames ne '.' and $ptrNames ne '') {
      $include = [$ptrNames];
    }
  }

  my $exclude;
  if (ref($ptrExclude) eq "ARRAY") {
    if (@{$ptrExclude} > 0) {
      $exclude = $ptrExclude;
    }
  } elsif (defined $ptrExclude) {
    if ($ptrExclude ne '') {
      $exclude = [$ptrExclude];
    }
  }

  my %options;
  $options{'directory'} = $srcDir;
  $options{'include'} = $include if $include;
  $options{'exclude'} = $exclude if $exclude;
  $options{'follow_symlinks'} = $followSymlinks if $followSymlinks;
  $options{'user'} = $user if $user;
  $options{'no_recursion'} = 1 if $noRecursion;

  return $storage->addTar($destFile, %options);   
}

sub dumpFile {
  my ($destFile, $srcFile, $extraParams) = @_;
  return unless -f $srcFile;

  my @parts = split /\//, $srcFile;
  my $basename = pop @parts;
  my $dirname = join "/", @parts;

  return makeDumpFile($destFile, $dirname, $basename, undef, $extraParams);
}

sub getDumpDir {
  my ($dumpdir) = @_;

  unless(system("rm -rf $dumpdir")==0){
    &printToError("Error: can not clean temporary dir '$dumpdir': $!");
  }
  unless(-d $dumpdir){
    mkdir ($dumpdir,0755) or
      die "Error: can not create temporary dir '$dumpdir': $!\n";
  }
}

#
# FOR BACKWARD COMPATIBILITY ONLY!
# Use XmlNode.pm for new code
#
sub printXml {
  my ($root, $outFh) = @_;

  ref($root) =~ /HASH/ && ref($root->{'PRINT'}) =~ /CODE/
    or die "Error: sub printXml: bad arguments";

  print $outFh "<?xml version='1.0' encoding='UTF-8'?>\n";

  $root->{'PRINT'}->($outFh, '  ');

  if (ref($outFh) !~ /GLOB/) {
    close $outFh;
  }
  return 1;
}

#
# FOR BACKWARD COMPATIBILITY ONLY!
# Use XmlNode.pm for new code
#
#
# XmlNode("blah", "content" => "blah",
#                 "children" => [$foo, $bar],
#                 "attributes" => {"botva"=>"7", "foo"=>"bar"})
#
#sub XmlNode {
#  my ($name, %args) = @_;

#  my $node;
#  if (defined $args{'content'}) {
#	$node = makeXmlNode($name, $args{'content'});
#  } else {
#	$node = makeXmlNode($name);
#  }

#  if ($args{'attributes'}) {
#	foreach $attribute (keys %{$args{'attributes'}}) {
#	  $node->{'ATTRIBUTE'}->($attribute, $args{'attributes'}->{$attribute});
#	}
#  }

#  if ($args{'children'}) {
#	foreach $child (@{$args{'children'}}) {
#	  $node->{'ADDCHILD'}->($child);
#	}
#  }
#  return $node;
#}

#
# FOR BACKWARD COMPATIBILITY ONLY!
# Use XmlNode.pm for new code
#
sub makeXmlNode {
  my ($this,$parent,%attributes,@content);
  my ($name,$plain)=@_;

  $plain = toUtf8(encodeHtml($plain));

  unless($name||$plain){
    return undef;
  }
  $this = {
	   'ATTRIBUTE' => sub {
	     my $key = shift;
	     if(@_){
	       my $value=shift;
		   die "Undefined value of attribute '$key'" unless defined($value);
	       if(@_){
					 $attributes{$key} = $value;
	       }else{
					 $attributes{$key} = toUtf8(normalizeAttribute($value));
	       }
	     }
	     return $attributes{$key};
	   },
	   'NAME' => sub {
	     if(@_){
	       $name = shift;
	     }
	     return $name;
	   },
	   'CONTENT' => sub {
	     if(@_){
	       $plain = toUtf8(encodeHtml(shift));
	     }
	     return $plain;
	   },
	   'ADDCHILD'=> sub {
	     my $kind = shift;
		 if (!defined $kind) {
		   Logging::warning("undef argument passed to XmlNode->{ADDCHILD}. Ignoring it.");
		   return;
		 }
		 push @content,$kind;
	     return $#content;
	   },
	   'KIND'=> sub {
	     my $index = shift;
			 my ($ret);
			 if ( $index =~ /^\d+$/ ) {
				 if($index >= 0 && $index <= $#content ) {
					 $ret =  $content[$index];
				 }
			 } else {
				 foreach my $item (@content){
					 if ( $item->{'NAME'}->() eq $index ) {
						 $ret = $item;
						 last;
					 }
				 }
			 }
			 return $ret;
	   },
	   'KINDSREF'=> sub {
			 my $name = shift;
			 my $ret = \@content;
			 if ( $name ) {
				 $ret = [];
				 foreach my $item (@content) {
					 if ( $item->{'NAME'}->() eq $name ) {
						 push @{$ret}, $item;
					 }
				 }
			 }
	     return $ret;
	   },
	   'ATTRIBUTESREF' => sub {
	     return \%attributes;
	   },
	   'PRINT'=>sub {
	     my $fh = shift || *STDOUT;
	     my $addPrefix = shift || '';
	     my $prefix = shift || '';
	     if($name){
	       my ($out,$item);
	       if(@content || defined($plain)){

					 $out = $prefix . &getTag($name,\%attributes,'start');
					 print $fh $out;
					 if(@content){
						 print $fh "\n";
						 foreach $item (@content){
						   $item->{'PRINT'}->($fh,$addPrefix,$prefix.$addPrefix);
						 }
					 }
					 if(defined($plain)){
						 print $fh $plain;
						 if(@content){
							 print $fh "\n$prefix";
						 }
					 }else{
						 if($prefix){
							 print $fh $prefix;
						 }
					 }
					 $out = &getTag($name,\%attributes,'stop') . "\n";
					 print $fh $out;

	       }else{
					 $out = $prefix . &getTag($name,\%attributes) . "\n";
					 print $fh $out;
	       }
	     }else{
	       if(defined($plain)){
					 print $fh $plain;
	       }
	     }
	   },
		   'PRINT_TREE' => sub {
			 printXml($this, @_);
		   }
	 };

  return $this;
}

sub getTag {
  my ($name,$ptrAttr,$mode) = @_;
  my ($out,$key,$value);
  $mode||='';
  if ($mode eq 'stop'){
    $out="</$name>";
  }else{
    $out="<$name";
    while (($key,$value)=each(%{$ptrAttr})){
      $out .= ' '.$key.'="'.$value.'"';
    }
    if($mode eq 'start'){
      $out .= ">";
    }else{
      $out .= "/>";
    }
  }
  return $out;
}

#
# end class for handle with XML
#

sub getDbConnect {
  my ($myDbType, $myDbUser, $myDbPasswd, $myDbDB, $myDbHost,
      $continueAfterError, $myDbSocket, $variables, $utf, $port) = @_;

  my $connection;

  if ($myDbType eq 'Pg') {
    $myDbType = 'postgresql';
  }

  my %params;
  $params{'name'} = $myDbDB;
  $params{'user'} = $myDbUser;
  $params{'password'} = $myDbPasswd;
  $params{'host'} = $myDbHost if $myDbHost;
  $params{'socket'} = $myDbSocket if $myDbSocket;
  $params{'port'} = $port if $port;
  $params{'preload_dirs'} = $variables if $variables;
  $params{'type'} = $myDbType;

  $connection = Db::Connection::getConnection(%params);

  $this =
    {
     'DISCONNECT' => sub { return $connection->disconnect(@_); },
     'EXECUTE' => sub { return $connection->execute_rownum(@_); },
     'FETCHROW' => sub { return $connection->fetchrow(@_); },
     'FETCHHASH' => sub { return $connection->fetchhash(@_); },
     'FINISH' => sub { return $connection->finish(@_); },
     'PING' => sub { return $connection->ping(@_); },
     'USER' => sub { return $myDbUser; },
     'PASSWORD' => sub { return $myDbPasswd; },
     'HOST' => sub { return $myDbHost; }
    };
  return $this;
}

#
# end class for handle database
#

sub makeMIMEBase64 {
  my ($useModule);
  if (eval "require MIME::Base64") {
    $useModule = 1;
  } else {
    $useModule = 0;
  }

  my $this =
	{
	 'ENCODE' => sub {
	   my ($text) = @_;

	   if ($useModule) {
		 my $encoded = MIME::Base64::encode($text);
		 chomp $encoded;
		 return $encoded;
	   } else {
		 return encodeBase64($text);
	   }
	 }
	};

  return $this;
}

#
# url decode (certificates in Plesk DB encoded by the url encoding)
#

sub urlDecode {
    my $url = $_[0];
    $url =~ tr/+/ /;
    $url =~ s/%([a-fA-F0-9]{2,2})/chr(hex($1))/eg;
    return $url;
}

sub encodeHtml {
  my(@ret,$src);
  if ($src=shift){
    if(ref $src){
      if(${$src}=~/[&"'><]/){
		${$src}=~s/&/&amp;/sg;
		${$src}=~s/>/&gt;/sg;
		${$src}=~s/</&lt;/sg;
		${$src}=~s/\'/&#39;/sg;
		${$src}=~s/\"/&quot;/sg;
      }
      return ${$src};
    }else{
      if($src=~/[&"'><]/){
		$src=~s/&/&amp;/sg;
		$src=~s/>/&gt;/sg;
		$src=~s/</&lt;/sg;
		$src=~s/\'/&#39;/sg;
		$src=~s/\"/&quot;/sg;
      }
    }
  }
  return $src;
}

# COMPATIBILITY WRAPPER FOR makeXmlNode
sub normalizeAttribute {
  return XmlNode::_xmlAttributeEscape(@_);
}

# COMPATIBILTY WRAPPER
sub toUtf8 {
  Encoding::encode(@_);
}

#
#   Converts localized domain name into IDN ASCII representation
#
sub utf8ToIdn {
    my ($utf8domain) = @_;

    unless (AgentConfig::idnconvBin()) {
        return $utf8domain;
    }

    if (open(IC, "echo '$utf8domain' | " . AgentConfig::idnconvBin() . " -in 'UTF-8' |")) {
        binmode IC;
        my $converted = join "", <IC>;
        chomp $converted;
        close IC;
        return $converted if $converted;
    }

    return $utf8domain;
}

#
#   Converts IDN ASCII domain representation into UTF8 localized name
#
sub idnToUtf8 {
    my ($idnDomain) = @_;

    unless (AgentConfig::idnconvBin()) {
        return $idnDomain;
    }

    if (open(IC, "echo '$idnDomain' | " . AgentConfig::idnconvBin() . " -reverse -out 'UTF-8' |")) {
        binmode IC;
        my $converted = join "", <IC>;
        chomp $converted;
        close IC;
        return $converted if $converted;
    }

    return $idnDomain;
}

#
# Based on the b64.c by Bob Trower (base64.sf.net)
#

my @cb64 = ('A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',
			'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
			'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
			'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
			'g', 'h', 'i', 'j',	'k', 'l', 'm', 'n',
			'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
			'w', 'x', 'y', 'z', '0', '1', '2', '3',
			'4', '5', '6', '7', '8', '9', '+', '/');

@cb64 = map { ord } @cb64;

sub encodeBlock {
  my ($len, @in) = @_;
  my @res;
  push @res, $cb64[$in[0] >> 2];
  push @res, $cb64[(($in[0] & 0x3)<<4)|(($in[1]&0xf0)>>4)];
  push @res, $len>1 ? $cb64[(($in[1]&0xf)<<2)|(($in[2]&0xc0)>>6)] : ord('=');
  push @res, $len>2 ? $cb64[$in[2]&0x3f] : ord('=');
  return @res;
}

sub encodeBase64 {
  my @in = unpack "C*", $_[0];

  my @out;
  while (scalar(@in)) {
	my @block = (0, 0, 0);
	my $length = 0;
	while (scalar(@in) && $length < 3) {
	  $block[$length++] = shift(@in);
	}
	push @out, encodeBlock($length, @block);
  }
  return pack "C*", @out;
}

sub getArguments {
  return CompatArgParse::parse(@_);
}

# Compatibility wrapper around DumpStatus module

sub makeDumpStatus($) {
  my $dumpStatus = MigrationDumpStatus->new(shift);

  my $this = {
			  'FILE' => sub { return ""; },
			  'ACCOUNT' => sub { $dumpStatus->startClient(shift); },
			  'DOMAIN' => sub { $dumpStatus->startDomain(shift); },
			  'COUNTS' => sub { $dumpStatus->start($_[0], $_[1]); },
			  'PRINT' => sub { }
			 };
  return $this;
}

sub makeContentList {
  #FIXME: HACK
  my $workDir = `pwd`;
  chomp $workDir;
  my $storage = Storage::FileStorage->new('output_dir' => $workDir);
  print $storage->getContentList();
}

my $outputIsFile;

sub openOutput
{
	my ($outPath) = @_;
	my ($outFh);

	if ($outPath) {
		open(OUTPUT, ">>$outPath") or die "Error: can not open '$outPath' for write: $!\n";
		binmode OUTPUT;
		$outFh = *OUTPUT;
		$outputIsFile = 1;
	} else {
		$outFh = *STDOUT;
		$outputIsFile = 0;
	}

	return $outFh;
}

sub closeOutput{
  if($outputIsFile){
    close (OUTPUT) or
      &printToError("Error: unable to close OUTPUT: $!\n");
    $outputIsFile=0;
  }
}

#=========================================================================
#
# dtd-specific (but common) subroutines
#
sub createPasswordNode {
  my ($parent, $password, $type) = @_;

  $parent->{'ADDCHILD'}->(XmlNode->new('password', 'content' => $password, 'attributes' => {'type' => $type}));
}

sub createSysuserNode {
  my ($parent, $name, $password, $type) = @_;

  my $item = XmlNode->new('sysuser', 'attributes' => { 'name' => $name});
  createPasswordNode($item, $password, $type);
  $parent->{'ADDCHILD'}->($item);
}

sub addPermissionNode {
	my ($parent, $name, $allowed) = @_;

	# [Bug 111477] Value "-1" for attribute allowed of permission is not among the enumerated set
	if ($allowed =~ /-1/) {
		$allowed = 'false';
	}

    $parent->{'ADDCHILD'}->(XmlNode->new('permission', 'attributes' => { 'name' => $name, 'value' => $allowed}));
}

sub addMailuserPermissionNode {
	my ($parent, $name, $allowed) = @_;

    $parent->{'ADDCHILD'}->(XmlNode->new('mailuser-permission', 'attributes' => { 'name' => $name, 'value' => $allowed}));
}

sub readFile {
  my ($fileName) = @_;

  unless (-f $fileName) {
	return undef;
  }

  open(INPUT, "< $fileName");
  binmode INPUT;
  my $content = join('', <INPUT>);
  close(INPUT);

  return $content;
}

# Returns array with removed duplicates
sub uniq( $ )
{
	my %seen = ();
	return grep { ! $seen{$_} ++ } @_;
}

# Returns string, suitable for using in 'IN ($list)' sql clause
sub getSqlList
{
	return join(',', map {"'$_'"} sort {$a <=> $b} @_);
}

# Returns random password
sub randomPasswd() {
	my $symbols = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
	my $seq_length = length($symbols);
	my $passwd = '';

	for ($i = 0; $i < 7; ++$i) {
		$passwd .= substr($symbols, rand($seq_length), 1);
	}

	return $passwd;
}

1;

#
# end common subroutines
#
#=========================================================================
