package Db::DbiBackend;

use strict;
use Db::Backend;
use Logging;

use vars qw|@ISA|;

@ISA = qw|Db::Backend|;

sub _init {
  my ($self, %params) = @_;

  $self->SUPER::_init(%params);

  $self->{type} = $params{type};
}

sub description {
  my ($self) = @_;
  return "$self->{type} DBI connection. " . $self->SUPER::description();
}

sub _composeDbiAddress {
  my ($self) = @_;

  my $address = "DBI:$self->{type}:dbname=$self->{name}";
  $address .= ";host=$self->{host}" if defined $self->{host};

  return $address;
}

sub connect {
  my ($self) = @_;

  unless (eval "require DBI" and eval "require DBD::$self->{type}") {
    $self->_set_errstr("Unable to find DBI or DBD::$self->{type} module");
    return;
  }

  my $address = $self->_composeDbiAddress();

  $self->{dbh} = DBI->connect($address, $self->{user}, $self->{password});

  return 1 if $self->{dbh};

  $self->_set_errstr("Unable to connect to database: $DBI::errstr");
  return;
}

sub disconnect {
  my ($self) = @_;
  $self->{dbh}->disconnect();
}

sub execute {
  my ($self, $sql, $quiet) = @_;

  $self->{result} = $self->{dbh}->prepare($sql);
  if (!$self->{result}) {
    Logging::error("Unable to prepare SQL for execution: $DBI::errstr");
    $self->_set_errstr("Unable to prepare SQL ( for execution: $DBI::errstr");
    return;
  }

  my $res = $self->{result}->execute();
  if (!defined $res) {
    Logging::error("Unable to execute SQL: $DBI::errstr") unless $quiet;
    $self->_set_errstr("Unable to execute SQL: $DBI::errstr");
    return;
  }

  return 1;
}

sub rownum {
  my ($self) = @_;
  return $self->{result}->rows;
}

sub fetchrow {
  my ($self) = @_;

  return $self->{result}->fetchrow_arrayref();
}

sub fetchhash {
  my ($self) = @_;

  my $h = $self->{result}->fetchrow_hashref();
  return if ref($h) eq "HASH" and !%$h;
  return $h;
}

sub finish {
  my ($self) = @_;

  $self->{result}->finish();
}

sub ping {
  my ($self) = @_;

  return $self->{dbh}->ping();
}

1;