package Mailman;

use strict;
use AgentConfig;
use Logging;

my $__version;

sub version {
  $__version = __getMailmanVersion() unless defined $__version;
  return $__version;
}

# bin/list_members support full usernames dump only since mailman 2.1

sub supportSubscribersFullName {
  return version() =~ /^2\.0/ ? 0 : 1;
}

# returns the list of list owners
sub getListOwners {
  my ($listName) = @_;
  my $python = AgentConfig::pythonBin();
  my $mailmanRoot = AgentConfig::mailmanRoot();

  my @owners;

  my $listAdminBin = AgentConfig::getSharedDir() . "/list_admin.py";

  open LIST_OWNER, "$python $listAdminBin $mailmanRoot $listName |";
  while (<LIST_OWNER>) {
    chomp;
    push @owners, $_;
  }
  close LIST_OWNER;

  return @owners;
}

sub getListPassword {
  my ($listName) = @_;

  my $python = AgentConfig::pythonBin();
  my $mailmanRoot = AgentConfig::mailmanRoot();

  my $listPasswordBin = AgentConfig::getSharedDir() . "/list_password.py";

  open LIST_PASSWORD, "$python $listPasswordBin $mailmanRoot $listName |";
  my $password = <LIST_PASSWORD>;
  chomp $password;
  close LIST_PASSWORD;

  return $password;
}

# Returns the hash of mail->fullname.
sub getListMembers {
  my ($listName) = @_;

  my $mailmanRoot = AgentConfig::mailmanRoot();

  my $options = "";
  if (supportSubscribersFullName()) {
    $options = "-f";
  }

  my %subscribers;

  open LIST_MEMBERS, "$mailmanRoot/bin/list_members $options $listName |";
  while (<LIST_MEMBERS>) {
    chomp;

    my ($fullname, $email) = ("", "");
    if (supportSubscribersFullName()) {
      if (/^(.*)\s+<(.*)>$/) {
        $fullname = $1;
        $email = $2;
      } else {
        $email = $_;
      }
    } else {
      $email = $_;
    }

    $subscribers{$email} = $fullname;
  }
  close LIST_MEMBERS;

  return %subscribers;
}

sub getMailLists{
    my($domainname) = @_;

    my @lists = ();
    my $mailmanRoot = AgentConfig::mailmanRoot();
    my $mailman = "$mailmanRoot/bin/list_lists";

    if( -X $mailman ){
       open MAILMAN_LISTS, "$mailman -b -V $domainname |";
       @lists = <MAILMAN_LISTS>;
       chomp @lists;
       close MAILMAN_LISTS;
    }

    return @lists;
}

sub getMailListArchiveDir{
  my ($maillist) = @_;

  my $archDir = AgentConfig::mailmanArchivesDir();

  unless( -e $archDir ){
  	Logging::info( "Cannot find mailman arhive directory '$archDir'" );
        return undef;
  }

  $archDir = "$archDir/private/$maillist.mbox";
  unless( -e $archDir ){
  	Logging::info( "Cannot find mailman maillist '$maillist' arhive directory '$archDir'" );
        return undef;
  }
  return $archDir;
}

sub getMailListArchiveFiles{
  my ($maillist) = @_;
  my @files = ();
  my $dir =  getMailListArchiveDir( $maillist );
  if( $dir ){
  	push @files, "$maillist.mbox";
  }
  return \@files;
}


# $cmd must exist and be executable
sub __parseVersion {
  my ($cmd) = @_;

  open MAILMAN_VERSION, "$cmd |";
  while (<MAILMAN_VERSION>) {
    chomp;
    if (/^Using Mailman version:? ([.0-9]+)$/) {
      close MAILMAN_VERSION;
      return $1;
    }
  }
  close MAILMAN_VERSION;
}

sub __getMailmanVersion {
  my $mailmanRoot = AgentConfig::mailmanRoot();
  return unless defined $mailmanRoot;
  my $versionCmd = "$mailmanRoot/bin/version";
  return unless -X $versionCmd;
  return __parseVersion($versionCmd);
}

1;
