package Agent;

use Logging;
use XmlNode;
use Transformer;

use AgentConfig;
use Dumper; # used for getAgentStatus only
use ContentDumper;
use Storage::Storage;


my $agentWorkDir;

#
# Agent.pm module implements Agent interface
#


#
# Begin interface methods
#
sub setWorkDir {
  my $workDir = shift;
  $agentWorkDir = $workDir;
}

sub getWorkDir {
  return $agentWorkDir || AgentConfig::cwd();;
}

sub getAgentName {
  return "cPanel";
}

sub getAgentStatus {
  my $statusNode = Dumper::getAgentStatus();
  return $statusNode;
}

sub getResellers {
  # resellers migration is not supported now

  # should return an array of reseller identifiers, that could be a number, name or guid. Should be unique through migration dump.
  return Transformer::getResellers();
}

sub getClients {
  my $owner = shift; # reseller that owns the clients returned. Could be 'undef' for default reseller ('root' or 'admin')

  # clients migration is not supported now

  # should return an array of clients identifiers, that could be a number, name or guid. Should be unique through migration dump.
  return Transformer::getClients($owner);
}

sub getDomains {
  my $owner = shift; # reseller or client that owns the domains returned. Could be 'undef' for default reseller or client ('root' or 'admin')

  # Should return an array of clients identifiers, that could be a number, name or guid. Should be unique through migration dump.

  # The current implementation of cPanel migration deals with domains only. Will return all the domains for 'undef' passed.

  return Transformer::getDomains($owner);
}

sub getDomain {
  my ($id, $recursive, $content, $do_gzip, $contentOptions) = @_; # domain identifier from 'getDomains' result, flag to return node with children (for optimization purporses), flag to dump content, flag to compress content, content options

  my $clientNode = Transformer::getClientNode4Domain( $id );

  my $myDomainNode;

  if ( $clientNode ) {
    my $domainsNode = $clientNode->getChild( 'domains' );
    if ( defined $domainsNode ) {
      my @domainNodes = $domainsNode->getChildren( 'domain' );
      foreach my $domainNode (@domainNodes) {
        if ($domainNode->getAttribute( 'name' ) eq $id ) {
          $myDomainNode = $domainNode;
          last;
        }
      }

    if ( defined $myDomainNode ) {
      if( $recursive ){
        unless ( $content ) {
          # Return from hash. Do not copy()
          return $myDomainNode;
        }
        else {
          return &addContent2DomainNode( $myDomainNode->copy(), $do_gzip, $contentOptions );
        }
      }
      return $myDomainNode->copy('no_child');
    }
  }
}
  return undef;
}

sub getDomainContent {
  my ($id, $content, $do_gzip) = @_; # domain identifier from 'getDomains' result, flag to dump content, flag to compress content

  # should return an XmlNode object

  return undef;
}

sub getDomainPreferences {
  my ($id, $content, $do_gzip) = @_; # domain identifier from 'getDomains' result, flag to dump content, flag to compress content

  my $domainNode = &getDomain( $id, 'recursive');
  return undef unless ( defined ( $domainNode ) ); 

  my $preferencesNode = $domainNode->getChild( 'preferences' );
  return undef unless ( defined ( $preferencesNode ) ); 
  
  if ( $content ) {
    return &addContent2DomainPreferencesNode( $preferencesNode->copy(), $do_gzip );
  }
  return $preferencesNode->copy();
}

sub getDomainProperties {
  my $id = shift; # domain identifier from 'getDomains' result

  my $domainNode = &getDomain( $id, 'recursive');
  return undef unless ( defined ( $domainNode ) ); 

  my $propertiesNode = $domainNode->getChild( 'properties' );
  return undef unless ( defined ( $propertiesNode ) ); 
  
  return $propertiesNode->copy();
}

sub getDomainLimitsAndPermissions {
  my $id = shift; # domain identifier from 'getDomains' result

  my $domainNode = &getDomain( $id, 'recursive');
  return undef unless ( defined ( $domainNode ) ); 

  my $limitsAndPermissionsNode = $domainNode->getChild( 'limits-and-permissions' );
  return undef unless ( defined ( $limitsAndPermissionsNode ) ); 
  
  return $limitsAndPermissionsNode->copy();
}

sub getDomainMailsystem {
  my ($id, $content, $do_gzip) = @_; # domain identifier from 'getDomains' result, flag to dump content, flag to compress content

  my $domainNode = &getDomain( $id, 'recursive');
  return undef unless ( defined ( $domainNode ) ); 

  my $mailsystemNode = $domainNode->getChild( 'mailsystem' );
  return undef unless ( defined ( $mailsystemNode ) ); 
  
  if ( $content ) {
    return &addContent2MailsystemNode( $mailsystemNode->copy(), $do_gzip );
  }
  return $mailsystemNode->copy();
}

sub getDomainDatabases {
  my ($id, $content, $do_gzip) = @_; # domain identifier from 'getDomains' result, flag to dump content, flag to compress content

  my $domainNode = &getDomain( $id, 'recursive');
  return undef unless ( defined ( $domainNode ) ); 

  my $databasesNode = $domainNode->getChild( 'databases' );
  return undef unless ( defined ( $databasesNode ) ); 
  
  if ( $content ) {
    return &addContent2DatabasesNode( $databasesNode->copy(), $do_gzip );
  }
  return $databasesNode->copy();
}

sub getDomainOdbcDsn {
  my $id = shift; # domain identifier from 'getDomains' result

  # should return an XmlNode object

  return undef;
}

sub getDomainColdfusionDsn {
  my $id = shift; # domain identifier from 'getDomains' result

  # should return an XmlNode object array

  return undef;
}

sub getDomainMaillists {
  my ($id, $content, $do_gzip) = @_; # domain identifier from 'getDomains' result, flag to dump content, flag to compress content

  my $domainNode = &getDomain( $id, 'recursive');
  return undef unless ( defined ( $domainNode ) ); 

  my $maillistsNode = $domainNode->getChild( 'maillists' );
  return undef unless ( defined ( $maillistsNode ) ); 
  
  if ( $content ) {
    return &addContent2MaillistsNode( $maillistsNode->copy(), $do_gzip );
  }
  return $maillistsNode->copy();
}

sub getDomainTraffic {
  my $id = shift; # domain identifier from 'getDomains' result

  my $domainNode = &getDomain( $id, 'recursive');
  return undef unless ( defined ( $domainNode ) ); 

  my $trafficNode = $domainNode->getChild( 'traffic' );
  return undef unless ( defined ( $trafficNode ) ); 
  
  return $trafficNode->copy();
}

sub getDomainCertificates {
  my $id = shift; # domain identifier from 'getDomains' result

  # should return an XmlNode object

  return undef;
}

sub getDomainTomcat {
  my ($id, $content, $do_gzip) = @_; # domain identifier from 'getDomains' result, flag to dump content, flag to compress content

  my $domainNode = &getDomain( $id, 'recursive');
  return undef unless ( defined ( $domainNode ) ); 

  my $tomcatNode = $domainNode->getChild( 'traffic' );
  return undef unless ( defined ( $tomcatNode ) ); 
  
  if ( $content ) {
    return &addContent2TomcatNode( $tomcatNode->copy(), $do_gzip );
  }
  return $tomcatNode->copy();
}

sub getDomainDomainuser {
  my $id = shift; # domain identifier from 'getDomains' result

  my $domainNode = &getDomain( $id, 'recursive');
  return undef unless ( defined ( $domainNode ) ); 

  my $domainuserNode = $domainNode->getChild( 'domainuser' );
  return undef unless ( defined ( $domainuserNode ) ); 
  
  return $domainuserNode->copy();
}

sub getDomainShosting {
  my $id = shift; # domain identifier from 'getDomains' result

  my $domainNode = &getDomain( $id, 'recursive');
  return undef unless ( defined ( $domainNode ) ); 

  my $shostingNode = $domainNode->getChild( 'shosting' );
  return undef unless ( defined ( $shostingNode ) ); 
  
  return $shostingNode->copy();
}

sub getDomainFhosting {
  my $id = shift; # domain identifier from 'getDomains' result

  # cPanel domain never maps to Plesk' domain with fhosting
  return undef;
}

sub getDomainPhosting {
  my ($id, $content, $do_gzip) = @_; # domain identifier from 'getDomains' result, flag to dump content, flag to compress content

  my $domainNode = &getDomain( $id, 'recursive');
  return undef unless ( defined ( $domainNode ) ); 

  my $phostingNode = $domainNode->getChild( 'phosting' );
  return undef unless ( defined ( $phostingNode ) ); 
  
  if ( $content ) {
    return &addContent2PhostingNode( $phostingNode->copy(), $do_gzip );
  }
  return $phostingNode->copy();
}
#
# End of interface methods
#



#
# Begin subs for content linking
#
sub addContent2DomainNode {
  my ($domainNode, $do_gzip, $contentOptions) = @_;
  
  return undef unless ( defined ( $domainNode ) );
  
  my $onlyMail    = exists ( $contentOptions->{'only-mail'} )    || undef;
  my $onlyHosting = exists ( $contentOptions->{'only-hosting'} ) || undef;
  
  my $domainPreferencesNode = $domainNode->getChild( 'preferences' );
  &addContent2DomainPreferencesNode( $domainPreferencesNode, $do_gzip);

  unless ( defined ( $onlyHosting ) ) {
    my $domainMailsystemNode = $domainNode->getChild( 'mailsystem' );
    &addContent2MailsystemNode( $domainMailsystemNode, $do_gzip) if ( defined ( $domainMailsystemNode ) );
  }

  unless ( defined ( $onlyMail ) ) {
    my $domainDatabasesNode = $domainNode->getChild( 'databases' );
    &addContent2DatabasesNode( $domainDatabasesNode, $do_gzip) if ( defined ( $domainDatabasesNode ) );
  }

  unless ( defined ( $onlyHosting ) ) {
    my $domainMaillistsNode = $domainNode->getChild( 'maillists' );
    &addContent2MaillistsNode( $domainMaillistsNode, $do_gzip) if ( defined ( $domainMaillistsNode ) );
  }

  unless ( defined ( $onlyMail ) ) {
    my $domainTomcatNode = $domainNode->getChild( 'tomcat' );
    &addContent2TomcatNode( $domainTomcatNode, $do_gzip) if ( defined ( $domainTomcatNode ) );
  }

  unless ( defined ( $onlyMail ) ) {
    my $domainPhostingNode = $domainNode->getChild( 'phosting' );
    &addContent2PhostingNode( $domainPhostingNode, $do_gzip) if ( defined ( $domainPhostingNode ) );
  }

  return $domainNode;
}

sub addContent2DomainPreferencesNode {
  my ($domainPreferencesNode, $do_gzip) = @_;
  
  return undef unless ( defined ( $domainPreferencesNode ) );
  
  return $domainPreferencesNode;
}

sub addContent2MailsystemNode {
  my ($mailsystemNode, $do_gzip) = @_;

  return undef unless ( defined ( $mailsystemNode ) );

  my $mailusersNode = $mailsystemNode->getChild( 'mailusers' );
  
  if ( defined ( $mailusersNode ) ) {
    my @mailuserNodes = $mailusersNode->getChildren( 'mailuser' );
    foreach my $mailuserNode ( @mailuserNodes ) {
      my $mailuserMetadata = $mailuserNode->getMetadata();
      my $mailname = $mailuserMetadata->{'mailname'};
      my $domain = $mailuserMetadata->{'domain'};
      my $account = $mailuserMetadata->{'account'};

      my $storage = Storage::Storage::createFileStorage($do_gzip, &getWorkDir() );
      my $contentDumper = ContentDumper->new($storage);

      next unless ( defined ( $mailname ) and defined ( $domain ) and defined ( $account ) );
      my $mailuserPreferencesNode = $mailuserNode->getChild( 'preferences' );
      my $mailboxNode = $mailuserPreferencesNode->getChild( 'mailbox' );
      if ( defined ( $mailboxNode ) ) {
        my $mailboxContentNode = $contentDumper->getMailnameContent($mailname,$domain,$account);
        $mailboxNode->addChild( $mailboxContentNode, 'first' ) if ( defined ( $mailboxContentNode ) );
      }
    }
  

  }

  return $mailsystemNode;
}

sub addContent2DatabasesNode {
  my ($databasesNode, $do_gzip) = @_;

  return undef unless ( defined ( $databasesNode ) );
  
  my %dbUser = ( 'mysql'      => 'root',
                 'postgresql' => 'postgres');

  my $databasesMetadata = $databasesNode->getMetadata();
  my $domain = $databasesMetadata->{'domain'};

  my @databaseNodes = $databasesNode->getChildren( 'database' );
  if ( @databaseNodes ) {
    my $storage = Storage::Storage::createFileStorage($do_gzip, &getWorkDir() );
    my $contentDumper = ContentDumper->new($storage);
    foreach my $databaseNode ( @databaseNodes ) {
      my $dbName = $databaseNode->getAttribute( 'name' );
      my $dbType = $databaseNode->getAttribute( 'type' );
      my $contentNode = $contentDumper->getDatabaseContent($dbName, $dbType, $dbUser{$dbType}, $domain );
      $databaseNode->addChild( $contentNode, 'first' ) if ( defined ( $contentNode ) );
    }
  }

  return $databasesNode;
}

sub addContent2MaillistsNode {
  my ($maillistsNode, $do_gzip) = @_;

  return undef unless ( defined ( $maillistsNode ) );

  my @maillistNodes = $maillistsNode->getChildren( 'maillist' );
  foreach my $maillistNode ( @maillistNodes ) {
    my $maillistMetadata = $maillistNode->getMetadata();
    my $domain = $maillistMetadata->{'domain'};
    my $listname = $maillistMetadata->{'listname'};
  
    next unless ( defined ( $domain ) and defined ( $listname ) );

    my $storage = Storage::Storage::createFileStorage($do_gzip, &getWorkDir() );
    my $contentDumper = ContentDumper->new($storage);
    my $maillistContentNode = $contentDumper->getMaillistContent($listname,$domain);

    $maillistNode->addChild( $maillistContentNode, 'first' ) if ( defined ( $maillistContentNode ) );
  }

  return $maillistsNode;
}

sub addContent2TomcatNode {
  my ($tomcatNode, $do_gzip) = @_;
  
  return undef unless ( defined ( $tomcatNode ) );
  
  return $tomcatNode;
}

sub addContent2PhostingNode {
  my ($phostingNode, $do_gzip) = @_;
  
  return undef unless ( defined ( $phostingNode ) );
  
  my $phostingMetadata = $phostingNode->getMetadata();
  my $domain = $phostingMetadata->{'domain'};
  my $account = $phostingMetadata->{'account'};
  my $from_subdomain = $phostingMetadata->{'from_subdomain'};
  my $to_domain = $phostingMetadata->{'to_domain'};
  
  my $storage = Storage::Storage::createFileStorage($do_gzip, &getWorkDir() );
  my $contentDumper = ContentDumper->new($storage);
  my $contentNode;
  if ( defined ( $from_subdomain ) and defined ( $to_domain ) ) {
    $contentNode = $contentDumper->getSubdomainAsDomainContent($from_subdomain, $domain, $to_domain, $account);
  }
  else {
    $contentNode = $contentDumper->getPhostingContent($domain, $account);
  }
  
  $phostingNode->addChild( $contentNode, 'first' ) if ( defined ( $contentNode ) );
  
  my $preferencesNode = $phostingNode->getChild( 'preferences' );
  # $preferencesNode is always defined here
  
  my $anonftpNode = $preferencesNode->getChild( 'anonftp' );
  if ( defined ( $anonftpNode ) ) {
    my $anonftpContentNode = $contentDumper->getAnonFtpContent($domain, $account);
    $anonftpNode->addChild( $anonftpContentNode, 'first' ) if ( defined ( $anonftpContentNode ) );
  }
  
  my $webusersNode = $phostingNode->getChild( 'webusers' );
  if ( defined ( $webusersNode ) ) {
    my @webuserNodes = $webusersNode->getChildren( 'webuser' );
    foreach my $webuserNode ( @webuserNodes ) {
      # It is better to extract webuser name from metadata, because 'name' is not required attribute for 'webuser'
      my $webuserMetadata = $webuserNode->getMetadata();
      my $webuserName = $webuserMetadata->{'name'};

      my $webuserContentNode = $contentDumper->getWebUserContent($webuserName, $domain, $account);
      $webuserNode->addChild( $webuserContentNode, 'first' ) if ( defined ( $webuserContentNode ) );
    }
  }

  my $subdomainsNode = $phostingNode->getChild( 'subdomains' );
  if ( defined ( $subdomainsNode ) ) {
    my @subdomainNodes = $subdomainsNode->getChildren( 'subdomain' );
    foreach my $subdomainNode ( @subdomainNodes ) {
      my $subdomainName = $subdomainNode->getAttribute( 'name' );
      my $subdomainContentNode = $contentDumper->getSubdomainContent($subdomainName, $domain, $account);
      $subdomainNode->addChild( $subdomainContentNode, 'first' ) if ( defined ( $subdomainContentNode ) );
    }
  }

  return $phostingNode;
}
#
# End of subs for content linking
#


sub printAgentStatus {
  my $root = XmlNode->new('agent-status');

  unless ( Dumper::getCpanelVersion() && defined AgentConfig::iconvBin() ) {
    my $item;
    if ( defined AgentConfig::iconvBin() ) {
      $item = XmlNode->new( 'wrong-platform' );
      $item->setText( '' );
    }
    else {
      $item = XmlNode->new( 'wrong-platform' );
      $item->setText( 'no iconv found on the source host' );
    }

    $root->addChild($item);
  }

  $root->serialize( \*STDOUT );
}

1;
