package ContentDumper;
use strict;
use XmlNode;
use Storage::FileNameCreator;
use Storage::Storage;
use Mailman;
use Logging;
use Db::Connection;

use vars qw|@ISA|;

sub new {
  my $self = {};
  bless( $self, shift );
  $self->_init(@_);
  return $self;
}

sub _init {
  my ( $self, $storagepolicy ) = @_;
  $self->{storage} = $storagepolicy;
  $self->{fnamecreator} = Storage::FileNameCreator->new();
}

sub getPhostingDstFile {
  my ($self, $type, $domainName) = @_;
  my $dir = "/domains/".$domainName."/phosting";
  return  $self->{fnamecreator}->getFileName($dir, '', '', $type);
}

sub getSubdomainDstFile {
  my ($self, $type, $domainName, $subdomainName) = @_;
  my $dir = "/domains/".$domainName."/phosting/subdomain";
  return $self->{fnamecreator}->getFileName($dir, '', '', $subdomainName."_".$type);
}

sub getDbDstFile {
  my ($self, $type, $domainName, $dbName) = @_;
  my $dir = "/domains/".$domainName."/databases/".$dbName;
  return $self->{fnamecreator}->getFileName($dir, '', '', $type);
}

sub getWebUserDstFile {
  my ($self, $domainName, $wuName) = @_;
  my $dir = "/domains/".$domainName."/phosting/";
  return $self->{fnamecreator}->getFileName($dir, '', '', $wuName);
}

sub getMailnameDstFile {
  my ($self, $mailname, $domainName) = @_;
  my $dir = "/domains/".$domainName."/mailnames";
  return $self->{fnamecreator}->getFileName($dir, '', '', "mailname_" . $mailname);
}

sub getMaillistDstFile {
  my ($self, $listname, $domainName) = @_;
  my $dir = "/domains/".$domainName;
  return $self->{fnamecreator}->getFileName($dir, '', '', "maillist_". $listname);
}

sub makeCidNode{
  my ($self, $id, $cid_type ) = @_;

  my $files = $self->{storage}->getFilesFromId( $id );

  return undef unless defined $files;
  return undef if (scalar(@{$files}) == 0);

  my $cid = XmlNode->new( 'cid' );
  $cid->setAttribute( 'type', $cid_type );
  $cid->setAttribute( 'unpacksize', $self->{storage}->getFilesUnpackSizeFromId($id) );
  $cid->setAttribute( 'path', $self->{storage}->getFilePathFromId( $id ) );
  foreach my $filedata( @{$files} ){
    $cid->addChild( XmlNode->new( 'content-file', 'content' => $filedata->[0], 'attributes' => {'size' => $filedata->[1] } ) );
  }
  return $cid;
}

sub getPhostingContent {
  my ($self, $domainName, $account) = @_;

  my @exclude = ();
  my (%options, $cgi, @childrens);
  my $home = $self->getHomeDir($account);
  $options{'checkEmptyDir'} = 1;

  if (-d "$home/cgi-bin") {
    @exclude = ('cgi-bin');
  }else{
    @exclude = ();
    my $cgi_dst = $self->getPhostingDstFile('cgi', $domainName);
    $options{'directory'} = "$home/public_html/cgi-bin";
    $cgi = $self->{storage}->addTar($cgi_dst, %options);
    my $cgiNode = $self->makeCidNode($cgi, 'cgi') if $cgi;
    push @childrens, $cgiNode if ref($cgiNode)=~/XmlNode/; 
  } 

  my $ptrSubdomainsDirs = $self->getSubdomainsDirs($domainName, $account);
  my $ptrFtpUsersDirs = $self->getFtpUsersDirs($account, $ptrSubdomainsDirs);
  push @exclude, @{$ptrSubdomainsDirs}, @{$ptrFtpUsersDirs};
 
  my $docroot_dst = $self->getPhostingDstFile('docroot', $domainName);
  
  $options{'directory'} = "$home/public_html";
  $options{'exclude'} = \@exclude;
  $options{'include_hidden_files'} = 1;
  my $htdocs = $self->{storage}->addTar($docroot_dst, %options);
  my $docrootNode = $self->makeCidNode($htdocs, 'docroot') if $htdocs;
  push @childrens, $docrootNode if ref($docrootNode)=~/XmlNode/;
 
  if (scalar(@childrens)>0) {
    my $contentNode = XmlNode->new('content');
    foreach my $node (@childrens){
        $contentNode->addChild($node);
    }
    return $contentNode;
  }

  return undef;
}

sub getAnonFtpContent {
  my ($self, $domainName, $account) = @_;
  my $home = $self->getHomeDir($account);

  my (%options, @childrens);
  $options{'checkEmptyDir'} = 1;
  $options{'include_hidden_files'} = 1;

  my $publicFtp = "$home/public_ftp";
  unless(-d $publicFtp){
        return undef;
  }
  
  my $incoming_dst = $self->getPhostingDstFile('ftp_incoming', $domainName);
  $options{'directory'} = "$publicFtp/incoming";
  my $incoming = $self->{storage}->addTar($incoming_dst, %options);

  $options{'directory'} = $publicFtp;
  my @exclude = ('incoming');
  $options{'exclude'} = \@exclude;
  my $pub_dst = $self->getPhostingDstFile('ftp_pub', $domainName);
  my $pub = $self->{storage}->addTar($pub_dst, %options);
  
  my $ftpIncNode = $self->makeCidNode($incoming, 'ftp_incoming') if $incoming;
  push @childrens, $ftpIncNode if ref($ftpIncNode)=~/XmlNode/;
  my $ftpPubNode = $self->makeCidNode($pub, 'ftp_pub') if $pub;
  push @childrens, $ftpPubNode if ref($ftpPubNode)=~/XmlNode/;

  if (scalar(@childrens)>0) {
    my $contentNode = XmlNode->new('content');
    foreach my $node (@childrens){
      $contentNode->addChild($node);
    }
    return $contentNode;
  }
  
  return undef;
}

sub getSubdomainContent {
  my ($self, $subdomainName, $domainName, $account) = @_;
  my $home = $self->getHomeDir($account);

  my $subdomainRoot = "$home/public_html/$subdomainName";

  my (%options, @exclude, $cgi, @childrens);
  $options{'checkEmptyDir'} = 1;

  if (-d "$subdomainRoot/cgi-bin") {
    my $cgi_dst = $self->getSubdomainDstFile('cgi', $domainName, $subdomainName );
    $options{'directory'} = $subdomainRoot."/cgi-bin";
    $cgi = $self->{storage}->addTar($cgi_dst, %options);
    @exclude=('cgi-bin');

  } else {
    @exclude=();
  }

  my $docroot_dst = $self->getSubdomainDstFile('docroot', $domainName, $subdomainName);
  $options{'directory'} = $subdomainRoot;
  $options{'exclude'} = \@exclude;

  my $docroot = $self->{storage}->addTar($docroot_dst, %options);

  my $docrootNode = $self->makeCidNode($docroot, 'docroot') if $docroot;
  push @childrens, $docrootNode if ref($docrootNode)=~/XmlNode/;
  my $cgiNode = $self->makeCidNode($cgi, 'cgi') if $cgi;
  push @childrens, $cgiNode if ref($cgiNode)=~/XmlNode/;

  if (scalar(@childrens)>0) {
    my $contentNode = XmlNode->new('content');
    foreach my $node (@childrens){
      $contentNode->addChild($node);
    }
    return $contentNode;
  }
  
  return undef;
}

sub getSubdomainAsDomainContent {
  my ($self, $subdomainName, $domainName, $resultName, $account) = @_;
  my $home = $self->getHomeDir($account);

  my $subdomainRoot = "$home/public_html/$subdomainName";

  my (%options, @exclude, $cgi, @childrens);
  $options{'checkEmptyDir'} = 1;

  if (-d "$subdomainRoot/cgi-bin") {
    my $cgi_dst = $self->getPhostingDstFile('cgi', $resultName );
    $options{'directory'} = $subdomainRoot."/cgi-bin";
    $cgi = $self->{storage}->addTar($cgi_dst, %options);
    @exclude=('cgi-bin');

  } else {
    @exclude=();
  }

  my $docroot_dst = $self->getPhostingDstFile('docroot', $resultName);
  $options{'directory'} = $subdomainRoot;
  $options{'exclude'} = \@exclude;

  my $docroot = $self->{storage}->addTar($docroot_dst, %options);
  
  my $docrootNode = $self->makeCidNode($docroot, 'docroot') if $docroot;
  push @childrens, $docrootNode if ref($docrootNode)=~/XmlNode/;
  my $cgiNode = $self->makeCidNode($cgi, 'cgi') if $cgi;
  push @childrens, $cgiNode if ref($cgiNode)=~/XmlNode/;

  if (scalar(@childrens)>0) {
    my $contentNode = XmlNode->new('content');
    foreach my $node (@childrens){
      $contentNode->addChild($node);
    }
    return $contentNode;
  }
  
  return undef;

}

sub getDatabaseContent {
  
  my ($self, $dbName, $dbType, $dbUser, $domain, $dbPassword, $dbHost, $ptrDomainBin, $variables, $domainSocket) = @_;

  my %params;

  if ($dbType eq 'Pg') { $dbType = 'postgresql'; }

  $params{'name'} = $dbName;
  $params{'user'} = $dbUser;
  $params{'type'} = $dbType;
  $params{'password'} = $dbPassword;
  $params{'host'} = $dbHost if $dbHost;
  $params{'socket'} = $domainSocket if $domainSocket;
  $params{'preload_dirs'} = $variables if $variables;
  $params{'utf8names'} = 1;

  my %testparams = %params;

  # check db existence, bug 83616
  my $connection = Db::Connection::getConnection(%testparams);
  my $db_exist = $connection->execute_rownum($dbType eq 'postgresql' ? "SELECT 1 FROM pg_database WHERE datname = '$dbName'" : 'SELECT 1');
  $connection->finish();
  $connection->disconnect();

  my $db_dst = $self->getDbDstFile('sqldump', $domain, $dbName);

  if ($db_exist) {
    my $db = $self->{storage}->addDb($db_dst, %params);
    my $contentNode = XmlNode->new('content');
    $contentNode->addChild($self->makeCidNode($db, 'sqldump'));
    return $contentNode;
  }

  return undef;
}

sub getWebUserContent {
  my ($self, $wuName, $domainName, $account) = @_;
  my $fileParser = makeFileParser("/etc/proftpd/$account");
  my $ptrProftpdRows = $fileParser->{'PARSE'}->('VALUESEPARATOR'=>':','KEYSEPARATOR'=>'',);

  my $wu_dst = $self->getWebUserDstFile($domainName, $wuName);

  my $dir;

  foreach my $ptrRow (@{$ptrProftpdRows}) {
    if ($ptrRow->[0] eq $wuName) {
	$dir = $ptrRow->[5]; 
	last;
    }
  }
  if ( -d $dir) {
    my %options;
    $options{'directory'} = $dir;
    $options{'checkEmptyDir'} = 1;

    my $wuContent = $self->{storage}->addTar($wu_dst, %options);
    my $contentNode = XmlNode->new('content');
    $contentNode->addChild($self->makeCidNode($wuContent, 'webuser_home'));
    return $contentNode;
  }
  
  return undef;
}

sub getMailnameContent {
  my ($self, $mailname, $domain, $account) = @_;
  my $home = $self->getHomeDir($account);

  my $mailname_dst = $self->getMailnameDstFile($mailname, $domain);

  my $dir = "$home/mail/$domain/$mailname";

  if ( -d $dir) {
    my %options;
    $options{'directory'} = $dir;
    $options{'checkEmptyDir'} = 1;

    my $mnContent = $self->{storage}->addTar($mailname_dst, %options);
    my $contentNode = XmlNode->new('content');
    $contentNode->addChild($self->makeCidNode($mnContent, 'mailbox'));
    return $contentNode;
  }
  return undef;
}

sub getMaillistContent {
  my ($self, $listname, $domain) = @_;

  return undef unless ( defined Mailman::version() );

  my $maillist_dst = $self->getMaillistDstFile($listname, $domain);

  my $dir = Mailman::getMailListArchiveDir($listname);

  if ( -d $dir) {
    my %options;
    $options{'directory'} = $dir;
    $options{'checkEmptyDir'} = 1;

    my $mlContent = $self->{storage}->addTar($maillist_dst, %options);
    my $contentNode = XmlNode->new('content');
    $contentNode->addChild($self->makeCidNode($mlContent, 'domainmaillist'));
    return $contentNode;
  }
  return undef;
}

sub getFtpUsersDirs {
  my ($self, $account, $subdomains) = @_;

  my ($ptrProftpdRows, $path, $dir, @ftpusersdirs, $ptrRow);

  my $home = $self->getHomeDir($account);

  if (-T "/etc/proftpd/$account") {
      my $fileParser = makeFileParser("/etc/proftpd/$account");
      $ptrProftpdRows = $fileParser->{'PARSE'}->('VALUESEPARATOR'=>':', 'KEYSEPARATOR'=>'',);

      if (ref($ptrProftpdRows)=~/ARRAY/) {
        my ($subdompattern, $pattern, $is_subdomain_directory);
        $pattern = qr/\Q$home\E\/public_html\/(.*)/;
        foreach $ptrRow (@{$ptrProftpdRows}) {
              next unless ($ptrRow->[4] eq $account);
              $path=$ptrRow->[5];
              if ($path=~/$pattern/) {
                $dir =$1;
                $is_subdomain_directory=0;
                foreach $subdompattern (@{$subdomains}) {
 		      $subdompattern = qr/^\Q$subdompattern\E(?:\/.*)?/; 
                      if ($dir=~/$subdompattern/) {
                        $is_subdomain_directory=1;
                        last;
                      }
                }
                next if ($is_subdomain_directory);
                push (@ftpusersdirs, $dir);
              }
        }
      }
  }
  
  return \@ftpusersdirs;
}

sub getSubdomainsDirs {
  my ($self, $domain, $account) = @_;

  my ($ptrUserdomainsHash, $subdomain, $acc, @subdomains, @subdompatterns, $dir);

  unless(ref($ptrUserdomainsHash)=~/HASH/){
        my $fileParser = makeFileParser('/etc/userdomains');
        $ptrUserdomainsHash = $fileParser->{'PARSE'}->('KEYSEPARATOR'=>':');
  }
  if (ref($ptrUserdomainsHash)=~/HASH/) {
        my $pattern = qr/^(.+)\.\Q$domain\E$/;
        while ( ($subdomain,$acc) = each %{$ptrUserdomainsHash} ) {
          next unless ($acc eq $account);
          if ($subdomain=~/$pattern/) {
                $dir = $1;
                push @subdomains,$dir;
#                push @subdompatterns,qr/^\Q$dir\E(?:\/.*)?/;
          }
        }
  }
  return \@subdomains;
}

sub getHomeDir {
        my ($self, $user) = @_;
        my $home;
        my @pw = getpwnam($user);
        if (-1 != $#pw) {
                $home = $pw[7];
        }
        unless ($home) {
                # how it is by default
                $home = "/home/$user";
        }
        return $home;
}

sub makeFileReader
{
        my ($filename, $doNotFail) = @_;
        local (*INPUT);

        unless (-T $filename) {

                if (defined($doNotFail)) {
                        Logging::info("Error: file '$filename' is not a text file");
                } else {
                        Logging::error("Error: file '$filename' is not a text file");
                }

                return undef;
        }

        unless (open(INPUT,"<$filename")) {

                binmode INPUT;
                if (defined($doNotFail)) {
                        Logging::info("Error: unable to open file '$filename' to read");
                } else {
                        Logging::error("Error: unable to open file '$filename' to read");
                }

                return undef;
        }

        my $input = *INPUT;

        my $this = {
                'READ' => sub {
                                my $a = <$input>;
                                return $a;
                        },

                'CLOSE' => sub { close($input); }

        };

        return $this;
}

sub makeParser {
  my $reader;
  my ($this,$ptrRows,$keySep,$valSep,$valKeySep,$comment,
      $typeRows,$maxIndex,$rowIndex);
  {
        $reader = shift;
        unless(defined($reader)) {
          return undef;
        }

    my %args=(@_);
    $keySep = $args{'KEYSEPARATOR'};
    $valSep = $args{'VALUESEPARATOR'};
    $valKeySep = $args{'KEYVALUESEPARATOR'};
  }
  $this={'READER'=>sub {
                   if(@_){
                         $reader=shift;
                   }
                   return $reader;
                 },
                 'FILE'=>sub {
                   if(@_){
                         $reader = makeFileReader(@_);
                   }
                   return undef;
                 },
                 'KEYSEPARATOR'=>sub{
                   if(@_){
                         $keySep=shift;
                   }
                   return $keySep;
                 },
                 'VALUESEPARATOR'=>sub{
                   if(@_){
                         $valSep=shift;
                   }
                   return $valSep;
                 },
                 'KEYVALUESEPARATOR'=>sub{
                   if(@_){
                         $valKeySep=shift;
                   }
                   return $valKeySep;
                 },
                 'COMMENT'=>sub{
                   if(@_){
                         $comment=shift;
                   }
                   return $comment;
                 },
                 'PARSE'=>sub{
                   my %args=(@_);

                   $keySep=$args{'KEYSEPARATOR'} if exists $args{'KEYSEPARATOR'};
                   $valSep=$args{'VALUESEPARATOR'} if exists $args{'VALUESEPARATOR'};
                   $valKeySep=$args{'KEYVALUESEPARATOR'} if exists $args{'KEYVALUESEPARATOR'};
                   $comment=$args{'COMMENT'} if exists $args{'COMMENT'};

                   $ptrRows = parseFile('reader'=>$reader,'keysep'=>$keySep,
                                                                 'valsep'=>$valSep,'keyvalsep'=>$valKeySep,
                                                                 'comment'=>$comment);
                   $typeRows = ref($ptrRows);
                   if($typeRows=~/ARRAY/){
                         $rowIndex=-1;
                         $maxIndex=scalar(@{$ptrRows})-1;
                   }
                   return $ptrRows;
                 },
                 'ROW'=>sub{
                   if($typeRows=~/HASH/){
                         my $key=shift;
                         return $ptrRows->{$key};
                   }elsif($typeRows=~/ARRAY/){
                         if($rowIndex<$maxIndex){
                           $rowIndex++;
                           return $ptrRows->[$rowIndex];
                         }else{
                           return undef;
                         }
                   }else{
                         return undef;
                   }
                 },
                };

  return $this;
}

sub makeFileParser {
  my $filename = shift;
  return makeParser(makeFileReader($filename), @_);
}

sub parseFile {
  my %arg = (@_);
  my $reader = $arg{'reader'};
  my $keySep = $arg{'keysep'};
  my $valSep = $arg{'valsep'};
  my $valKeySep = $arg{'keyvalsep'};
  my $comment = $arg{'comment'};

  my ($ret,$key,$value,$reKey,$reValue,@values,$keyBlank,$valBlank,
     $vkey,$vvalue,$reKeyValue,$reComment);

  if($keySep){
    $ret = {};
    if(ref($keySep)=~/regexp/i || ($keySep eq ' ')){
      $reKey = $keySep;
    }else{
      $reKey = qr/$keySep/;
    }
  }else{
    $ret = [];
  }
  if($valSep){
    if(ref($valSep)=~/regexp/i || ($valSep eq ' ')){
      $reValue = $valSep;
    }else{
      $reValue = qr/$valSep/;
    }
  }
  if($valKeySep){
    if(ref($valKeySep)=~/regexp/i || ($valKeySep eq ' ')){
      $reKeyValue = $valKeySep;
    }else{
      $reKeyValue = qr/$valKeySep/;
    }
  }
  if($comment){
    if(ref($comment)=~/regexp/i){
      $reComment = $comment;
    }else{
      $reComment = qr/$comment/;
    }
  }

  while($_ = $reader->{'READ'}->()){
    chomp;
    next unless $_;
    if($comment && /$reComment/){
          next;
    }
    if ($keySep){
      ($key,$value)=split($reKey,$_,2);
      if($key){
                $key=~s/^\s+//;
                $key=~s/\s+$//;
                if($valSep){

                  if ($valKeySep){
                        $ret->{$key}={};
                        foreach $value (split($reValue,$value)){
                          ($vkey,$vvalue)=split($reKeyValue,$value);
                          $vkey=~s/^\s+//;
                          $vkey=~s/\s+$//;
                          $ret->{$key}->{$value}=$vvalue;
                        }
                  }else{
                        push @{$ret->{$key}},split($reValue,$value);
                  }
                }else{
                  $value=~s/^\s+//;
                  $value=~s/\s+$//;
                  $ret->{$key}=$value;
                }
      }
    }else{
      if($valSep){
                push @{$ret},[split($valSep,$_)];
      }else{
                push @{$ret},$_;
      }
    }
  }

  $reader->{'CLOSE'}->();

  return $ret;
}

1;
