package Dumper;

use strict;

use XmlNode;
use AgentConfig;
use Mailman;
use Parser;
use SpamAssassinCfg;
use Logging;
use Db::DbConnect;
use Db::MysqlUtils;
use Db::Connection;
use EncodeBase64;

my $DEFAULT_RESELLER = 'root';

my %mapLimits = (
  'max_addon'     => 'MAXADDON',
  'max_parked'    => 'MAXPARK',
  'max_ftpusers'  => 'MAXFTP',
  'max_maillists' => 'MAXLST',
  'max_box'       => 'MAXPOP',
  'max_db'        => 'MAXSQL',
  'max_subdom'    => 'MAXSUB',
);

# List of cPanel features.
# Features marked as 2 are included to the dump.
# Do not forget change cPanel.xsd, if change flags.
my %mapFeatures = (
  'addoncgi'               => 1,    #Addon Cgi Scripts
  'addondomains'           => 2,    #Addon Domain Manager
  'advguest'               => 1,    #Advanced Guestbook
  'agora'                  => 1,    #Agora Shopping Cart
  'analog'                 => 1,    #Analog Stats
  'emailauth'              => 1,    #Email Authentication
  'autoresponders'         => 1,    #Autoresponder Manager
  'awstats'                => 1,    #Awstats Stats
  'backup'                 => 2,    #Backup Manager
  'bandwidth'              => 1,    #Bandwidth Stats
  'bbs'                    => 1,    #PHPBB2
  'blockers'               => 1,    #Email Filtering Manager
  'boxtrapper'             => 1,    #BoxTrapper Spam Trap
  'cgi'                    => 1,    #CGI Center
  'changemx'               => 1,    #Ability to Change MX
  'chat'                   => 1,    #Chat Rooms
  'clamavconnector_scan'   => 2,    #Virus Scanner
  'clock'                  => 1,    # Java Clock
  'countdown'              => 1,    #Java Countdown
  'counter'                => 1,    #Counter
  'cpanelpro_support'      => 1,    #Support System Submission
  'cpanelpro_images'       => 1,    #Image Manager
  'cron'                   => 2,    #Crontab
  'defaultaddress'         => 1,    #Default Address Manager
  'diskusageviewer'        => 1,    #Disk Usage Viewer
  'emaildomainfwd'         => 1,    #Email Domain Forwarding
  'emailscripts'           => 1,    #Email Scripts (cgiemail,formmail)
  'entropybanner'          => 1,    #Entropy Banner
  'entropysearch'          => 1,    #Entropy Search
  'errlog'                 => 1,    #Error Log
  'errpgs'                 => 1,    #Customer Error Pages
  'fantastico'             => 1,    #
  'filemanager'            => 1,    #File Manager
  'forwarders'             => 1,    #Forwarder Manager
  'frontpage'              => 1,    #Frontpage
  'ftpaccts'               => 1,    #Ftp Account Manager
  'ftpsetup'               => 2,    #Ftp Settings
  'getstart'               => 1,    #Getting Started Wizard
  'guest'                  => 1,    #Simple Guestbook
  'handlers'               => 1,    #Apache Handlers Manager
  'hotlink'                => 1,    #Hotlink
  'ipdeny'                 => 1,    #IP Deny Manager
  'indexmanager'           => 1,    #File Manager Directory Selection
  'interchange'            => 1,    #Interchange Shopping Cart
  'lastvisits'             => 1,    #Latest Visitors
  'cpanelpro_leechprotect' => 1,    #Leech Protect
  'lists'                  => 2,    #Mailman List Manager
  'mime'                   => 1,    #Mime Types Manager
  'modules-perl'           => 1,    #Install Perl Modules
  'modules-php-pear'       => 1,    #Install PHP Pear Modules
  'modules-ruby'           => 1,    #Install Ruby Modules
  'mysql'                  => 1,    #Mysql
  'nettools'               => 1,    #Network Tools
  'parkeddomains'          => 2,    #Parked Domain Manager
  'password'               => 2,    #Password Change
  'pgp'                    => 1,    #PGP/GPG
  'php-config'             => 1,    #See PHP Configuration
  'phpmyadmin'             => 1,    #PhpMyAdmin
  'phppgadmin'             => 1,    #PhpPgAdmin
  'popaccts'               => 1,    #Email Account Manager
  'postgres'               => 1,    #PostgresSQL
  'randhtml'               => 1,    #Random Html Generator
  'rawlog'                 => 2,    #Raw Access Logs
  'redirects'              => 1,    #Redirect Manage
  'ror'                    => 1,    #Ruby on Rails
  'scgiwrap'               => 1,    #Simple Cgi Wrapper
  'searchsubmit'           => 1,    #Search Engine Submit Tool
  'serverstatus'           => 1,    #Server Status Viewer
  'setlang'                => 1,    #Change Language
  'spamassassin'           => 2,    #SpamAssassin
  'spambox'                => 1,    #SpamAssassin Spam Box
  'ssh'                    => 1,    #SSH Connection Window
  'sslinstall'             => 1,    #SSL Host Installer
  'sslmanager'             => 1,    #SSL Manager
  'statmanager'            => 2,    #Statistics Program Manager
  'statselect'             => 1,    #Choose Log Programs
  'style'                  => 1,    #Change Style
  'subdomains'             => 2,    #Subdomain Manager
  'subdomainstats'         => 1,    #Subdomain Stats
  'traceaddy'              => 1,    #Ability to Trace an email address
  'updatecontact'          => 1,    #Update Contact Information
  'videotut'               => 1,    #Video Tutorials
  'webalizer'              => 1,    #Webalizer Stats
  'webdisk'                => 1,    #Web Disk
  'webmail'                => 1,    #Webmail
  'webprotect'             => 1,    #Webprotect
  'backupwizard'           => 1,    #Backup Wizard
);


#
# global variables
#

my $ptrQuota;

my %servers_ips;

my $ptrUserdomainsHash;

my %trueUserDomains;

my %anonftps;

my %accountNodes;

my %apacheAliases;

my (%cPanelConfig);

my (%featureList);

my $installDir = '/var/cpanel';

#
# end global variables
#


my $wrapUserMysql;


#
# Initialization routine
#
sub initialize {

  loadMainConfig();

  my @all_accounts = getAllAccounts();

  %featureList = loadFeatureList();

  loadQuota();
}


#
# Parse main config file, 
# Initialize $installDir and cPanelConfig{'engineroot'}
#
sub loadMainConfig {
  my $pathToConf = shift || '';
  my $ptrHash = shift || \%cPanelConfig;

  unless ( -T $pathToConf ) {
    foreach my $prefix ( './', '/var/cpanel/' ) {
      $pathToConf = $prefix . 'cpanel.config';
      if ( -T $pathToConf ) {
        last;
      }
    }
  }

  if ( -T $pathToConf ) {
    if ( ref($ptrHash) =~ /HASH/ ) {
      if ( $pathToConf =~ /(.+)\/[^\/]+/ ) {
        $ptrHash->{'installDir'} = $1;
      }
      if ( open( CF, "<$pathToConf" ) ) {
        binmode CF;
        my ( $key, $value );
        while ( my $line = <CF> ) {
          chomp $line;
          next unless $line;
          next if ( $line =~ /^#/ );
          ( $key, $value ) = split( /\s*=\s*/, $line, 2 );
          $key =~ s/^\s+//;
          $ptrHash->{$key} = $value;
        }
        close(CF);
      }
    }
  }
  else {
    $pathToConf = '';
  }

  $installDir = $cPanelConfig{'installDir'} if defined ( $cPanelConfig{'installDir'} );

  return $pathToConf;
}

# Returns home directory for the system account $user
sub getHomeDir {
	my ($user) = @_;
	my $home;
	my @pw = getpwnam($user);
	if (-1 != $#pw) {
		$home = $pw[7];
	}
	unless ($home) {
		# how it is by default
		$home = '/home/$account';
	}
	return $home;
}

#######################################################
#
#  Get all client accounts, registered in the system
#
#######################################################
sub getAllAccounts {
  my @all_accounts;

  my $users_dir = "$installDir/users";
  opendir( USRS, $users_dir ) || die "Can't open directory $users_dir: $!";
  my @accts = readdir(USRS);
  closedir(USRS);
  chomp(@accts);

  # remove duplicate and possible wrong entries
  foreach my $account (@accts) {
    next if ( $account =~ /\./ || grep( /^$account$/, @all_accounts ) );
    push @all_accounts, $account;
  }

  return @all_accounts;
}

#
# parse httpd.conf, init $servers_ips
#
sub parseHttpd {
  my $httpdconf_file = "/etc/httpd/conf/httpd.conf";
  unless ( open( INPUT, "<$httpdconf_file" ) ) {
     Logging::error("Error: unable to open '$httpdconf_file': $!");
  }
  binmode INPUT;
  my ( $ip, $state, @vDomains, $vDomain );
  $state = 0;
  while (<INPUT>) {
    chomp;
    next unless $_;
    next if (/^#/);
    if ( $state == 1 ) {
      if (/<\/VirtualHost>/) {
        foreach $vDomain (@vDomains) {
          $vDomain =~ s/^www\.//;
          $servers_ips{$vDomain} = $ip;
        }
        @vDomains = ();
        $state    = 0;
      }
      elsif (/^\s*Server(?:Name|Alias)\s+(.*)$/) {
        push @vDomains, split( ' ', $1 );
      }
    }
    else {
      if (/<VirtualHost\s(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})>/) {
        $ip       = $1;
        @vDomains = ();
        $state    = 1;
      }
    }
  }
  close(INPUT);
  return \@vDomains;
}

sub loadQuota {
  # Get disk quotas
  my $fileUser = Parser::makeFileParser("/etc/quota.conf");
  $ptrQuota = $fileUser->{'PARSE'}->('KEYSEPARATOR' => '=','VALUESEPARATOR' => '');
  unless ( ref($ptrQuota) =~ /HASH/ ) {
    $ptrQuota = {};
  }
}

sub loadFeatureList {
  my %list;

  my $features_dir = "$installDir/features";
  opendir( FLST, $features_dir ) || die "Can't open directory $features_dir: $!";
  my @lsts = readdir(FLST);
  closedir(FLST);

  foreach my $lst (@lsts) {
    my $cfg = "$features_dir/$lst";
    if ( -T $cfg ) {
      Logging::info("Reading feature list: $lst");
      my $cfgParser = Parser::makeFileParser($cfg);
      my $values = $cfgParser->{'PARSE'}->( 'KEYSEPARATOR' => '=' );
      $list{$lst} = $values;
      Logging::info("Feature list '$lst' is read");
    }
  }

  #convert disabled options
  if ( exists $list{'disabled'} ) {
    Logging::info("Apply disabled feature list");
    foreach my $key ( keys %list ) {
      if ( not $key eq 'disabled' ) {
        my $vals = $list{$key};
        foreach my $diskey ( keys %{ $list{'disabled'} } ) {
          ${$vals}{$diskey} = 0
            if ${ $list{'disabled'} }{$diskey} == 0
              and exists ${$vals}{$diskey};
        }
      }
    }
  }

  #create default
  if ( exists $list{'default'} ) {
    Logging::info("Apply default feature list");
    foreach my $mapkey ( keys %mapFeatures ) {
      ${ $list{'default'} }{$mapkey} = 1
        if not exists ${ $list{'default'} }{$mapkey};
    }
  }
  return %list;

}

sub getUserDomains {
  unless ( keys %trueUserDomains ) {
    my $fileParser = Parser::makeFileParser('/etc/trueuserdomains');
    my $ptrHash = $fileParser->{'PARSE'}->( 'KEYSEPARATOR' => ':' );
    if ( ref($ptrHash) =~ /HASH/ ) {
      %trueUserDomains = ( %{$ptrHash} );
    }
  }
  return \%trueUserDomains;
}

sub getDomainName {
  my $account = shift;
  my $userDomains = getUserDomains();
  if ( ref($userDomains) =~ /HASH/ ) {
    my ( $domain, $owner );
    while ( ( $domain, $owner ) = each( %{$userDomains} ) ) {
      if ( $owner eq $account ) {
        return $domain;
      }
    }
  }
  return undef;
}

sub getDomainOwner {
  my $domain = shift;

  my $userDomains = getUserDomains();
  if ( ref($userDomains) =~ /HASH/ ) {
    return $userDomains->{$domain};
  }
}

sub epoch2CrDate {
  my $epoch = shift;
  unless ( $epoch ) {
    return "1970-01-01";
  }
  my ($mday, $mon, $year) = (localtime($epoch)) [3 .. 5];
  return sprintf("%04D-%02D-%02D", $year+1900, $mon+1, $mday);
}

sub makeAccountNode {
  
  my ( $account, $ptrUserHash, $isReseller, $shallowMode ) = @_;

  my (
    $ptrLines, $ptrHash, $xmlKey, 	$userKey,
    $value,    $key,     $domain,   $item
  );

  unless ($shallowMode) {
    if ( exists $accountNodes{ $account } ) {
      return $accountNodes{ $account };
    }
  }

  my $home = getHomeDir($account);

  my $root = XmlNode->new('account');
  $root->setAttribute( 'name', $account );

  unless ($shallowMode) {
    if ( -T "$home/.contactemail" ) {

      my $fileParser = Parser::makeFileParser("$home/.contactemail");
      $ptrLines = $fileParser->{'PARSE'}->();
      if ( ref($ptrLines) =~ /ARRAY/ ) {
        $root->setAttribute( 'email', $ptrLines->[0] || '' );
      }

    }
    else {
      $root->setAttribute( 'email', '' );
    }
  }

  unless ( ref($ptrUserHash) =~ /HASH/ ) {
    my $fileParser = Parser::makeFileParser("$installDir/users/$account");
    $ptrUserHash = $fileParser->{'PARSE'}->( 'KEYSEPARATOR' => '=' );
  }

  unless ($shallowMode) {
    if ( ref($ptrUserHash) =~ /HASH/ ) {
      $root->setAttribute( 'date', epoch2CrDate( $ptrUserHash->{'STARTDATE'} ) );
    }

    unless ($isReseller) {
      my $diskSpace = $ptrQuota->{$account};
      if ( $diskSpace =~ /(\d+)/ ) {
        $diskSpace = $1 * 1024 * 1024;
        $root->addChild(&makeLimitNode('disk_space', $diskSpace));
      }

      if ( ref($ptrUserHash) =~ /HASH/ ) {
        # handle limits
        $mapLimits{'max_traffic'} = 'BWLIMIT';
        while ( ( $xmlKey, $userKey ) = each(%mapLimits) ) {
          $value = $ptrUserHash->{$userKey};
          if ( $value =~ /\d+/ ) {
            $root->addChild(&makeLimitNode($xmlKey, $value));
          }
          else {
            $root->addChild(&makeLimitNode($xmlKey));
          }
        }

        delete $mapLimits{'max_traffic'};    ## clean map
      }
    }
    if ( ref($ptrUserHash) =~ /HASH/ ) {
      makeFeaturesNode( $root, $ptrUserHash->{'FEATURELIST'} ) if exists $ptrUserHash->{'FEATURELIST'};
    }

    # SpamAssssin migration
    makeSpamassassinNode( $root, $home );
  }

  #
  # Handle domain
  #
  $domain = &getDomainName($account);

  if ($domain) {

    Logging::info("Domain '$domain' is started");

    $item = &makeIpNode( $domain, $ptrUserHash );

    $root->addChild($item);

    $item = &makeDomainNode( $domain, $account, $shallowMode );
    $root->addChild($item);

    Logging::info("Domain '$domain' is dumped");
  }

  unless ($shallowMode) {
    $accountNodes{$account} = $root;
  }

  return $root;
}

sub makeFeaturesNode {
  my ( $root, $feature ) = @_;

  if ( exists $featureList{$feature} ) {
    my $fvalues = $featureList{$feature};
    foreach my $fkey ( keys %mapFeatures ) {
      if ( 2 == $mapFeatures{$fkey} and exists ${$fvalues}{$fkey} ) {
        my $fnode = XmlNode->new( 'feature' );
        $fnode->setAttribute( 'name', $fkey );
        if ( ${$fvalues}{$fkey} ) {
          $fnode->setAttribute( 'allowed', 'true' );
        }
        else {
          $fnode->setAttribute( 'allowed', 'false' );
        }
        $root->addChild($fnode);
      }
    }
  }
  else { Logging::info("Cannot get featureList '$feature'"); }
}

sub makeIpNode {
  my ( $domain, $ptrUserHash ) = @_;

  my ( $ptrRows, $ip, $item, $ptrRow, $dmn );

  unless ( $ip = $servers_ips{$domain} ) {
    my $fileParser = Parser::makeFileParser("$installDir/accts.db");
    $ptrRows = $fileParser->{'PARSE'}->( 'VALUESEPARATOR' => ',' );
    foreach $ptrRow ( @{$ptrRows} ) {
      $dmn = $ptrRow->[0];
      $dmn =~ s/^\s+//;
      $dmn =~ s/\s+$//;
      if ( $ptrRow->[2] =~ /(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})/ ) {
        $ip = $1;
        unless ( exists( $servers_ips{$domain} ) ) {
          $servers_ips{$domain} = $ip;
        }
      }
    }
    unless ( $ip = $servers_ips{$domain} ) {
      unless ( $ip = $ptrUserHash->{'IP'} ) {
        my $zone = "/var/named/$domain.db";
        my $pattern =
          qr/^\Q$domain\E\.\s+(?:\d+\w?\s+)IN\s+A\s+(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})/;
        if ( -T $zone ) {
          if ( open( INPUT, "<$zone" ) ) {
            binmode INPUT;
            while (<INPUT>) {
              if (/$pattern/) {
                $ip = $1;
                last;
              }
            }
            close(INPUT);
            if ($ip) {
              $servers_ips{$domain} = $ip;
            }
          }
        }
      }
    }
  }
  unless ( $ip ) {
    $ip = getMainIP();
  }
  $item = XmlNode->new( 'ip' );
  $item->setText( $ip );

  return $item;
}

sub makeDomainNode {
  my ( $domain, $account, $shallowMode ) = @_;

  my $root = XmlNode->new('domain');
  $root->setAttribute( 'name', $domain );

  my (
    @subdomains, @ftpusersdirs,   $subdomain, $acc,
    $ptrRows,    $ptrRow,         $path,      @dirs,
    $dir,        @subdompatterns, @exclude,   $dumpFile,
    $item,       $ptrProftpdRows
  );

  my $home = getHomeDir($account);

  #
  # subdomain dirs
  #
  unless ( ref($ptrUserdomainsHash) =~ /HASH/ ) {
    my $fileParser = Parser::makeFileParser('/etc/userdomains');
    $ptrUserdomainsHash = $fileParser->{'PARSE'}->( 'KEYSEPARATOR' => ':' );
  }
  if ( ref($ptrUserdomainsHash) =~ /HASH/ ) {
    my $pattern = qr/^(.+)\.\Q$domain\E$/;
    while ( ( $subdomain, $acc ) = each %{$ptrUserdomainsHash} ) {
      next unless ( $acc eq $account );
      if ( $subdomain =~ /$pattern/ ) {
        $dir = $1;
        push @subdomains,     $dir;
        push @subdompatterns, qr/^\Q$dir\E(?:\/.*)?/;
      }
    }
  }
  #
  # end subdomain dirs
  #

  #
  # ftp dirs
  #
  if ( -T "/etc/proftpd/$account" ) {
    my $fileParser = Parser::makeFileParser("/etc/proftpd/$account");
    $ptrProftpdRows = $fileParser->{'PARSE'}->(
      'VALUESEPARATOR' => ':',
      'KEYSEPARATOR'   => '',
    );
    if ( ref($ptrProftpdRows) =~ /ARRAY/ ) {
      my ( $subdompattern, $pattern, $is_subdomain_directory );
      $pattern = qr/\Q$home\E\/public_html\/(.*)/;
      foreach $ptrRow ( @{$ptrProftpdRows} ) {
        next unless ( $ptrRow->[4] eq $account );
        $path = $ptrRow->[5];
        if ( $path =~ /$pattern/ ) {
          $dir                    = $1;
          $is_subdomain_directory = 0;
          foreach $subdompattern (@subdompatterns) {
            if ( $dir =~ /$subdompattern/ ) {
              $is_subdomain_directory = 1;
              last;
            }
          }
          next if ($is_subdomain_directory);
          push( @ftpusersdirs, $dir );
        }
      }
    }
  }
  @subdompatterns = ();
  #
  # end ftp dirs
  #

  unless ($shallowMode) {
    if ( $item = &makeAnonFtpdNode($account) ) {
      $root->addChild($item);
    }
  }

  &addFtpUserNodes($root, $domain, $account, \@subdomains, \@ftpusersdirs, $ptrProftpdRows);

  unless ($shallowMode) {
    &addProtectedDirs($root, $domain, $account,	\@subdomains);
  }

  &addSubDomains($root, $domain, $account, \@subdomains, $shallowMode);

  &addAddOnDomains($root, $domain, $account);

  unless ($shallowMode) {
    &addMailNode($root, $domain, $account);

    &addDatabases($root, $account);

    &addMailLists($root, $domain, $account);
  }

  return $root;
}

sub addMailLists {
  my ( $root, $domain, $account ) = @_;

  unless ( defined Mailman::version() ) {
    Logging::info("Unable to found Mailman installation. Skip dumping maillists.");
    return;
  }

  Logging::info("Get maillists of '$domain'");
  my $maillists = XmlNode->new( 'maillists' );

  my @lists = Mailman::getMailLists($domain);
  foreach my $listname (@lists) {
    Logging::info("Found maillist '$listname'");
    my @owners = Mailman::getListOwners($listname);
    if (@owners) {
      my $reallistname = $listname;
      $reallistname =~ s/(.*)(_$domain)/$1/;
      my $maillist = XmlNode->new( 'maillist' );
      $maillist->setAttribute( 'name', $reallistname );
      for my $listOwner (@owners) {
      	my $owner_node = XmlNode->new( 'owner' );
      	$owner_node->setText( $listOwner );
        $maillist->addChild($owner_node);
      }
      $maillist->addChild( makePasswordNode( Mailman::getListPassword($listname) , 'encrypted' ));

      my %listMembers = Mailman::getListMembers($listname);
      for my $memberEmail ( keys %listMembers ) {
        my $member = XmlNode->new( 'recipient' );
        $member->setText( $memberEmail );
        if ( $listMembers{$memberEmail} ne "" ) {
          $member->setAttribute( 'fullname', $listMembers{$memberEmail} );
        }
        $maillist->addChild($member);
      }

      $maillists->addChild($maillist);

    }
    else {
      Logging::info("Bad maillist '$listname', cannot find owner, skipped");
    }

  $root->addChild($maillists);
  }
}

sub addDatabases {
  my ( $root, $account ) = @_;
  addMySQLDatabases( $root, $account );
  addPgSQLDatabases( $root, $account );
}

sub addMySQLDatabases {
  my ( $root, $account ) = @_;

  my ( $ptrRow, $database, $item, $dumpFile, $sql, @dbNames );

  unless ( ( ref($wrapUserMysql) =~ /HASH/ )
    && ( ref( $wrapUserMysql->{'EXECUTE'} ) =~ /CODE/ ) )
  {

    $wrapUserMysql = Db::DbConnect::getDbConnect( 'mysql', 'root', undef, 'mysql', 'localhost' );
  }

  my $datadir;
  $sql = "SHOW variables LIKE 'datadir'";
  if (  $wrapUserMysql->{'EXECUTE'}->($sql)
    and $ptrRow = $wrapUserMysql->{'FETCHROW'}->() )
  {
    $datadir = $ptrRow->[1];
  }
  $wrapUserMysql->{'FINISH'}->();

  $sql =
      "SELECT DISTINCT db FROM db WHERE db like '"
    . $account
    . "\_%' or db like '"
    . $account
    . "\\\_%'";
  if ( $wrapUserMysql->{'EXECUTE'}->($sql) ) {
    while ( $ptrRow = $wrapUserMysql->{'FETCHROW'}->() ) {
      push @dbNames, $ptrRow->[0];
    }
  }
  $wrapUserMysql->{'FINISH'}->();

  foreach $database (@dbNames) {
    $database =~ s/\\_/_/g;
    next unless -d $datadir . $database;

    $item = XmlNode->new('database');
    $item->setAttribute( 'name',    $database );
    $item->setAttribute( 'type',    'mysql' );
    $item->setAttribute( 'version', Db::MysqlUtils::getVersion() );

    Logging::info("Started $database dumping...");

    Db::DbConnect::addDbUsers( $item, $database, $wrapUserMysql );

    $root->addChild($item);

    Logging::info("done");
  }
}

sub addPgSQLDatabases {
  my ( $root, $account ) = @_;

  my ( $ptrRow, $database, $item, $dumpFile, $sql, @dbNames, %dbUsers );

  if ( !AgentConfig::psqlBin() ) {
    Logging::warning("Unable to find psql");
    return;
  }
  $Db::Connection::UsePostgreShellBackendAsSudo = 1;

  my $wrapUserPgsql =
    getDbConnect( 'postgresql', 'postgres', undef, 'template1', 'localhost' );
  if ( ref($wrapUserPgsql) eq 'HASH' ) {
    if (  ref( $wrapUserPgsql->{'EXECUTE'} ) eq 'CODE'
      and ref( $wrapUserPgsql->{'FETCHROW'} ) eq 'CODE'
      and ref( $wrapUserPgsql->{'FINISH'} )   eq 'CODE' )
    {
      if (
        $wrapUserPgsql->{'EXECUTE'}->(
          "select datname from pg_database where datname like '$account\_\_%'"
        )
        )
      {
        while ( $ptrRow = $wrapUserPgsql->{'FETCHROW'}->() ) {
          Logging::info( "Found database: " . $ptrRow->[0] );
          push @dbNames, $ptrRow->[0];
        }
        $wrapUserPgsql->{'FINISH'}->();
        if (
          $wrapUserPgsql->{'EXECUTE'}->(
            "select usename, passwd from pg_shadow where usename like '$account\_\_%'"
          )
          )
        {
          while ( $ptrRow = $wrapUserPgsql->{'FETCHROW'}->() ) {
            Logging::info( "Found database user: " . $ptrRow->[0] );
            $dbUsers{ $ptrRow->[0] } = $ptrRow->[1];
          }
          $wrapUserPgsql->{'FINISH'}->();
        }
      }
    }
    else { Logging::info("Connection to postgresql is not valid"); }
  }
  else { Logging::info("Cannot connect to postgresql"); }

  my $psql = AgentConfig::psqlBin();

  my @out     = `$psql --version | awk '{print \$3}'`;
  my $version = '';
  chomp $out[0];
  if ( $out[0] =~ /(\d+\.\d+\.\d+).*/ ) {
    $version = $1;
  }

  foreach $database (@dbNames) {
    $item = XmlNode->new('database');
    $item->setAttribute( 'name',    $database );
    $item->setAttribute( 'type',    'postgresql' );
    $item->setAttribute( 'version', $version );

    Logging::info("Started $database dumping...");

    Logging::info("Started $database user dumping...");
    if ( $wrapUserPgsql->{'EXECUTE'}
      ->("select datacl from pg_database where datname='$database'") )
    {
      while ( $ptrRow = $wrapUserPgsql->{'FETCHROW'}->() ) {
        Logging::info( "Parse DACL: " . $ptrRow->[0] );
        my @usernames = keys(%dbUsers);
        my $uname;
        foreach $uname (@usernames) {
          if ( $ptrRow->[0] =~ m/.*$uname\=.*/m ) {
            Logging::info("User '$uname' granted to '$database");
            my $useritem = XmlNode->new( 'dbuser' );
            $useritem->setAttribute( 'name', $uname );
            my $pwditem = makePasswordNode( $dbUsers{$uname}, 'encrypted' );
            if ($pwditem) {
              $useritem->addChild($pwditem);
            }
            $item->addChild($useritem);
          }
        }
      }
      $wrapUserPgsql->{'FINISH'}->();
    }
    Logging::info("$database user dumping done");

    $root->addChild($item);

    Logging::info("done");
  }
}

sub addSubDomains {
  my ( $root, $domain, $account, $ptrSubDomains, $shallowMode ) = @_;

  my (
    $nodeSubDomain, $subdomain,  $dumpFile,
    $subdomainRoot, $fileParser, $ptrRows,
    $ptrRow,        $item,       @exclude
  );

  my $home = getHomeDir($account);

  foreach $subdomain ( @{$ptrSubDomains} ) {

    Logging::info( "Subdomain '$subdomain.$domain' ..." );

    $subdomainRoot = "$home/public_html/$subdomain";

    $nodeSubDomain = XmlNode->new('subdomain');
    $nodeSubDomain->setAttribute( 'name', $subdomain, 'noencode' );
    $root->addChild($nodeSubDomain);

    #
    # ftp user
    #
    my $ftp_accounts_file = "/etc/proftpd/$account";
    if ( -T $ftp_accounts_file ) {
      $fileParser = Parser::makeFileParser($ftp_accounts_file);
      $ptrRows = $fileParser->{'PARSE'}->( 'VALUESEPARATOR' => ':' );
      if ( ref($ptrRows) =~ /ARRAY/ ) {
        foreach $ptrRow ( @{$ptrRows} ) {
          if ( ( $ptrRow->[0] eq $subdomain )
            && ( $ptrRow->[4] eq $account ) )
          {
            if ( $ptrRow->[1] ) {
              $item = &makeFtpUserNode( $subdomain, $ptrRow->[1], undef );
              $nodeSubDomain->addChild($item);
            }
            last;
          }
        }
      }
    }

    # Dump mail system and addon domains: if there is 1 addon domain for
    # given subdomain - dump mail system of ADDON DOMAIN, not subdomain.
    # This case reflects common usage of addon domains in cPanel...
    {
      my $maildomain = "$subdomain.$domain";

      my @addondomains = addAddOnDomains( $nodeSubDomain, "$subdomain.$domain", $account );
      if ( 0 == $#addondomains ) {
        $maildomain = pop @addondomains;
      }

      unless ($shallowMode) {
        addMailNode( $nodeSubDomain, $maildomain, $account );
      }
    }
    Logging::info(' OK');
  }
}

###
# Returns array of addon domain names that has been added.
#
# @return array
###
sub addAddOnDomains {
  my ( $root, $domain, $account ) = @_;

  my ( $userdomain, $owner, %userdomains, $maindomain, %aliases, $alias, $item, @found );

  unless ( keys %apacheAliases ) {
    my $httpdconf_file = "/etc/httpd/conf/httpd.conf";
    unless ( open( INPUT, "<$httpdconf_file") ) {
      Logging::error("Error: unable to open '$httpdconf_file': $!");
      return ();
    }
    binmode INPUT;
    my ( $state, $vserver, %valiases );
    $state = 0;
    while (<INPUT>) {
      chomp;
      if ( $state == 1 ) {
        if (/^\s*<\/VirtualHost\s*>/) {
          if ( $vserver && ( keys %valiases ) ) {
            push @{ $apacheAliases{$vserver} }, keys %valiases;
          }
          $vserver  = '';
          %valiases = ();
          $state    = 0;
        }
        elsif (/^\s*ServerName\s+(\S+)/) {
          $vserver = $1;
          $vserver =~ s/^www\.//;
        }
        elsif (/^\s*ServerAlias\s+(.*)/) {
          map { s/^www\.//; $valiases{$_} = 1 } split( ' ', $1 );
        }
      }
      else {
        if (/^\s*<VirtualHost\s.*>/) {
          $state    = 1;
          $vserver  = '';
          %valiases = ();
        }
      }
    }
    close(INPUT);
  }

  unless ( exists $apacheAliases{$domain} ) {
    return ();
  }

  unless ( ref($ptrUserdomainsHash) =~ /HASH/ ) {
    my $fileParser = Parser::makeFileParser('/etc/userdomains');
    $ptrUserdomainsHash = $fileParser->{'PARSE'}->( 'KEYSEPARATOR' => ':' );
  }

  $maindomain = &getDomainName($account);

  if ( ref($ptrUserdomainsHash) =~ /HASH/ ) {
    my $pattern = qr/\Q$maindomain\E$/;
    while ( ( $userdomain, $owner ) = each( %{$ptrUserdomainsHash} ) ) {
      next unless $owner eq $account;
      next if ( $userdomain =~ /$pattern/ );    ## exclude subdomains
      $userdomains{$userdomain} = 1;
    }
  }

  %aliases = map { $_ => 1 } @{ $apacheAliases{$domain} };
  foreach $alias ( keys %aliases ) {
    next unless exists $userdomains{$alias};
    $item = XmlNode->new( 'addondomain' );
    $item->setText( $alias );
    $root->addChild($item);
    push @found, $alias;
  }

  return @found;
}

sub addMailNode {
  my ( $root, $domain, $account ) = @_;

  my (
    $mailNode,   $ptrRows, $ptrRow, $default,  $item,
    $box,        $ptrHash, $value,  $password, $dumpFile,
    $ptrAliases, $src,     $dst,    $from
  );

  my $home = getHomeDir($account);

  Logging::info("Mail for $domain migration started...");
  $mailNode   = XmlNode->new('mail');
  $ptrAliases = [];
  if ( -T "/etc/valiases/$domain" ) {
    my $fileParser = Parser::makeFileParser("/etc/valiases/$domain");
    $ptrAliases = $fileParser->{'PARSE'}->( 'VALUESEPARATOR' => ':' );
    if ( ref($ptrAliases) =~ /ARRAY/ ) {
      foreach $ptrRow ( @{$ptrAliases} ) {
        if ( $ptrRow->[0] eq '*' ) {
          if ( $ptrRow->[1] =~ /\s*([^@]+\@[^@+]+)\s*/ ) {
            $default = $1;
            last;
          }
          elsif ( $ptrRow->[2] =~ /blackhole/ ) {
            $default = ':blackhole:';
            last;
          }
          elsif ( $ptrRow->[2] =~ /fail/ ) {
            $default = ':fail:' . $ptrRow->[3];
            last;
          }
        }
      }
    }
  }

  if ($default) {
    $item = XmlNode->new('default');
    if ( $default eq ':blackhole:' ) {
      $item->setAttribute( 'target', 'ignore' );
    }
    elsif ( $default =~ /:fail:(.*)/ ) {
      $item->setAttribute( 'target', 'fail' );
      $item->setText($1);
    }
    else {
      $item->setAttribute( 'target', 'email' );
      $item->setText($default);
    }
    $mailNode->addChild($item);
  }

  Logging::info("Examining $home/etc/$domain");

  if ( -d "$home/etc/$domain" && -T "$home/etc/$domain/passwd" ) {
    my $fileParser = Parser::makeFileParser("$home/etc/$domain/passwd");
    $ptrRows = $fileParser->{'PARSE'}->( 'VALUESEPARATOR' => ':' );

    if ( ref($ptrRows) =~ /ARRAY/ ) {
      foreach $ptrRow ( @{$ptrRows} ) {
        $box  = $ptrRow->[0];
        $item = XmlNode->new('mailname');
        $item->setAttribute( 'name', $box );

        Logging::info( "migrating " . $box . "@" . "$domain" );

        #
        # password
        #
        $password = '';
        my $shadowParser = Parser::makeFileParser("$home/etc/$domain/shadow");
        $ptrHash = $shadowParser->{'PARSE'}->(
          'KEYSEPARATOR'   => ':',
          'VALUESEPARATOR' => ''
        );

        if ( ref($ptrHash) =~ /HASH/ ) {
          if ( $value = $ptrHash->{$box} ) {
            ($password) = split( ':', $value );
          }
        }
        $item->setAttribute( 'password', $password );
        #
        # end password
        #

        #
        # quota
        #
        if ( -T "$home/etc/$domain/quota" ) {
          my $quotaParser = Parser::makeFileParser("$home/etc/$domain/quota");
          $ptrHash =
            $quotaParser->{'PARSE'}->( 'KEYSEPARATOR' => ':', 'VALUESEPARATOR' => '' );
          if ( ref($ptrHash) =~ /HASH/ && ( $value = $ptrHash->{$box} ) ) {
            $item->setAttribute( 'quota', sprintf( "%.0f", $value ) );
          }
        }
        #
        # end quota
        #

				# SpamAssssin migration
				# Cannot dump individual mailuser spamassassin settings
        # makeSpamassassinNode( $item, "$home/etc/$domain/$box" );

        $mailNode->addChild($item);
      }
    }
  }
  if ( @{$ptrAliases} ) {
    addForwarders( $mailNode, $ptrAliases );
    addAutoresponders( $mailNode, $account, $ptrAliases );
  }

  $root->addChild($mailNode);
  Logging::info("Finish dumping mail configuration for $domain.");
}

sub makeSpamassassinNode {
  my ( $root, $home ) = @_;

  return unless -e "$home/.spamassassin";

  my $saEnabled = 'off';
  $saEnabled = 'on' if -e "$home/.spamassassinenable";

  my $sanode = XmlNode->new('spamassassin');
  $sanode->setAttribute( 'status', $saEnabled );

  $root->addChild($sanode);

  return unless -e "$home/.spamassassin/user_prefs";

  my $saConfig =
    SpamAssassinCfg::parseConfig( "$home/.spamassassin/user_prefs", $home );

  my $cfgvalues = SpamAssassinCfg::getConfigRequireScore($saConfig);
  $sanode->setAttribute( 'hits', $cfgvalues ) if $cfgvalues;

  $cfgvalues = SpamAssassinCfg::getConfigRewriteHeadr($saConfig);
  if ($cfgvalues) {
    if ( scalar( @{$cfgvalues} ) == 2 ) {
      $sanode->setAttribute( 'subj-text', ${$cfgvalues}[1] )
        if ${$cfgvalues}[0] eq 'subject';
    }
    $sanode->setAttribute( 'action', 'mark' );
  }

  $cfgvalues = SpamAssassinCfg::getConfigBlackList($saConfig);
  makeSpamassassinLists( $sanode, $cfgvalues, 'blacklist-member' );

  $cfgvalues = SpamAssassinCfg::getConfigWhiteList($saConfig);
  makeSpamassassinLists( $sanode, $cfgvalues, 'whitelist-member' );

  $cfgvalues = SpamAssassinCfg::getConfigUnBlackList($saConfig);
  makeSpamassassinLists( $sanode, $cfgvalues, 'unblacklist-member' );

  $cfgvalues = SpamAssassinCfg::getConfigUnWhiteList($saConfig);
  makeSpamassassinLists( $sanode, $cfgvalues, 'unwhitelist-member' );
}

sub makeSpamassassinLists {
  my ( $root, $cfgvalues, $listname ) = @_;
  return unless defined $cfgvalues;

  foreach my $item ( @{$cfgvalues} ) {
    if ($item) {
      my $list_name_node = XmlNode->new( $listname );
      $list_name_node->setText( $item );
      $root->addChild( $list_name_node );
    }
  }
}

sub addAutoresponders {
  my ( $mailNode, $account, $ptrAliases ) = @_;
  my $home = getHomeDir($account);

  foreach my $ptrRow ( @{$ptrAliases} ) {

    my $src = $ptrRow->[0];
    my $dst = $ptrRow->[1];
    chomp $src;

    next unless ( $dst =~ /^\s*\|.*\/cpanel\/bin\/autorespond/ );
    next unless ( $src =~ /^([^@]+@[^@]+)$/ );
    next unless ( open( INPUT, "<$home/.autorespond/$src" ) );
    binmode INPUT;

    my $autoresponderNode = XmlNode->new('autoresponder');
    $src =~ /^([^@]+)@/;
    $autoresponderNode->setAttribute( 'mailname', $1 );

    while (<INPUT>) {
      chomp;
      last unless $_;

      if (/^From:.+<(.+)>/) {
        $autoresponderNode->setAttribute( 'from', $1 ) if ( $src ne $1 );
      }
      elsif (/^Content-type:\s+text\/([^;]+);\s+charset=(.+)/) {
        $autoresponderNode->setAttribute( 'type',    $1 );
        $autoresponderNode->setAttribute( 'charset', $2 );
      }
      elsif (/^Subject:\s*(.*)/) {
        $autoresponderNode->setAttribute( 'subject', EncodeBase64::encode($1) );
      }
    }

    $autoresponderNode->setText( EncodeBase64::encode( join( '', (<INPUT>) ) ) );
    close(INPUT);

    $mailNode->addChild($autoresponderNode);
  }
}

sub addForwarders {
  my ( $mailNode, $ptrAliases ) = @_;

  foreach my $ptrRow ( @{$ptrAliases} ) {

    my $src = $ptrRow->[0];
    my $dst = $ptrRow->[1];

    next unless ( $src =~ /^([^@]+)@[^@]+$/ );
    $src = $1;

    $dst =~ s/^\s+//;
    $dst =~ s/\s+$//;
    next if ( $dst =~ /^['"]?\|/ );    # autoresponder

    foreach my $email ( split /,/, $dst ) {
      next unless ( $email =~ /[^@]+@[^@]+/ );
      my $forwardNode = XmlNode->new('forward');
      $forwardNode->setAttribute( 'mailname', $src );
      $forwardNode->setAttribute( 'redirect', $email );
      $mailNode->addChild($forwardNode);
    }
  }
}

sub addProtectedDirs {
  my ( $root, $domain, $account, $ptrSubDomains ) = @_;

  my (
    $cmd,   $exclude, $pdir,     $find, $item,
    $title, $ptrHash, $nodePdir, $name, $password
  );

  my $home    = getHomeDir($account);
  my $rootDir = "$home/.htpasswds";
  unless ( -d $rootDir ) {
    return;
  }

  $find = AgentConfig::findBin();
  $cmd  = " cd $rootDir; $find . ";
  if ( @{$ptrSubDomains} ) {
    $exclude = join( ' -o ', map {"-path './$_' -prune"} @{$ptrSubDomains} );
    $cmd .= "\\( $exclude \\) -o";
  }
  $cmd .= " \\( -type f -name passwd -printf '\%h\\n' \\)";
  foreach $pdir (`$cmd`) {
    chomp $pdir;
    $pdir =~ s/^\.\///;
    $nodePdir = XmlNode->new('pdir');
    $nodePdir->setAttribute( 'name', $pdir );

    $title = '';
    my $htaccess_file = "$home/public_html/$pdir/.htaccess";
    if ( -T $htaccess_file ) {
      my $fileParser = Parser::makeFileParser($htaccess_file);
      $ptrHash = $fileParser->{'PARSE'}->();

      if ( ref($ptrHash) =~ /HASH/ ) {
        if ( exists( $ptrHash->{'AuthName'} ) ) {
          if ( $ptrHash->{'AuthName'} =~ /"([^"]+)"/ ) {
            $title = $1;
          }
        }
      }
    }
    $nodePdir->setAttribute( 'title', $title );

    my $fileParser = Parser::makeFileParser("$rootDir/$pdir/passwd");
    $ptrHash = $fileParser->{'PARSE'}->( 'KEYSEPARATOR' => ':' );
    if ( ref($ptrHash) =~ /HASH/ ) {
      while ( ( $name, $password ) = each( %{$ptrHash} ) ) {
        $item = XmlNode->new('pduser');
        $name =~ s/ /_/;
        $item->setAttribute( 'name', $name );
        if ( $password !~ /^[\x20-\x7f]*$/ ) {
          $item->setAttribute( 'password', EncodeBase64::encode($password) );
          $item->setAttribute( 'encoding', 'base64' );
        }
        else {
          $item->setAttribute( 'password', $password );
        }
        $nodePdir->addChild($item);
      }
    }

    $root->addChild($nodePdir);
  }
}

sub addFtpUserNodes {
  my ( $root, $domain, $account, $ptrSubDomains, $ptrFtpDirs, $ptrProftpdRows ) = @_;

  my ( $password, @users );

  my $home    = getHomeDir($account);
  my $pattern = qr/\Q$home\E\/public_html\/(.*)/;

  my $ftpquota;
  my $ftpquota_file = "$home/etc/ftpquota";
  if ( -e $ftpquota_file ) {
    Logging::info("Parse ftp quota '$ftpquota_file'");
    my $ftpquotaParser = Parser::makeFileParser($ftpquota_file);
    $ftpquota = $ftpquotaParser->{'PARSE'}->( 'KEYSEPARATOR' => ':' );
    $ftpquota = undef if not ref($ftpquota) =~ /HASH/;
  }

  foreach my $ptrRow ( @{$ptrProftpdRows} ) {

    if ( $ptrRow->[4] eq $account ) {
      if ( $ptrRow->[5] =~ /$pattern/ ) {
        my $subdir = $1;
        my $pass = ( $ptrRow->[1] ne '*' ) ? $ptrRow->[1] : '';

        if ( grep( ( $_ eq $subdir ), @{$ptrFtpDirs} ) ) {
          push @users, [ $ptrRow->[0], $pass, $subdir ];
        }
      }
    }

    if ( !$password && ( $ptrRow->[0] eq $account ) ) {
      $password = $ptrRow->[1];
    }
  }

  if ( '*' eq $password ) {
    $password = getSystemPassword($account);
    if ( '*' eq $password ) {
      $password = '';
    }
  }

  # makes main domains's ftp user node
  $root->addChild( makeFtpUserNode( $account, $password, undef ) );

  # other ftp users of domain
  foreach my $ptrRow (@users) {

    my ( $acc, $pass, $path ) = @{$ptrRow};
    my $quota = '';
    $quota = $ftpquota->{$acc} if $ftpquota and exists $ftpquota->{$acc};
    my $ftpUserNode = makeFtpUserNode( $acc, $pass, $quota );

    if ($path) {
      $ftpUserNode->setAttribute( 'directory', $path );
    }

    $root->addChild($ftpUserNode);
  }
}

sub makeFtpUserNode {
  my ( $account, $password, $quota ) = @_;
  my $item = XmlNode->new('ftpuser');
  $item->setAttribute( 'name', $account );

  # User may be locked
  if ( $password && $password !~ /^\!/ ) {
    $item->setAttribute( 'password', $password );
  }
  $item->setAttribute( 'quota', $quota ) if $quota;

  return $item;
}

sub makeAnonFtpdNode {
  my $account = shift;
  my ( $node, $publicFtp, $pureFtpd, $dumpFile );
  my $home = getHomeDir($account);

  $publicFtp = "$home/public_ftp";
  unless ( -d $publicFtp ) {
    return undef;
  }
  $node = XmlNode->new('anonftp');

  #
  # prepare anonftps
  #
  $pureFtpd = '/etc/pure-ftpd';
  if ( -d $pureFtpd ) {
    unless ( keys %anonftps ) {
      my ( $link, $linkTo, $fullPath );
      if ( opendir( FTPD, "$pureFtpd" ) ) {
        foreach $link ( readdir(FTPD) ) {
          $fullPath = "$pureFtpd/$link";
          next unless ( -l $fullPath );
          $linkTo = readlink($fullPath);
          if ( $linkTo =~ /\Qhome\E\/([^\/]+)\/public_ftp/ ) {
            $anonftps{$1} = $linkTo;
          }
        }
        closedir(FTPD);
      }
    }
  }
  #
  # end prepare anonftps
  #

  #
  # check permissions
  #
  # check for read permission
  if ( exists( $anonftps{$account} ) && ( ( stat($publicFtp) )[2] & 05 ) )
  {
    $node->setAttribute( 'pub', 'true' );
    # check for write permission
    if ( ( stat("$publicFtp/incoming") )[2] & 03 ) {
      $node->setAttribute( 'incoming', 'true' );
    }
  }
  #
  # end check permissions
  #

  return $node;
}

sub makePermissionNode {
  my ($name) = @_;
  my $item = XmlNode->new('permission');
  $item->setAttribute( 'name', $name );
  return $item;
}

sub makeLimitNode {
  my ( $name, $value ) = @_;
  unless ( defined($value) && ( $value !~ '/^\s*$/' ) ) {
    $value = '-1';
  }
  my $item = XmlNode->new( 'limit' );
  $item->setAttribute( 'name', $name );
  $item->setText( $value );
  return $item;
}

sub makePasswordNode {
  my ($password, $type) = @_;

  unless ( defined ( $password ) ) {
    Logging::warning("'undef' password passed to makePasswordNode. Set to empty");
    $password = '';
  }

  if ( $password eq '' ) {
	  $type = 'plain';
  } 
  else {
	  if ( $type ne 'plain' ) {
	    $type = 'encrypted';
	  }
  }
  my $passwordNode = XmlNode->new( 'password' );
  $passwordNode->setAttribute( 'type', $type ); 
  $passwordNode->setText( $password );
  return $passwordNode;
}

sub getAgentStatus {
  my $root = XmlNode->new('agent-status');

  unless ( getCpanelVersion() && defined AgentConfig::iconvBin() ) {
    my $item;
    if ( defined AgentConfig::iconvBin() ) {
      $item = XmlNode->new( 'wrong-platform' );
      $item->setText( '' );
    }
    else {
      $item = XmlNode->new( 'wrong-platform' );
      $item->setText( 'no iconv found on the source host' );
    }

    $root->addChild($item);
  }

  return $root;
}

sub getCpanelVersion {
  my ( $engineRoot, $util );

  loadMainConfig();

  if ( $engineRoot = $cPanelConfig{'engineroot'} ) {
    unless ( -d $engineRoot ) {
      return;
    }
    $util = "$engineRoot/cpanel";
    unless ( -x $util ) {
      return;
    }
    my @outs = `$util`;
    foreach my $line (@outs) {
      chomp $line;
      if ( $line =~ /cPanel\s+\[(\d+)\.(\d+).\d+/ ) {
        if ( $1 ne '9' && $1 ne '10' && $1 ne '11' ) {
          return;
        }
        return "$1.$2";
      }
    }
  }
  elsif ( -f "$installDir/cpanel.config" ) {
    return '9';
  }
}

sub getMainIP {
  my $mainip_file = "$installDir/mainip";
  my $ip;
  if ( open( INPUT, "<$mainip_file" ) ) {
    binmode INPUT;
    my $pattern = qr/(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})/;
    while (<INPUT>) {
      if (/$pattern/) {
        $ip = $1;
        last;
      }
    }
    close(INPUT);
    if ($ip) {
      return $ip;
    }
  }
}


1;
