package Transformer;

use Logging;
use XmlNode;
use Dumper;
use Validator;
use Error qw(:try);
use Exception::XmlNode;

#
# Begin global variables
#

# Hash of client names and client XmlNode elements, Plesk format
my %clientNodes;

# Initialized in initQuickInfo:
# Hash of client names and shallow client XmlNode elements, Plesk format
my %shallowClientNodes;

# Hash of Plesk client and cPanel accounts
my %clients2cpAccount;

# Hash of domains and its owners (clients)
my %domain2client;

# Hash of domains and its hosting type
my %domain2hosting;

my $initialized;
#
# End of global variables
#

#
# Begin constants:
#

# Maximum significant value for the disk space / traffic limit.
#	Values exceeding that would be counted as 'unlimited'
my $max_limit_value = 999999*1024*1024;

my %limit_map = ( 'max_traffic' => 'max_traffic',
                  'disk_space'  => 'disk_space',
                  'max_subdom'  => 'max_subdom',
                  'max_box'     => 'max_box',
                  'max_ftpuser' => 'max_wu');
#
# End of constants
#


#
# Begin service subs
#
sub initialize {
  # Get all accounts from Dumper
  # Get account' basic information, only need for getting the domain list
  # Should initialize %resellerNodes, %clientNodes and %domainNodes with 'undef'
  #
  unless ( $initialized ) {
    Dumper::initialize();

    # Get accounts
    my @cp_accounts = Dumper::getAllAccounts();
    foreach my $account ( sort @cp_accounts ) {
      my $cpAccountNode = Dumper::makeAccountNode( $account, undef, undef, 'shallow_mode');
      if ( ref($cpAccountNode) =~ /XmlNode/ ) {
        # Transformation now is realized in 'one-to-one' mode. This does not preserve cPanel accounts hierarhy.
        my $accountNode = transformAccountNode($cpAccountNode);
        unless ( defined $accountNode ) {
          Logging::error("Account '$accountNode' transformation failed");
        }
      }
    }
    $initialized = 1;
  }
}

sub getResellers {
  my @resellers;
  # empty
  return @resellers;
}

sub getClients {
  my $owner = shift; # reseller that owns the clients returned. Could be 'undef' for default reseller ('root' or 'admin')

  my @clients;
  # empty
  return @clients;
}

sub getDomains {
  my $owner = shift; # reseller or client that owns the domains returned. Could be 'undef' for default reseller or client ('root' or 'admin')

  # Should return an array of clients identifiers, that could be a number, name or guid. Should be unique through migration dump.

  # The current implementation of cPanel migration deals with domains only.

  initialize();
  my @domains;

  unless ( defined $owner ) {
    foreach my $domain (sort keys %domain2client) {
      push @domains, $domain;
    }
  }

  return @domains;
}

#
# Basic transform routine
#
# Works in 'shallow mode' by default
# Important: dump in 'shallow_mode' is not required to be valid for plesk.xsd.
#
sub transformAccountNode {
  my ($cpAccountNode, $not_shallow_mode) = @_;

  if ( ref($cpAccountNode) =~ /XmlNode/ ) {
    if ($cpAccountNode->getName() eq 'account' ) {

      my $clientNode = XmlNode->new ('client');
      my $cp_account_name = $cpAccountNode->getAttribute( 'name' );
      my $client_name = $cp_account_name;

      $clientNode->setAttribute( 'name', $client_name );
      $clientNode->setAttribute( 'guid', '' );

      if ( $not_shallow_mode ) {
        $clientNode->setAttribute( 'contact', 'client ' . $client_name );
        my $crDate = $cpAccountNode->getAttribute( 'date' );
        $clientNode->setAttribute( 'cr-date', $crDate ) if ( defined $crDate );
        $clientNode->addChild( XmlNode->new( 'preferences' ));
        my $clientPropertiesNode = XmlNode->new( 'properties' );
        my $clientPasswordNode = &getClientPassword($cpAccountNode);
        $clientPropertiesNode->addChild( $clientPasswordNode ) if ( defined $clientPasswordNode );
        $clientPropertiesNode->addChild( &makeStatusEnabled() );
        $clientNode->addChild( $clientPropertiesNode );

        addClientLimitsAndPermissions( $clientNode, $cpAccountNode );
      }
      my $clientIpPoolNode = XmlNode->new( 'ip_pool' );
      $clientIpPoolNode->addChild( &getIp($cpAccountNode) );
      $clientNode->addChild( $clientIpPoolNode );

      my @domains = ();

      my $cpDomainNode = $cpAccountNode->getChild( 'domain' );
      if ( defined ( $cpDomainNode ) ) {

        # 1) Transform cPanel account's domain to Plesk domain
        my $domainNode = XmlNode->new ('domain');

        my $domain_name = $cpDomainNode->getAttribute( 'name' );
        $domainNode->setAttribute( 'name', $domain_name );

        # Plesk dump format requires guid, could be empty however, will be fixed with guid fixer before restore
        $domainNode->setAttribute( 'guid', '' );

        my $crDate = $cpAccountNode->getAttribute( 'date' );
        $domainNode->setAttribute( 'cr-date', $crDate ) if ( defined $crDate );

        if ( $not_shallow_mode ) {
          my $domainPreferencesNode = &getDomainPreferences($cpDomainNode);
          $domainNode->addChild( $domainPreferencesNode );
        }

        my $domainPropertiesNode = &getDomainProperties($cpAccountNode);
        $domainNode->addChild( $domainPropertiesNode );

        if ( $not_shallow_mode ) {
          addDomainLimitsAndPermissions( $domainNode, $cpAccountNode );

          addDomainMailsystem( $domainNode, $cpAccountNode );

          addDomainDatabases( $domainNode, $cpDomainNode );

          addDomainMaillists( $domainNode, $cpDomainNode );

          addDomainuser( $domainNode, $cpAccountNode);

          my $domainPhostingNode = &getPhosting( $domainNode, $cpAccountNode);
          $domainNode->addChild( $domainPhostingNode );
        }

        unless( $not_shallow_mode ) {
          $domain2hosting{$domain_name} = 'phosting';
        }

        # Store 'domain' node. Then we'll create 'domains' node if any 'domain' node is created
        push @domains, $domainNode;

        # 2) Transform cPanel account's subdomains into Plesk domains
        my @cpSubdomainNodes = $cpDomainNode->getChildren( 'subdomain' );
        if ( @cpSubdomainNodes ) {
          foreach my $cpSubdomainNode (sort @cpSubdomainNodes) {
            my $has_ftp_account;
            $has_ftp_account = defined ( $cpSubdomainNode->getChild( 'ftpuser' ) );

            my @cpAddonDomainNodes = $cpSubdomainNode->getChildren( 'addondomain' );

            my $addon_domain_name;
            $addon_domain_name = $cpAddonDomainNodes[0]->getText() if ( @cpAddonDomainNodes );

            if ( $addon_domain_name ) {
              my $domainFromSubdomainNode = XmlNode->new ('domain');
              $domainFromSubdomainNode->setAttribute( 'name', $addon_domain_name );
              # Plesk dump format requires guid, could be empty however, will be fixed with guid fixer before restore
              $domainFromSubdomainNode->setAttribute( 'guid', '' );

              # Subdomains which have addondomains and        have ftp account are mapped into Plesk domains with phosting
              # Subdomains which have addondomains but do not have ftp account are mapped into Plesk domains with shosting
              my $hosting_type = $has_ftp_account? 'phosting' : 'shosting' ;

              if ( $not_shallow_mode ) {
                if ( $hosting_type eq 'phosting' ) {
                  # Transform subdomain to domain with hosting
                  &processSubdomain2Phosting($domainFromSubdomainNode, $cpSubdomainNode, $cpDomainNode, $cpAccountNode);
                }
                else {
                  # Transform subdomain to domain with shosting
                  &processSubdomain2Shosting($domainFromSubdomainNode, $cpSubdomainNode, $cpDomainNode, $cpAccountNode);
                }
              }

              unless ( $not_shallow_mode ) {
                $domain2hosting{$addon_domain_name} = $hosting_type;
              }

              # Store 'domain' node. Then we'll create 'domains' node if any 'domain' node is created
              push @domains, $domainFromSubdomainNode;
            }
          }
        }

        if ( @domains ) {
          my $domainsNode = XmlNode->new( 'domains' );
          foreach $domainNode ( @domains ) {
            $domainsNode->addChild( $domainNode );
            $domain2client{$domainNode->getAttribute( 'name' )} = $client_name;
          }
          $clientNode->addChild( $domainsNode );
        }
      }

      if ( $not_shallow_mode ) {
        $clientNodes{$client_name} = $clientNode;
      }
      else {
        $clients2cpAccount{$client_name} = $cp_account_name;
        $shallowClientNodes{$client_name} = $clientNode;
      }

      return $clientNode;

    }
  }
  return undef;
}


sub getClientNode4Domain {
  my $id = shift; # domain identifier from 'getDomains' result

  initialize();

  if ( exists ( $domain2client{$id} ) ){
    my $client_name = $domain2client{$id};

    return $clientNodes{$client_name} if exists ( $clientNodes{$client_name} );

    my $cpAccountNode = Dumper::makeAccountNode( $client_name, undef, undef);
    if ( Logging::getVerbosity() > 3 ) {
      my $cpAccountNodeDump = $cpAccountNode->serialize();
      Logging::debug("-" x 16 . "  begin cPanel account dump " . "-" x 16);
      Logging::debug($cpAccountNodeDump);
      Logging::debug("-" x 16 . " end of cPanel account dump " . "-" x 16);
    }

    if ( ref($cpAccountNode) =~ /XmlNode/ ) {
      my $error = undef;
      my $clientNode = undef;
      try {
        Validator::validateCpAccountNode($cpAccountNode);
      }
      catch Exception::XmlNode with {
	      my $ex = shift;
	      Logging::error($ex->print());
	      Logging::error("Domain '" . $id . "'will not be migrated due to errors above");
	      $error = 1;
      };

      unless ( $error ) {
        $clientNode = transformAccountNode($cpAccountNode, 'not_shallow_mode');
        if ( Logging::getVerbosity() > 3 ) {
          my $clientNodeDump = $clientNode->serialize();
          Logging::debug("-" x 17 . " begin Plesk client dump  " . "-" x 17);
          Logging::debug($clientNodeDump);
          Logging::debug("-" x 17 . " end of Plesk client dump " . "-" x 17);
        }
      }

      if ( defined ( $clientNode ) ) {
        $clientNodes{$client_name} = $clientNode;
        return $clientNode;
      }
    }
    return undef;
  }
}
#
# End of service subs
#


#
# Begin node transformation subs
#
sub getClientPassword {
  my $cpAccountNode = shift;

  unless ( testXmlNodeParam($cpAccountNode, 'account', 'getClientPassword') ) {return undef;}

  # do nothing
  return undef;
}

sub addClientLimitsAndPermissions {
  my ($clientNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($clientNode,    'client',  'addClientLimitsAndPermissions') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode, 'account', 'addClientLimitsAndPermissions') ) {return undef;}

  # do nothing
}

sub getIp {
  my $cpAccountNode = shift;

  unless ( testXmlNodeParam($cpAccountNode, 'account', 'getIp') ) {return undef;}

  my $cpIpNode = $cpAccountNode->getChild( 'ip' );
  my $ipNode;
  if ( defined $cpIpNode) {
    $ipNode = XmlNode->new('ip');

    my $cpDomainNode = $cpAccountNode->getChild( 'domain' );
    my $allow_anon_ftp = 'false';
    if (defined $cpDomainNode) {
      $allow_anon_ftp = $cpDomainNode->getChildAttribute( 'anonftp', 'pub' );
    }

    # Plesk requires exclusive IP on domain to allow anon tfp access
    my $ipTypeNode = XmlNode->new( 'ip-type' );
    $ipTypeNode->setText( $allow_anon_ftp eq 'true'? 'exclusive' : 'shared' );
    $ipNode->addChild( $ipTypeNode );

    my $ipAddressNode = XmlNode->new( 'ip-address' );
    $ipAddressNode->setText( $cpIpNode->getText() );
    $ipNode->addChild( $ipAddressNode );
  }
  return $ipNode;
}

sub getDomainPreferences {
  my $cpDomainNode = shift;

  unless ( testXmlNodeParam($cpDomainNode, 'domain', 'getDomainPreferences') ) {return undef;}

  my $domainPreferencesNode = XmlNode->new( 'preferences' );

  # Since PPP9.x cPanel account's addondomains are transformed into Plesk domain aliases instead of domain with shosting
  my @cpAddondomainNodes = $cpDomainNode->getChildren( 'addondomain' );
  if (@cpAddondomainNodes) {
    foreach my $cpAddondomainNode (sort @cpAddondomainNodes) {
      my $domainAliasNode = XmlNode->new( 'domain-alias' );
      $domainAliasNode->setAttribute( 'name', $cpAddondomainNode->getText() );
      $domainAliasNode->setAttribute( 'mail', 'true' );
      $domainAliasNode->setAttribute( 'web', 'true' );
      $domainAliasNode->addChild( &makeStatusEnabled() );
      $domainPreferencesNode->addChild( $domainAliasNode );
    }
  }
  return $domainPreferencesNode;
}

sub getDomainProperties {
  my $cpAccountNode = shift;

  unless ( testXmlNodeParam($cpAccountNode, 'account', 'getDomainProperties') ) {return undef;}

  my $domainPropertiesNode = XmlNode->new( 'properties' );

  $domainPropertiesNode->addChild( &getIp($cpAccountNode) );
  $domainPropertiesNode->addChild( &makeStatusEnabled() );

  return $domainPropertiesNode;
}

sub addDomainLimitsAndPermissions {
  my ($domainNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($domainNode,    'domain',  'addDomainLimitsAndPermissions') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode, 'account', 'addDomainLimitsAndPermissions') ) {return undef;}

  my @limitNodes;
  my @cpLimitNodes = $cpAccountNode->getChildren( 'limit' );

  if ( @cpLimitNodes ) {
    foreach my $cpLimitNode ( @cpLimitNodes ) {
      my $limitName = $cpLimitNode->getAttribute( 'name' );
      if ( exists $limit_map{$limitName} ) {
        my $limitValue = $cpLimitNode->getText();
        $limitValue = -1 if ( int($limitValue) > $max_limit_value );
        my $limitNode = XmlNode->new( 'limit' );
        $limitNode->setAttribute( 'name', $limit_map{$limitName} );
        $limitNode->setText( $limitValue );
        push @limitNodes, $limitNode;
      }
    }

    if ( @limitNodes ) {
      my $limitsAndPermissionsNode = XmlNode->new( 'limits-and-permissions' );
      foreach my $limitNode ( @limitNodes ) {
        $limitsAndPermissionsNode->addChild( $limitNode );
      }
      $domainNode->addChild( $limitsAndPermissionsNode );
    }
  }
}

sub addDomainMailsystem {
  my ($domainNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($domainNode,    'domain',  'addDomainMailsystem') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode, 'account', 'addDomainMailsystem') ) {return undef;}

  my $cpDomainNode = $cpAccountNode->getChild( 'domain' );

  my $cpMailNode = $cpDomainNode->getChild( 'mail' );

  return undef unless ( defined ( $cpMailNode ) );

  my $mailsystemNode = XmlNode->new( 'mailsystem' );

  $mailsystemNode->addChild( &getMailsystemProperties() );

  &addMailsystemMailusers( $mailsystemNode, $cpMailNode, $cpAccountNode );

  &addMailsystemPreferences( $mailsystemNode, $cpMailNode );

  $domainNode->addChild( $mailsystemNode );
}

sub addSubdomainMailsystem {
  my ($domainNode, $cpSubdomainNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($domainNode,      'domain',    'addSubdomainMailsystem') ) {return undef;}
  unless ( testXmlNodeParam($cpSubdomainNode, 'subdomain', 'addSubdomainMailsystem') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode,   'account',   'addSubdomainMailsystem') ) {return undef;}

  my $cpMailNode = $cpSubdomainNode->getChild( 'mail' );

  return undef unless ( defined ( $cpMailNode ) );

  my $mailsystemNode = XmlNode->new( 'mailsystem' );

  $mailsystemNode->addChild( &getMailsystemProperties() );

  &addMailsystemMailusers( $mailsystemNode, $cpMailNode, $cpAccountNode );

  &addMailsystemPreferences( $mailsystemNode, $cpMailNode );

  $domainNode->addChild( $mailsystemNode );
}

sub getMailsystemProperties {
  my $propertiesNode = XmlNode->new( 'properties' );
  $propertiesNode->addChild( &makeStatusEnabled() );
  return $propertiesNode;
}

sub addMailsystemMailusers {
  my ($mailsystemNode, $cpMailNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($mailsystemNode, 'mailsystem', 'addMailsystemMailusers') ) {return undef;}
  unless ( testXmlNodeParam($cpMailNode,     'mail',       'addMailsystemMailusers') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode,  'account',    'addMailsystemMailusers') ) {return undef;}

  my $mailusersNode = XmlNode->new( 'mailusers' );

  my $cpAccountName = $cpAccountNode->getAttribute( 'name' );
  my $cpDomainName = $cpAccountNode->getChildAttribute( 'domain', 'name' );

  my @cpMailnameNodes = $cpMailNode->getChildren( 'mailname' );
  if ( @cpMailnameNodes ) {
    foreach my $cpMailnameNode ( @cpMailnameNodes ) {
      my $mailuserNode = XmlNode->new( 'mailuser' );

      my $cpMailnameName = $cpMailnameNode->getAttribute( 'name' );
      my %mailuserMetadata;
      $mailuserMetadata{'mailname'} = $cpMailnameName;
      $mailuserMetadata{'domain'} = $cpDomainName;
      $mailuserMetadata{'account'} = $cpAccountName;
      $mailuserNode->setMetadata(\%mailuserMetadata);

      $mailuserNode->setAttribute( 'name', $cpMailnameName );
      $mailuserNode->setAttribute( 'mailgroup-enabled', 'false' );
      # Plesk dump format requires guid, could be empty however, will be fixed with guid fixer before restore
      $mailuserNode->setAttribute( 'guid', '' );
      my $mailboxQuota = $cpMailnameNode->getAttribute( 'quota' );
      $mailuserNode->setAttribute( 'mailbox-quota', $mailboxQuota ) if ( defined ( $mailboxQuota ) );
      
      my $mailuserPropertiesNode = XmlNode->new( 'properties' );
      my $mailuserPasswordNode = XmlNode->new( 'password' );
      $mailuserPasswordNode->setAttribute( 'type', 'encrypted' );
      $mailuserPasswordNode->setText( $cpMailnameNode->getAttribute( 'password' ) );
      $mailuserPropertiesNode->addChild( $mailuserPasswordNode );
      $mailuserNode->addChild( $mailuserPropertiesNode );

      my $mailuserLimitsAndPermissionsNode = XmlNode->new( 'limits-and-permissions' );
      &addMailuserPermission( $mailuserLimitsAndPermissionsNode, 'multiple-sessions',  'false');
      &addMailuserPermission( $mailuserLimitsAndPermissionsNode, 'cp-access',          'false');
      &addMailuserPermission( $mailuserLimitsAndPermissionsNode, 'manage-spamfilter',  'false');
      &addMailuserPermission( $mailuserLimitsAndPermissionsNode, 'manage-virusfilter', 'false');

      $mailuserNode->addChild( $mailuserLimitsAndPermissionsNode );

      $mailuserNode->addChild( &getMailuserPreferences($cpMailnameNode, $cpMailNode, $cpAccountNode) );

      $mailusersNode->addChild( $mailuserNode );
    }
  }

  &addMailusersFromAutoresponders($mailusersNode, $cpMailNode, $cpAccountNode);

  &addMailusersFromForwards($mailusersNode, $cpMailNode, $cpAccountNode);

  $mailsystemNode->addChild( $mailusersNode ) if ( $mailusersNode->getChildren( 'mailuser' ) );
}

sub getMailuserPreferences {
  my ($cpMailnameNode, $cpMailNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($cpMailnameNode, 'mailname', 'getMailuserPreferences') ) {return undef;}
  unless ( testXmlNodeParam($cpMailNode,     'mail',     'getMailuserPreferences') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode,  'account',  'getMailuserPreferences') ) {return undef;}

  my $mailname = $cpMailnameNode->getAttribute( 'name' );

  my $mailuserPreferences = XmlNode->new( 'preferences' );

  my $mailboxNode = XmlNode->new( 'mailbox' );
  $mailboxNode->setAttribute( 'enabled' , 'true' );
  $mailboxNode->setAttribute( 'type' ,    'mdir' );

  $mailuserPreferences->addChild( $mailboxNode );

  my @cpForwardNodes = $cpMailNode->getChildren( 'forward' );
  foreach my $cpForwardNode ( @cpForwardNodes ) {
    next unless ( $cpForwardNode->getAttribute( 'mailname' ) eq $mailname );

    $mailuserPreferences->addChild( &transformForwardNode($cpForwardNode) );
    last;
  }

  my @cpAutoresponderNodes = $cpMailNode->getChildren( 'autoresponder' );
  foreach my $cpAutoresponderNode ( @cpAutoresponderNodes ) {
    next unless ( $cpAutoresponderNode->getAttribute( 'mailname' ) eq $mailname );

    my $autoresponderNode = &transformAutoresponderNode($cpAutoresponderNode);

    my $autorespondersNode = XmlNode->new( 'autoresponders' );
    $autorespondersNode->setAttribute( 'enabled' , 'true' );

    $autorespondersNode->addChild( $autoresponderNode );

    $mailuserPreferences->addChild( $autorespondersNode );
    last;
  }

  my $cpSpamassassinNode = $cpMailnameNode->getChild( 'spamassassin' );
  $cpSpamassassinNode = $cpAccountNode->getChild( 'spamassassin' ) unless ( defined ( $cpSpamassassinNode ) );
  my $spamassassinNode = undef;

  $mailuserPreferences->addChild( &transformSpamassassinNode( $cpSpamassassinNode) ) if ( defined ($cpSpamassassinNode ) );

  return $mailuserPreferences;
}

sub addMailuserPermission {
  my ($mailuserLimitsAndPermissionsNode, $permissionName, $permissionValue) = @_;

  unless ( testXmlNodeParam($mailuserLimitsAndPermissionsNode, 'limits-and-permissions', 'addMailuserPermission') ) {return undef;}

  my $mailuserPermissionNode = XmlNode->new( 'mailuser-permission' );
  $mailuserPermissionNode->setAttribute( 'name', $permissionName);
  $mailuserPermissionNode->setAttribute( 'value', $permissionValue) if defined ( $permissionValue );

  $mailuserLimitsAndPermissionsNode->addChild( $mailuserPermissionNode );
}

sub addMailsystemPreferences {
  my ($mailsystemNode, $cpMailNode) = @_;

  unless ( testXmlNodeParam($mailsystemNode, 'mailsystem', 'addMailsystemPreferences') ) {return undef;}
  unless ( testXmlNodeParam($cpMailNode,     'mail',       'addMailsystemPreferences') ) {return undef;}

  my $cpMailDefaultNode = $cpMailNode->getChild( 'default' );
  if ( defined ( $cpMailDefaultNode ) ) {

    my $cpTargetAttr = $cpMailDefaultNode->getAttribute( 'target' );
    my $target = '';
    if ( $cpTargetAttr eq 'fail' ) {
      $target .= 'bounce:';
    }
    if ($cpTargetAttr eq 'ignore') {
      $target .= 'reject';
    }
    else {
      $target .= $cpMailDefaultNode->getText();
    }

    my $catchAllNode = XmlNode->new( 'catch-all' );
    $catchAllNode->setText( $target );
    my $preferencesNode = XmlNode->new( 'preferences' );
    $preferencesNode->addChild( $catchAllNode );
    $mailsystemNode->addChild( $preferencesNode );
  }
}

sub transformSpamassassinNode {
  my $cpSpamassassinNode = shift;

  unless ( testXmlNodeParam($cpSpamassassinNode, 'spamassassin', 'transformSpamassassinNode') ) {return undef;}

  # cPanel spamassassin element format is compatible with Plesk's one
  my $spamassassinNode = $cpSpamassassinNode->copy();

  return $spamassassinNode;
}

sub transformForwardNode {
  my $cpForwardNode = shift;

  unless ( testXmlNodeParam($cpForwardNode, 'forward', 'transformForwardNode') ) {return undef;}

  my $redirectNode = XmlNode->new( 'redirect' );
  $redirectNode->setAttribute( 'enabled' , 'true' );
  $redirectNode->setText( $cpForwardNode->getAttribute( 'redirect' ) );
	return $redirectNode;
}

sub transformAutoresponderNode {
  my $cpAutoresponderNode = shift;

  unless ( testXmlNodeParam($cpAutoresponderNode, 'autoresponder', 'transformAutoresponderNode') ) {return undef;}

  my $autoresponderNode = XmlNode->new( 'autoresponder' );
  $autoresponderNode->setAttribute( 'status' , 'on' );
  my $from = $cpAutoresponderNode->getAttribute( 'from' );
  $autoresponderNode->setAttribute( 'replyto', $from ) if defined $from;

  my $subject = $cpAutoresponderNode->getAttribute( 'subject' );
  $autoresponderNode->setAttribute( 'subject', $subject ) if defined $subject;

  my $type = $cpAutoresponderNode->getAttribute( 'type' );
  if ( $type eq 'html' ) {
    $autoresponderNode->setAttribute( 'content-type', 'text/html' );
  }

  my $autoresponderTextNode = XmlNode->new( 'text' );
  my $charset = $cpAutoresponderNode->getAttribute( 'charset' );
  $autoresponderTextNode->setAttribute( 'charset', $charset ) if defined $charset;
  $autoresponderTextNode->setText( $cpAutoresponderNode->getText() );

  $autoresponderNode->addChild( $autoresponderTextNode );

  return $autoresponderNode;
}

sub addMailusersFromAutoresponders {
  my ($mailusersNode, $cpMailNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($mailusersNode, 'mailusers', 'addMailusersFromAutoresponders') ) {return undef;}
  unless ( testXmlNodeParam($cpMailNode,    'mail',      'addMailusersFromAutoresponders') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode, 'account',   'addMailusersFromAutoresponders') ) {return undef;}

  my $cpAccountName = $cpAccountNode->getAttribute( 'name' );
  my $cpDomainName = $cpAccountNode->getChildAttribute( 'domain', 'name' );

  my %mailnameNames = ();
  my @cpMailnameNodes = $cpMailNode->getChildren( 'mailname' );
  foreach my $cpMailnameNode ( @cpMailnameNodes ) {
    $mailnameNames{$cpMailnameNode->getAttribute( 'name' )} = 1;
  }

  my @cpAutoresponderNodes = $cpMailNode->getChildren( 'autoresponder' );
  foreach my $cpAutoresponderNode ( @cpAutoresponderNodes ) {
    next if ( exists ($mailnameNames{$cpAutoresponderNode->getAttribute('mailname')}) );

    my $mailuserNode = XmlNode->new( 'mailuser' );

    my %mailuserMetadata;
    $mailuserMetadata{'mailname'} = $cpAutoresponderNode->getAttribute('mailname');
    $mailuserMetadata{'domain'} = $cpDomainName;
    $mailuserMetadata{'account'} = $cpAccountName;
    $mailuserNode->setMetadata(\%mailuserMetadata);

    $mailuserNode->setAttribute( 'name', $cpAutoresponderNode->getAttribute( 'mailname' ) );
    $mailuserNode->setAttribute( 'mailgroup-enabled', 'false' );
    # Plesk dump format requires guid, could be empty however, will be fixed with guid fixer before restore
    $mailuserNode->setAttribute( 'guid', '' );

    $mailuserNode->addChild( XmlNode->new( 'properties' ) );

    my $mailuserLimitsAndPermissionsNode = XmlNode->new( 'limits-and-permissions' );
    &addMailuserPermission( $mailuserLimitsAndPermissionsNode, 'multiple-sessions',  'false');
    &addMailuserPermission( $mailuserLimitsAndPermissionsNode, 'cp-access',          'false');
    &addMailuserPermission( $mailuserLimitsAndPermissionsNode, 'manage-spamfilter',  'false');
    &addMailuserPermission( $mailuserLimitsAndPermissionsNode, 'manage-virusfilter', 'false');

    $mailuserNode->addChild( $mailuserLimitsAndPermissionsNode );

    my $preferencesNode = XmlNode->new( 'preferences' );
    my $autorespondersNode = XmlNode->new( 'autoresponders' );
    $autorespondersNode->addChild( &transformAutoresponderNode( $cpAutoresponderNode ) );
    $preferencesNode->addChild( $autorespondersNode );

    $mailuserNode->addChild( $preferencesNode );

    $mailusersNode->addChild( $mailuserNode );
  }
}

sub addMailusersFromForwards {
  my ($mailusersNode, $cpMailNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($mailusersNode, 'mailusers', 'addMailusersFromForwards') ) {return undef;}
  unless ( testXmlNodeParam($cpMailNode,    'mail',      'addMailusersFromForwards') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode, 'account',   'addMailusersFromForwards') ) {return undef;}

  my $cpAccountName = $cpAccountNode->getAttribute( 'name' );
  my $cpDomainName = $cpAccountNode->getChildAttribute( 'domain', 'name' );

  my %mailnameNames = ();
  my @cpMailnameNodes = $cpMailNode->getChildren( 'mailname' );
  foreach my $cpMailnameNode ( @cpMailnameNodes ) {
    $mailnameNames{$cpMailnameNode->getAttribute( 'name' )} = 1;
  }

  my %forwardsPerName;
  my @cpForwardNodes = $cpMailNode->getChildren( 'forward' );
  foreach my $cpForwardNode ( @cpForwardNodes ) {
    my $mailname = $cpForwardNode->getAttribute('mailname');
    $forwardsPerName{$mailname} = 0 unless ( exists ( $forwardsPerName{$mailname} ) );
    $forwardsPerName{$mailname} += 1;
  }

  foreach my $cpForwardNode ( @cpForwardNodes ) {
    my $mailname = $cpForwardNode->getAttribute('mailname');
    next if ( exists ( $mailnameNames{$mailname} ) );
    next unless ( exists ( $forwardsPerName{$mailname} ) );

    my $mailuserNode = XmlNode->new( 'mailuser' );

    my %mailuserMetadata;
    $mailuserMetadata{'mailname'} = $mailname;
    $mailuserMetadata{'domain'} = $cpDomainName;
    $mailuserMetadata{'account'} = $cpAccountName;
    $mailuserNode->setMetadata(\%mailuserMetadata);

    $mailuserNode->setAttribute( 'name', $mailname );
    # Plesk dump format requires guid, could be empty however, will be fixed with guid fixer before restore
    $mailuserNode->setAttribute( 'guid', '' );

    $mailuserNode->addChild( XmlNode->new( 'properties' ) );

    my $mailuserLimitsAndPermissionsNode = XmlNode->new( 'limits-and-permissions' );
    &addMailuserPermission( $mailuserLimitsAndPermissionsNode, 'multiple-sessions',  'false');
    &addMailuserPermission( $mailuserLimitsAndPermissionsNode, 'cp-access',          'false');
    &addMailuserPermission( $mailuserLimitsAndPermissionsNode, 'manage-spamfilter',  'false');
    &addMailuserPermission( $mailuserLimitsAndPermissionsNode, 'manage-virusfilter', 'false');
    $mailuserNode->addChild( $mailuserLimitsAndPermissionsNode );

    my $preferencesNode = XmlNode->new( 'preferences' );

    if ( $forwardsPerName{$mailname} == 1  ) {
      $mailuserNode->setAttribute( 'mailgroup-enabled', 'false' );
      $preferencesNode->addChild( &transformForwardNode( $cpForwardNode ) );

    }
    else {
      $mailuserNode->setAttribute( 'mailgroup-enabled', 'true' );

      foreach my $cpForwardNode2 ( @cpForwardNodes ) {
        next unless ( $cpForwardNode2->getAttribute( 'mailname' ) eq $mailname );

        my $mailgroupMemberNode = XmlNode->new( 'mailgroup-member' );
        $mailgroupMemberNode->setText( $cpForwardNode2->getAttribute( 'redirect' ) );
        $preferencesNode->addChild( $mailgroupMemberNode );
      }
    }
    $mailuserNode->addChild( $preferencesNode );

    $mailusersNode->addChild( $mailuserNode );
    
    # All forwards with @mailname='$mailname' are processed. Should remove it from processing.
    delete $forwardsPerName{$mailname};
  }
}

sub addDomainMaillists {
  my ($domainNode, $cpDomainNode) = @_;

  unless ( testXmlNodeParam($domainNode,   'domain', 'addDomainMaillists') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode, 'domain', 'addDomainMaillists') ) {return undef;}

  my $cpMaillistsNode = $cpDomainNode->getChild( 'maillists' );
  if ( defined ( $cpMaillistsNode ) ) {
    my @cpMaillistNodes = $cpMaillistsNode->getChildren( 'maillist' );
    if ( @cpMaillistNodes ) { 
      my $maillistsNode = XmlNode->new( 'maillists' );

      $maillistsNode->addChild( &makeStatusEnabled() );
      foreach my $cpMaillistNode ( @cpMaillistNodes ) {
        my $maillistNode = XmlNode->new( 'maillist' );
        my $domainName = $cpDomainNode->getAttribute( 'name' );
        my $listname = $cpMaillistNode->getAttribute( 'name' ) . "_" . $domainName;

        my %maillistMetadata;
        $maillistMetadata{'domain'} = $domainName;
        $maillistMetadata{'listname'} = $listname;
        $maillistNode->setMetadata(\%maillistMetadata);

        $maillistNode->setAttribute( 'name', $cpMaillistNode->getAttribute( 'name' ) );

        $maillistNode->addChild( &makeStatusEnabled() );

        my @cpOwnerNodes = $cpMaillistNode->getChildren( 'owner' );
        foreach my $cpOwnerNode ( @cpOwnerNodes ) {
          # 'owner' format is the same, just copy element
          $maillistNode->addChild($cpOwnerNode->copy());
        }

        my $cpPasswordNode = $cpMaillistNode->getChild( 'password' );
        if ( defined ( $cpPasswordNode ) ) {
          my $passwordNode = XmlNode->new( 'password' );
          $passwordNode->setAttribute( 'type', $cpPasswordNode->getAttribute( 'type' ) );
          $passwordNode->setText( $cpPasswordNode->getText() );
          $maillistNode->addChild($passwordNode);
        }

        my @cpRecipientNodes = $cpMaillistNode->getChildren( 'recipient' );
        foreach my $cpRecipientNode ( @cpRecipientNodes ) {
          # 'recipient' format is the same, just copy element
          $maillistNode->addChild($cpRecipientNode->copy());
        }

        $maillistsNode->addChild( $maillistNode );
      }
      $domainNode->addChild( $maillistsNode );
    }
  }
}

sub addDomainDatabases {
  my ($domainNode, $cpDomainNode) = @_;

  unless ( testXmlNodeParam($domainNode,   'domain', 'addDomainDatabases') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode, 'domain', 'addDomainDatabases') ) {return undef;}

  my @cpDatabaseNodes = $cpDomainNode->getChildren( 'database' );
  if (@cpDatabaseNodes) {
    my @databaseNodes = [];

    foreach my $cpDatabaseNode (@cpDatabaseNodes) {
      my $databaseNode = XmlNode->new( 'database' );
      $databaseNode->setAttribute( 'name', $cpDatabaseNode->getAttribute( 'name' ) );
      $databaseNode->setAttribute( 'type', $cpDatabaseNode->getAttribute( 'type' ) );
      # Plesk dump format requires guid, could be empty however, will be fixed with guid fixer before restore
      $databaseNode->setAttribute( 'guid', '' );

      my $version = $cpDatabaseNode->getAttribute( 'version' );
      $databaseNode->setAttribute( 'version', $version ) if defined $version;

      my @cpDbuserNodes = $cpDatabaseNode->getChildren( 'dbuser' );
      if (@cpDbuserNodes) {
        my @dbuserNodes = [];

        foreach my $cpDbuserNode (@cpDbuserNodes) {
          my $dbuserNode = XmlNode->new( 'dbuser' );
          $dbuserNode->setAttribute( 'name', $cpDbuserNode->getAttribute( 'name' ) );
          my $cpDbuserPassword = $cpDbuserNode->getChild( 'password' );
          my $dbuserPassword = XmlNode->new( 'password' );
          # cPanel password type is 'encrypted' or 'empty'. Both are allowed to be Plesk' password type. No token conversion required.
          $dbuserPassword->setAttribute( 'type', $cpDbuserPassword->getAttribute( 'type') );
          $dbuserPassword->setText( $cpDbuserPassword->getText() );
          $dbuserNode->addChild( $dbuserPassword );

          my @cpAccesshostNodes = $cpDbuserNode->getChildren( 'accesshost' );
          if (@cpAccesshostNodes) {
            foreach my $cpAccesshostNode (@cpAccesshostNodes) {
              my $accesshostNode = XmlNode->new( 'accesshost' );
              $accesshostNode->setText( $cpAccesshostNode->getText() );
              $dbuserNode->addChild( $accesshostNode );
            }
          }

          $databaseNode->addChild( $dbuserNode );
        }
      }

      push @databaseNodes, $databaseNode;
    }
    if (@databaseNodes) {
      my $databasesNode = XmlNode->new( 'databases' );

      my %databasesMetadata;
      $databasesMetadata{'domain'} = $cpDomainNode->getAttribute( 'name' );

      $databasesNode->setMetadata(\%databasesMetadata);

      foreach my $databaseNode (@databaseNodes) {
        $databasesNode->addChild( $databaseNode );
      }
      $domainNode->addChild( $databasesNode );
    }
  }
}

sub addDomainuser {
  my ($domainNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($domainNode,    'domain',  'addDomainuser') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode, 'account', 'addDomainuser') ) {return undef;}

  # cPanel account (not reseller) that belongs to reseller account should be mapped into Plesk domain user.
  # So far until only cPanel domains are migrated, no domain users mapping will be performed.
}

sub getPhosting {
  my ($domainNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($domainNode, 'domain', 'getPhosting') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode, 'account', 'getPhosting') ) {return undef;}

  my $quota;
  my @cpLimitNodes = $cpAccountNode->getChildren( 'limit' );
  if (@cpLimitNodes) {
    foreach my $cpLimitNode (@cpLimitNodes) {
      if ( $cpLimitNode->getAttribute( 'name' ) eq 'disk_space' ) {
        $quota = $cpLimitNode->getText();
        last;
      }
    }
  }

  my $cpDomainNode = $cpAccountNode->getChild( 'domain' );

  my $phostingNode = &makePhostingAttrs();

  my %phostingMetadata;
  $phostingMetadata{'domain'} = $cpDomainNode->getAttribute( 'name' );
  $phostingMetadata{'account'} = $cpAccountNode->getAttribute( 'name' );

  $phostingNode->setMetadata(\%phostingMetadata);

  my $phostingPreferencesNode = getPhostingPreferences( $phostingNode, $cpDomainNode);
  $phostingNode->addChild( $phostingPreferencesNode );

  &addPhostingLimitsAndPermissions( $phostingNode);

  &addPhostingWebusersAndFtpusers( $phostingNode, $cpDomainNode);

  &addPhostingSubdomains( $phostingNode, $cpDomainNode);

  return $phostingNode;
}

sub getPhostingPreferences {
  my ($phostingNode, $cpDomainNode) = @_;

  unless ( testXmlNodeParam($phostingNode, 'phosting', 'getPhostingPreferences') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode, 'domain',   'getPhostingPreferences') ) {return undef;}

  my $preferencesNode = XmlNode->new( 'preferences' );

  my $sysuserNode;

  my @cpFtpuserNodes = $cpDomainNode->getChildren( 'ftpuser' );
  if (@cpFtpuserNodes) {
    foreach my $cpFtpuserNode (@cpFtpuserNodes) {
      unless ( defined $cpFtpuserNode->getAttribute( 'directory' ) ) {
        $sysuserNode = XmlNode->new( 'sysuser' );
        $sysuserNode->setAttribute( 'name', $cpFtpuserNode->getAttribute( 'name' ) );
        $sysuserNode->setAttribute( 'quota', $cpFtpuserNode->getAttribute( 'quota' ) ) if ( defined ( $cpFtpuserNode->getAttribute( 'quota' ) ) );
        my $passwordNode = XmlNode->new( 'password' );
        $passwordNode->setAttribute( 'type', 'encrypted' );
        $passwordNode->setText( $cpFtpuserNode->getAttribute( 'password' ) );
        $sysuserNode->addChild( $passwordNode );
        last;
      }
    }
  }
  unless ( defined ( $sysuserNode ) ) {
    Logging::error("Unable get ftp user account for domain '" . $cpDomainNode->getAttribute( 'name' ) . "'");
    $sysuserNode = XmlNode->new( 'sysuser' );
    $sysuserNode->setAttribute( 'name', '' );
  }
  $preferencesNode->addChild( $sysuserNode );

  my $cpAnonftpNode = $cpDomainNode->getChild( 'anonftp' );
  if ( defined ( $cpAnonftpNode ) ) {
    my $anonftpNode = XmlNode->new( 'anonftp' );
    if ( defined ($cpAnonftpNode->getAttribute( 'pub' )) and $cpAnonftpNode->getAttribute( 'pub' ) eq 'true' ) {
      $anonftpNode->setAttribute( 'pub', 'true' );
      my $permissionNode = XmlNode->new( 'anonftp-permission' );
      $permissionNode->setAttribute( 'name', 'incoming-download' );
      $anonftpNode->addChild( $permissionNode );
    }
    if ( defined ($cpAnonftpNode->getAttribute( 'incoming' )) and $cpAnonftpNode->getAttribute( 'incoming' ) eq 'true' ) {
      $anonftpNode->setAttribute( 'incoming', 'true' );
      my $permissionNode = XmlNode->new( 'anonftp-permission' );
      $permissionNode->setAttribute( 'name', 'incoming-mkdir' );
      $anonftpNode->addChild( $permissionNode );
    }
    $preferencesNode->addChild( $anonftpNode );
  }

  addPhostingPreferencesPdirs( $preferencesNode, $cpDomainNode);

  return $preferencesNode;
}

sub addPhostingLimitsAndPermissions {
  my ($phostingNode) = @_;

  unless ( testXmlNodeParam($phostingNode, 'phosting', 'addPhostingLimitsAndPermissions') ) {return undef;}

  my $limitsAndPermissionsNode = XmlNode->new( 'limits-and-permissions' );
  my $scriptingNode = &makeScriptingAll();

  $limitsAndPermissionsNode->addChild( $scriptingNode );
  $phostingNode->addChild( $limitsAndPermissionsNode );
}

sub addPhostingWebusersAndFtpusers {
  my ($phostingNode, $cpDomainNode) = @_;

  unless ( testXmlNodeParam($phostingNode, 'phosting', 'addPhostingWebusersAndFtpusers') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode, 'domain',   'addPhostingWebusersAndFtpusers') ) {return undef;}

  my @webuserNodes;

  my @cpFtpuserNodes = $cpDomainNode->getChildren( 'ftpuser' );
  if (@cpFtpuserNodes) {
    foreach my $cpFtpuserNode (@cpFtpuserNodes) {
      if ( defined $cpFtpuserNode->getAttribute( 'directory' ) ) {
        my $webuserNode = XmlNode->new( 'webuser' );
        $webuserNode->setAttribute( 'name', $cpFtpuserNode->getAttribute( 'name' ) );

        my %webuserMetadata;
        $webuserMetadata{'name'} = $cpFtpuserNode->getAttribute( 'name' );
        $webuserNode->setMetadata(\%webuserMetadata);

        my $sysuserNode = XmlNode->new( 'sysuser' );
        $sysuserNode->setAttribute( 'name', $cpFtpuserNode->getAttribute( 'name' ) );
        $sysuserNode->setAttribute( 'quota', $cpFtpuserNode->getAttribute( 'quota' ) ) if ( defined ( $cpFtpuserNode->getAttribute( 'quota' ) ) );
        my $passwordNode = XmlNode->new( 'password' );
        $passwordNode->setAttribute( 'type', 'encrypted' );
        $passwordNode->setText( $cpFtpuserNode->getAttribute( 'password' ) );
        $sysuserNode->addChild( $passwordNode );

        $webuserNode->addChild( $sysuserNode );

        $webuserNode->addChild( &makeScriptingAll() );
        push @webuserNodes, $webuserNode;
      }
    }
    if ( @webuserNodes ) {
      my $webusersNode = XmlNode->new( 'webusers' );
      foreach my $webuserNode ( @webuserNodes ) {
        $webusersNode->addChild( $webuserNode );
      }
      $phostingNode->addChild( $webusersNode );
    }
  }

}

sub addPhostingSubdomains {
  my ($phostingNode, $cpDomainNode) = @_;

  unless ( testXmlNodeParam($phostingNode, 'phosting', 'addPhostingSubdomains') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode, 'domain',   'addPhostingSubdomains') ) {return undef;}

  my @cpSubdomainNodes = $cpDomainNode->getChildren( 'subdomain' );
  if (@cpSubdomainNodes) {
    my @subdomainNodes;

    foreach my $cpSubdomainNode ( @cpSubdomainNodes ) {
      my @cpAddonDomainNodes = $cpSubdomainNode->getChildren( 'addondomain' );
      my $cpFtpuserNode = $cpSubdomainNode->getChild( 'ftpuser' );
      if ((@cpAddonDomainNodes and !defined $cpFtpuserNode) or !@cpAddonDomainNodes) {
        # Convert cPanel subdomains without addon domains to Plesk subdomains
        my $subdomainNode = XmlNode->new( 'subdomain' );
        $subdomainNode->setAttribute( 'name', $cpSubdomainNode->getAttribute( 'name' ) );
        $subdomainNode->setAttribute( 'guid', '' );
        $subdomainNode->setAttribute( 'shared-content', 'true' );
        $subdomainNode->setAttribute( 'https', 'true');

        if ( defined ( $cpFtpuserNode ) ) {
          my $sysuserNode = XmlNode->new( 'sysuser' );
          $sysuserNode->setAttribute( 'name', $cpFtpuserNode->getAttribute( 'name' ) );
          $sysuserNode->setAttribute( 'quota', $cpFtpuserNode->getAttribute( 'quota' ) ) if ( defined ( $cpFtpuserNode->getAttribute( 'quota' ) ) );
          my $passwordNode = XmlNode->new( 'password' );
          $passwordNode->setAttribute( 'type', 'encrypted' );
          $passwordNode->setText( $cpFtpuserNode->getAttribute( 'password' ) );
          $sysuserNode->addChild( $passwordNode );
          $subdomainNode->addChild( $sysuserNode );
        }

	$subdomainNode->addChild( &makeScriptingAll() );
        push @subdomainNodes, $subdomainNode;
      }
    }

    if ( @subdomainNodes ) {
      my $subdomainsNode = XmlNode->new( 'subdomains' );
      foreach my $subdomainNode ( @subdomainNodes ) {
        $subdomainsNode->addChild( $subdomainNode );
      }
      $phostingNode->addChild( $subdomainsNode );
    }

  }
}

sub addPhostingPreferencesPdirs {
  my ($preferencesNode, $cpDomainNode) = @_;

  unless ( testXmlNodeParam($preferencesNode, 'preferences', 'addPhostingPreferencesPdirs') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode,    'domain',      'addPhostingPreferencesPdirs') ) {return undef;}

  my @cpPdirNodes = $cpDomainNode->getChildren( 'pdir' );
  if (@cpPdirNodes) {
    foreach my $cpPdirNode (@cpPdirNodes) {
      my $pdirNode = XmlNode->new( 'pdir' );
      $pdirNode->setAttribute( 'name', $cpPdirNode->getAttribute( 'name' ) );
      my $title = $cpPdirNode->getAttribute( 'title' );
      $pdirNode->setAttribute( 'title', $title ) if ( defined $title );
      $pdirNode->setAttribute( 'nonssl', 'true' );
      
      my @cpPduserNodes = $cpPdirNode->getChildren( 'pduser' );
      if (@cpPduserNodes) {
        foreach my $cpPduserNode (@cpPduserNodes) {
          my $pduserNode = XmlNode->new( 'pduser' );
          $pduserNode->setAttribute( 'name', $cpPduserNode->getAttribute( 'name' ) );

          my $password = $cpPduserNode->getAttribute( 'password' );
          if ( defined ($password) ) {
            my $passwordNode = XmlNode->new( 'password' );
            $passwordNode->setAttribute( 'type', 'encrypted' );

            my $encoding = $cpPduserNode->getAttribute( 'encoding' );
            $passwordNode->setAttribute( 'encoding', $encoding ) if defined ( $encoding );
            $passwordNode->setText( $password );

            $pduserNode->addChild( $passwordNode );
          }

          $pdirNode->addChild( $pduserNode );
        }
      }
      $preferencesNode->addChild( $pdirNode );
    }
  }
}

sub processSubdomain2Phosting {
  my ($domainFromSubdomainNode, $cpSubdomainNode, $cpDomainNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($domainFromSubdomainNode, 'domain',    'processSubdomain2Phosting') ) {return undef;}
  unless ( testXmlNodeParam($cpSubdomainNode,         'subdomain', 'processSubdomain2Phosting') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode,            'domain',    'processSubdomain2Phosting') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode,           'account',   'processSubdomain2Phosting') ) {return undef;}

  my $domainName = $domainFromSubdomainNode->getAttribute( 'name' );

  # Get domain preferences (aliases only)
  my $domainPreferencesNode = XmlNode->new( 'preferences' );

  my @cpSubdomainAddondomainNodes = $cpSubdomainNode->getChildren( 'addondomain' );
  # At least one addondomain should always exist, because addondomain is required for subdomain-to-domain conversion

  foreach my $cpSubdomainAddondomainNode ( @cpSubdomainAddondomainNodes ) {
    next if ( $cpSubdomainAddondomainNode->getText() eq $domainName );

    my $domainAliasNode = XmlNode->new( 'domain-alias' );
    $domainAliasNode->setAttribute( 'name', $cpSubdomainAddondomainNode->getText() );
    $domainAliasNode->setAttribute( 'mail', 'true');
    $domainAliasNode->setAttribute( 'web',  'true');
    $domainPreferencesNode->addChild( $domainAliasNode );
  }

  $domainFromSubdomainNode->addChild( $domainPreferencesNode );

  # Get domain properties (ip and status only)
  my $domainPropertiesNode = &getDomainProperties($cpAccountNode);
  $domainFromSubdomainNode->addChild( $domainPropertiesNode );

  # Add the same limits as for main account's domain
  addDomainLimitsAndPermissions( $domainFromSubdomainNode, $cpAccountNode );

  addSubdomainMailsystem( $domainFromSubdomainNode, $cpSubdomainNode, $cpAccountNode );

  my $phostingNode = &makePhostingAttrs();

  my %phostingMetadata;
  $phostingMetadata{'domain'} = $cpDomainNode->getAttribute( 'name' );
  $phostingMetadata{'account'} = $cpAccountNode->getAttribute( 'name' );
  $phostingMetadata{'from_subdomain'} = $cpSubdomainNode->getAttribute( 'name' );
  $phostingMetadata{'to_domain'} = $domainName;
  $phostingNode->setMetadata(\%phostingMetadata);

  my $cpFtpuserNode = $cpSubdomainNode->getChild( 'ftpuser' );
  # $cpFtpuserNode should always exist, because ftpuser is required for subdomain-to-domain conversion

  my $sysuserNode = XmlNode->new( 'sysuser' );
  $sysuserNode->setAttribute( 'name', $cpFtpuserNode->getAttribute( 'name' ) );
  $sysuserNode->setAttribute( 'quota', $cpFtpuserNode->getAttribute( 'quota' ) ) if ( defined ( $cpFtpuserNode->getAttribute( 'quota' ) ) );
  my $passwordNode = XmlNode->new( 'password' );
  $passwordNode->setAttribute( 'type', 'encrypted' );
  $passwordNode->setText( $cpFtpuserNode->getAttribute( 'password' ) );
  $sysuserNode->addChild( $passwordNode );
  my $preferencesNode = XmlNode->new( 'preferences' );
  $preferencesNode->addChild( $sysuserNode );

  $phostingNode->addChild( $preferencesNode );

  &addPhostingLimitsAndPermissions($phostingNode);

  $domainFromSubdomainNode->addChild( $phostingNode );
}

sub processSubdomain2Shosting {
  my ($domainFromSubdomainNode, $cpSubdomainNode, $cpDomainNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($domainFromSubdomainNode, 'domain',    'processSubdomain2Shosting') ) {return undef;}
  unless ( testXmlNodeParam($cpSubdomainNode,         'subdomain', 'processSubdomain2Shosting') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode,            'domain',    'processSubdomain2Shosting') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode,           'account',   'processSubdomain2Shosting') ) {return undef;}

  my $domainName = $cpDomainNode->getAttribute( 'name' );

  my $domainPreferencesNode = XmlNode->new( 'preferences' );
  $domainFromSubdomainNode->addChild( $domainPreferencesNode );

  # Get domain properties (ip and status only)
  my $domainPropertiesNode = &getDomainProperties($cpAccountNode);
  $domainFromSubdomainNode->addChild( $domainPropertiesNode );

  $domainFromSubdomainNode->addChild( &makeZeroDomainLimitsAndPermissions() );

  my $shostingNode = XmlNode->new( 'shosting' );
  my $subdomainName = $cpSubdomainNode->getAttribute( 'name' );
  $shostingNode->setText( $subdomainName.".".$domainName);

  $domainFromSubdomainNode->addChild( $shostingNode );
}
#
# End of node transformation subs
#

#
# Begin static content getters
#
sub makeStatusEnabled {
  my $statusEnabledNode = XmlNode->new( 'status' );
  $statusEnabledNode->addChild( XmlNode->new( 'enabled' ) );
	return $statusEnabledNode;
}

sub makeScriptingAll {
  my $scriptingNode = XmlNode->new( 'scripting' );
  $scriptingNode->setAttribute( 'ssi',    'true' );
  $scriptingNode->setAttribute( 'php',    'true' );
  $scriptingNode->setAttribute( 'cgi',    'true' );
  $scriptingNode->setAttribute( 'perl',   'true' );
  $scriptingNode->setAttribute( 'python', 'true' );
  return $scriptingNode;
}

sub makePhostingAttrs {
  my $phostingNode = XmlNode->new( 'phosting' );
  $phostingNode->setAttribute( 'wu_script',      'true' );
  $phostingNode->setAttribute( 'guid',           '' );
  $phostingNode->setAttribute( 'https',          'true' );
  $phostingNode->setAttribute( 'fp',             'true' );
  $phostingNode->setAttribute( 'fpssl',          'true' );
  $phostingNode->setAttribute( 'fpauth',         'true' );
  $phostingNode->setAttribute( 'webstat',        'true' );
  $phostingNode->setAttribute( 'errdocs',        'true' );
  $phostingNode->setAttribute( 'shared-content', 'true' );
  return $phostingNode;
}

sub makeZeroDomainLimitsAndPermissions {
  my $limitsAndPermissionsNode = XmlNode->new( 'limits-and-permissions' );
  $limitsAndPermissionsNode->addChild( 'limit', 'content' => '0',     'attributes' => { 'name' => 'max_traffic' } );
  $limitsAndPermissionsNode->addChild( 'limit', 'content' => '0',     'attributes' => { 'name' => 'max_maillists' } );
  $limitsAndPermissionsNode->addChild( 'limit', 'content' => '0',     'attributes' => { 'name' => 'max_box' } );
  $limitsAndPermissionsNode->addChild( 'limit', 'content' => '0',     'attributes' => { 'name' => 'max_db' } );
  $limitsAndPermissionsNode->addChild( 'limit', 'content' => '0',     'attributes' => { 'name' => 'max_subdom' } );
  $limitsAndPermissionsNode->addChild( 'limit', 'content' => '51200', 'attributes' => { 'name' => 'disk_space' } );
  $limitsAndPermissionsNode->addChild( 'limit', 'content' => '0',     'attributes' => { 'name' => 'max_wu' } );
  return $limitsAndPermissionsNode;
}
#
# End of static content getters
#



#
# Begin miscellanneous subs
#
sub testXmlNodeParam {
  my ($param, $test4name, $message) = @_;
  unless ( ref($param) =~ /XmlNode/ ) {
    Logging::error("XmlNode instance is required : " . $message);
    return undef;
  }

  unless ( $param->getName() eq $test4name ) {
    Logging::error( "'" . $test4name . "' XmlNode is required, got '" . $param->getName() . "' : " . $message);
    return undef;
  }
  return 1;
}
#
# End of miscellanneous subs
#


1;
