package ContentDumper;

use strict;
use warnings;

use Logging;
use XmlNode;
use ConfixxConfig;
use File::Path;     # for rmtree()
use File::Basename; # for dirname()

sub getPHostingContent {
  my ($base, $userName, $domainName, $domainDir, $pDirs_ptr) = @_;

  Logging::trace("Getting phosting content for domain '$domainName' ... ");

  my $topDir = ConfixxConfig::getValue("user_homeDir") . "/$userName/html";

  my @children;
  my (@excludedPaths, @excludedCgiPaths);
  while (my ($path, $info) = each %{$pDirs_ptr}) {
    if ($path =~ /^\/cgi-bin$/) { # cgi directory - /cgi-bin itself
      push @excludedCgiPaths, "./.htaccess";
      push @excludedCgiPaths, "./.htpasswd" if ($info->{'users'});
    } elsif ($path =~ /^\/cgi-bin(\/.*)/) { # cgi directory - a directory under /cgi-bin
      push @excludedCgiPaths, "." . $1 . "/.htaccess";
      push @excludedCgiPaths, "." . $1 . "/.htpasswd" if ($info->{'users'});
    } else { # regular directory under document root
      push @excludedPaths, "." . $path . "/.htaccess";
      push @excludedPaths, "." . $path . "/.htpasswd" if ($info->{'users'});
    }
  }
  push @excludedPaths, "./cgi-bin";

  my $wwwContent = _packUpSinglePhostingDirectory($base, $domainName, $topDir . "/$domainDir", "docroot", \@excludedPaths);
  push @children, $wwwContent if (ref($wwwContent) eq 'XmlNode');
  my $cgiContent = _packUpSinglePhostingDirectory($base, $domainName, $topDir . "/cgi-bin", "cgi", \@excludedCgiPaths);
  push @children, $cgiContent if (ref($cgiContent) eq 'XmlNode');

  if ( @children ) {
    return XmlNode->new('content',
      'children' => \@children
    );
  }

  Logging::warning("No phosting content found for domain '$domainName'.");
  return undef;
}

sub getSitePHostingContent {
  my ($base, $domainName, $vhostFileContent) = @_;

  unless (length($vhostFileContent) > 0) {
    return undef;
  }

  my $tempDir = '/tmp/migration_vhost';
  my $vhostFileName = "$tempDir/vhost.conf";
  mkdir $tempDir;
  DumperUtils::writeFile($vhostFileName, $vhostFileContent);

  my $confContent = _packUpSinglePhostingDirectory($base, $domainName, $tempDir, "conf"); 

  rmtree($tempDir);

  unless (ref($confContent) eq 'XmlNode') {
    return undef;
  }
  
  return XmlNode->new('content',
    'children' => [ $confContent ]
  );
}

sub _packUpSinglePhostingDirectory {
  my ($base, $domainName, $rootPath, $contentType, $excludedPaths) = @_;

  my $contentFileName = $base->{namecreator}->getPhostingDstFile($contentType, $domainName);
  return _packUpSingleDirectory($base, $rootPath, $contentType, $contentFileName, $excludedPaths);
}

# $excludedPaths must be specified relative to $rootPath, and start with "./",
# otherwise tar won't match them and will include them into tarball.
# Examples of $contentFileName:
#   * $base->{namecreator}->getPhostingDstFile('cgi', $domainName)
#   * $base->{namecreator}->getSubdomainDstFile('docroot', $domainName, $subdomainName )
# full list of content filename creator functions can see at common/.../shared/Storage/ContentNameCreator.pm
sub _packUpSingleDirectory {
  my ($base, $rootPath, $contentType, $contentFileName, $excludedPaths) = @_;

  my $contentNode;
  if (-d $rootPath) {
    my %options;
    $options{'checkEmptyDir'} = 1;
    $options{'directory'} = $rootPath;
    $options{'include_hidden_files'} = 1;
    $options{'exclude'} = $excludedPaths if ( ref($excludedPaths) eq 'ARRAY' );
    $contentNode = $base->{content_transport}->addContent($contentType, $contentFileName, %options);
  }
  return $contentNode;
}

sub getMailBoxContent {
  my ($base, $mboxName, $domainName) = @_;
  my $mailboxNode;
  my %options;
  my $convertedMailDir;

  $options{'checkEmptyDir'} = 1;
  $options{'include_hidden_files'} = 1;
  $options{'follow_symlinks'} = 0;

  # Syntax of confixx_main.conf:
  # - maildrop can be : /var/spool/mail/USER | HOMEDIR/Mailbox | HOMEDIR/Maildir/
  # - pop_homeDir is location of mail users homedir. Example: $pop_homeDir = '/home/email';
  # - mailBoxName is name of Mailbox file. Example: $mailBoxName = 'Maildir';
  # - mailSpool is location of mailspool directory. Example: $mailSpool = '/var/spool/mail';
  #
  if (ConfixxConfig::getValue('maildrop') eq "HOMEDIR/Maildir/") {
    $options{'directory'} = ConfixxConfig::getValue("pop_homeDir") . "/$mboxName/" . ConfixxConfig::getValue('mailBoxName');
  } else {
    #
    #   Convert from mbox format to mdir.
    #   Attention: in this case will migrate only Inbox, because:
    #       - old migrator did the same
    #       - Confixx dont known about mail storage (use IMAP protocol via dovecot for access to mailboxes dirs)
    #       - for access to other maildirs require parse /etc/dovecot/dovecot.conf (mail_location parameter)
    my $inbox;
    $convertedMailDir = "/tmp/migration_dump_mail";       # temporary dir

    if (ConfixxConfig::getValue('maildrop') eq "HOMEDIR/Mailbox") {
      $inbox = ConfixxConfig::getValue('pop_homeDir') . "/$mboxName/" . ConfixxConfig::getValue('mailBoxName');
    } elsif (ConfixxConfig::getValue('maildrop') eq "/var/spool/mail/USER") {
      $inbox = ConfixxConfig::getValue('mailSpool') . "/$mboxName";
    } else {
      Logging::warning("Can not identify format of maildrop in Confixx config: " . ConfixxConfig::getValue('maildrop'));
      return undef;
    }

    unless(-r $inbox) {
      Logging::warning("Can not read mailbox for user '$mboxName': $inbox\n");
      return undef;
    }

    rmtree($convertedMailDir);
        # we do not use Mbox2Mdir::convert since it contains customizations, which we don't want
    my $mb2mdPath = dirname(__FILE__) . '/shared_legacy/mb2md.pl';
    `/usr/bin/perl $mb2mdPath -s $inbox -d $convertedMailDir`;

    $options{'directory'} = $convertedMailDir;
  }

  my $contentFileName = $base->{namecreator}->getMailnameDstFile($mboxName, $domainName);
  $mailboxNode = $base->{content_transport}->addContent('mailbox', $contentFileName, %options);

  if (defined($convertedMailDir)) {
    rmtree($convertedMailDir);
  }

  if (ref($mailboxNode) ne 'XmlNode') {
      Logging::warning("No content found for mbox $mboxName .");
      return undef;
  }

  return XmlNode->new('content',
    'children' => [
        $mailboxNode
    ]
  );
}

1;
