package ContentDumper;

use strict;

use AgentConfig;
use Logging;
use Dumper;
use File::Basename;
use File::Path;
use Imapd2Md;
use XmlNode;

sub getPHostingContent {
  my ($base, $domainName) = @_;

  Logging::trace("Getting phosting content for domain '$domainName' ... ");

  my $wwwRoot = "/var/www/html";
  my $cgiRoot = "/var/www/cgi-bin";

  my @children;
  my $wwwContent = _packUpSinglePhostingDirectory($base, $domainName, $wwwRoot, "docroot", _makeExcludedHtaccessList($domainName, $wwwRoot));
  push @children, $wwwContent if (ref($wwwContent) =~ /XmlNode/);
  my $cgiContent = _packUpSinglePhostingDirectory($base, $domainName, $cgiRoot, "cgi", _makeExcludedHtaccessList($domainName, $cgiRoot));
  push @children, $cgiContent if (ref($cgiContent) =~ /XmlNode/);

  if ( @children ) {
    my $contentNode = XmlNode->new('content',
      'children' => \@children
    );
    return $contentNode;
  }

  Logging::warning("No phosting content found for domain '$domainName'.");
}

sub getSubdomainPHostingContent {
  my ($base, $domainName, $pHostingNode, $subdomainName, $wwwdir, $cgidir) = @_;
  my @children;

  Logging::trace("Getting SubdomainPhosting content for domain '$domainName' with subdomain '$subdomainName' ... ");

  if (defined($cgidir) && $cgidir !~ m/^\s*$/)
  {
    my $cgiContent = _packUpSingleSubdomainDirectory($base, $domainName, $subdomainName, $cgidir, "cgi");
        # can use '_makeExcludedHtaccessList($domainName, $wwwdir)' for fill last argument (@exclude_list),
        # but subdomains not using Protect Directories at webpanel management.
        # So, keep it empty.

    if (ref($cgiContent) =~ /XmlNode/) {
        push @children, $cgiContent;
        $pHostingNode->setAttribute('cgi_bin_mode', "www-root");     # this is realy require for Plesk, for access to "cgi-bin"
                                                                     # directories for ftpusers/owners of subdomains.
                                                                     # And yes, it's UGLY hack for this place, but it simplest way.
    }
  }

  my $wwwContent = _packUpSingleSubdomainDirectory($base, $domainName, $subdomainName, $wwwdir, "docroot",
    ($cgidir eq "$wwwdir/cgi-bin" ? "./cgi-bin" : undef ) );        # same thing about _makeExcludedHtaccessList() see above
  push @children, $wwwContent if (ref($wwwContent) =~ /XmlNode/);

  if ( @children ) {
    my $contentNode = XmlNode->new('content',
      'children' => \@children
    );
    return $contentNode;
  }

  Logging::warning("No Subdomain phosting content found for domain '$domainName' subdomain '$subdomainName' .");
}


sub getAnonFtpContent {
  my ($base, $domainName) = @_;

  my @children;
  my $pubContent = _packUpSinglePhostingDirectory($base, $domainName, "/var/ftp/pub", "ftp_pub");
  push @children, $pubContent if (ref($pubContent) =~ /XmlNode/);
  my $incomingContent = _packUpSinglePhostingDirectory($base, $domainName, "/var/ftp/uploads", "ftp_incoming");
  push @children, $incomingContent if (ref($incomingContent) =~ /XmlNode/);

  if (@children) {
    my $contentNode = XmlNode->new('content',
      'children' => \@children
    );
    return $contentNode;
  }

  Logging::info("No anonymous FTP content found for domain '$domainName'.");
}

sub getWebUserSiteContent {
  my ($base, $domainName, $webUserName) = @_;

  my $contentNode = _packUpSingleWebUserDirectory($base, $domainName, $webUserName, "/home/$webUserName/public_html", "webuser_home");
  if (ref($contentNode) =~ /XmlNode/) {
    $contentNode->setAttribute('offset', "web_users/$webUserName");
    return XmlNode->new('content',
      'children' => [$contentNode]
    );
  }

  Logging::info("No site content found for web user $webUserName at domain $domainName");
}

sub getMailBoxContent {
  my ($base, $domainName, $userName, $userInfo) = @_;

  # Inbox is stored in mailbox format in var/spool/mail
	my $mailHome = Dumper::getFilesystemByDomainName($domainName) . "/home/$userName";
  my $dumpDir = "$mailHome/migration_dump";
  my $idname = $dumpDir . "/$userName" . "@" . "$domainName.mdir";
  my $srcdir = $idname . ".converted";
  mkpath($srcdir);
  unless (-d $srcdir) {
    die "Cannot create folder to convert mail messages!";
  }

	my $inboxDirectory = Dumper::getFilesystemByDomainName($domainName) . "/var/spool/mail/";
  my $inboxFile = $inboxDirectory . $userName;
  if (-f $inboxFile) {
    # mb2md.pl was taken from http://batleth.sapienti-sat.org/projects/mb2md/mb2md-3.20.pl.gz
    my $mb2mdPath = dirname(__FILE__) . '/shared_legacy/mb2md.pl';
    `/usr/bin/perl $mb2mdPath -s $inboxFile -d $srcdir`;
  }

  # all other IMAP-directories (except Inbox) are stored in IMAP format in user's home directory
	my @mail;
	my $mailBoxList = "$mailHome/.mailboxlist";
	if (-f $mailBoxList) {
	  open MAILBOXLIST, "<$mailBoxList";
	  binmode MAILBOXLIST;
    while (<MAILBOXLIST>) {
      chomp;
      if (-f "$mailHome/$_") {
        push @mail, $_;
      }
    }
	}
  if (@mail) {
    foreach my $mailFolder(@mail) {
      Logging::debug("Convert $userName\@$domainName mail folder '$mailFolder'");
      my $destPath = $mailFolder;
      $destPath =~ s/^mail//g;
      $destPath =~ s/^\///g;
      $destPath =~ s/\./_/g;
      $destPath =~ s/\//\./g;
      $destPath = ".$destPath";
      my $res = Imapd2Md::convertit( "$mailHome/$mailFolder", "$srcdir/$destPath" );
      if($res) {
        Logging::warning("Error while convertion mailbox '$mailFolder': $res ");
      } else {
        system("touch", "$srcdir/$destPath/maildirfolder") if -e "$srcdir/$destPath";
      }
    }
  }

  my %options;
  $options{'directory'} = $srcdir;
  $options{'checkEmptyDir'} = 1;
  $options{'include_hidden_files'} = 1;

  my $contentFileName = $base->{namecreator}->getMailnameDstFile($userName, $domainName);
  my $mailboxNode = $base->{content_transport}->addContent('mailbox', $contentFileName, %options);
  rmtree($dumpDir);
  if (ref($mailboxNode) =~ /XmlNode/) {
    return XmlNode->new('content', 'children' => [$mailboxNode]);
  }

  Logging::warning("No content found for $userName\@$domainName maildir.");
}

# In PPCPL, a .htaccess file becomes owned by root after site admin protects
# appropriate directory using CP. Such protected directories we report to Plesk
# separately. If we don't exclude them here, Plesk has problems importing such site -
# unable to remove these .htaccess files at restoration stage.
sub _makeExcludedHtaccessList {
  my ($domainName, $jailedPath) = @_;

  my $rootPath = Dumper::getFilesystemByDomainName($domainName) . $jailedPath;

  my @result;
  my $find = AgentConfig::findBin();
  my $searchedFile = '.htaccess';
  my $htaccessList;
  open($htaccessList, "$find '$rootPath' -name $searchedFile -type f -user root -printf \"./%P\n\" |");
  binmode $htaccessList;
  while (<$htaccessList>) {
    chomp $_;
    push @result, $_;
  }
  close $htaccessList;

  return \@result;
}

sub _packUpSinglePhostingDirectory {
  my ($base, $domainName, $jailedPath, $contentType, $excludedPaths) = @_;

  my $contentFileName = $base->{namecreator}->getPhostingDstFile($contentType, $domainName);
  return _packUpSingleDirectory($base, $domainName, $jailedPath, $contentType, $contentFileName, $excludedPaths);
}

sub _packUpSingleSubdomainDirectory {
  my ($base, $domainName, $subdomainName, $jailedPath, $contentType, $excludedPaths) = @_;

  my $contentFileName = $base->{namecreator}->getSubdomainDstFile($contentType, $domainName, $subdomainName);
  return _packUpSingleDirectory($base, $domainName, $jailedPath, $contentType, $contentFileName, $excludedPaths);
}

sub _packUpSingleWebUserDirectory {
  my ($base, $domainName, $webUserName, $jailedPath, $contentType, $excludedPaths) = @_;

  my $contentFileName = $base->{namecreator}->getWebUserDstFile($domainName, $webUserName);
  return _packUpSingleDirectory($base, $domainName, $jailedPath, $contentType, $contentFileName, $excludedPaths);
}

# $excludedPaths must be specified relative to $jailedPath, and start with "./",
# otherwise tar won't match them and will include them into tarball.
# Examples of $contentFileName:
#   * $base->{namecreator}->getPhostingDstFile('cgi', $domainName)
#   * $base->{namecreator}->getSubdomainDstFile('docroot', $domainName, $subdomainName )
# full list of content filename creator functions can see at common/.../shared/Storage/ContentNameCreator.pm
sub _packUpSingleDirectory {
  my ($base, $domainName, $jailedPath, $contentType, $contentFileName, $excludedPaths) = @_;

  my $fullPath = Dumper::getFilesystemByDomainName($domainName) . "$jailedPath";
  my $contentNode;

  if (-d $fullPath) {
    my %options;
    $options{'checkEmptyDir'} = 1;
    $options{'directory'} = $fullPath;
    $options{'include_hidden_files'} = 1;
    $options{'exclude'} = $excludedPaths if ( ref($excludedPaths) =~ /ARRAY/ );
    $contentNode = $base->{content_transport}->addContent($contentType, $contentFileName, %options);
  }
  return $contentNode;
}

1;
