package Dumper;

use strict;
use warnings;

use AgentConfig;
use Logging;
use Parser;
use SimpleDbConnect;

my $_db;

sub _getDb() {
  unless (defined($_db)) {
    $_db = SimpleDbConnect->new('Pg', undef, undef, 'appldb', undef);
  }

  return $_db;
}

sub _readConfigFile {
  my ( $siteId, $fileName ) = @_;

  my $configReader =
    Parser::makeSafeFileParser(getSiteinfoBySiteId($siteId) . "/current/" . $fileName);

  if ( defined $configReader ) {
    return $configReader->{'PARSE'}->(
      'KEYSEPARATOR' => '=',
      'COMMENT'      => '\[DEFAULT\]'
    );
  }
  else {
    return { "enabled" => 0 };
  }
}

sub getSiteinfoBySiteId {
  my ($siteId) = @_;
  return "/home/virtual/site$siteId/info";
}

sub getFilesystemByDomainName {
  my ($domainName) = @_;
  return "/home/virtual/$domainName";
}

sub getListOfSubdomains {
  my ($domainName) = @_;
  my $configListOfSubdomains;
  my $currentSubdomain;
  my $subcount=0;

  my $siteId = getSiteIdByName($domainName);

  # Parser::makeSafeFileParser("/etc/virtualhosting/subdomain/site" . $siteId) not supported
  # syntax of this config. So, will read config by hand...

  my ($subdomainsConfigFile);
  open($subdomainsConfigFile, "/etc/virtualhosting/subdomain/site" . $siteId) || return undef;
  while (<$subdomainsConfigFile>) {
    chomp;
    if (/^\[(.*)\]$/) {
        $currentSubdomain = $1;
        $subcount++ if ($currentSubdomain ne "DEFAULT");
        next;
    }
    next if (!defined($currentSubdomain));

    next if (/^#/ || /^$/);
    next if (! m/^(.*?)[ \t]*=[ \t]*(.*)$/ );
    $configListOfSubdomains->{"$currentSubdomain"}{"$1"} = "$2";

  }
  close($subdomainsConfigFile);

  if (defined($configListOfSubdomains->{"DEFAULT"})) {
    return undef unless $subcount;      # check if define only DEFAULT subdomain (and not have other subdomain(s))

    # fill now default values in all subdomain(s)
    foreach my $subdomainName ( keys %{$configListOfSubdomains} ) {
        foreach my $values_subdomain (keys %{$configListOfSubdomains->{"DEFAULT"}}) {
            if (!defined($configListOfSubdomains->{"$subdomainName"}{$values_subdomain})) {
                $configListOfSubdomains->{"$subdomainName"}{$values_subdomain}
                    = $configListOfSubdomains->{DEFAULT}{$values_subdomain};
            }
        }
    }
    delete($configListOfSubdomains->{DEFAULT});
  }

  return $configListOfSubdomains;
}


sub getResellers {
  Logging::trace('Getting resellers list...');

  # TODO ddidenkov@ revise WHERE clause
  #  Compare with old clause (taken from EnsimX.pl):
  # " WHERE (reseller_info.reseller_id <> 0) OR (site_id IS NOT NULL)";
  my $sql =
      "SELECT DISTINCT reseller_info.username"
    . " FROM reseller_info LEFT JOIN reseller"
    . " ON reseller_info.reseller_id = reseller.reseller_id"
    . " WHERE (reseller_info.reseller_id <> 0)";
  my $wrapDbh = _getDb()->getDbh();
  my @resellers;
  if ( $wrapDbh->{'EXECUTE'}->($sql) ) {
    my $ptrRow;
    while ( $ptrRow = $wrapDbh->{'FETCHROW'}->() ) {
      my @resellerInfo = @$ptrRow;
      push @resellers, $resellerInfo[0];
    }
  }

  $wrapDbh->{'FINISH'}->();
  Logging::debug( "Found " . scalar(@resellers) . " resellers: @resellers" );
  return @resellers;
}

sub getSiteAdminsForReseller {

  # Could be 'undef' for admin
  my ($reseller) = @_;

  Logging::trace(
    "Getting site admins for '" . ( defined($reseller) ? $reseller : 'undef' ) . "' reseller..." );

 # temporary "resellerId by username" resolving to avoid current DumpComposer's cPanel-ish behaviour
  my $resellerId;
  my $wrapDbh = _getDb()->getDbh();
  if ( defined $reseller ) {
    my $sql = "SELECT reseller_id FROM reseller_info WHERE username = '$reseller'";
    if ( $wrapDbh->{'EXECUTE'}->($sql) ) {
      $resellerId = $wrapDbh->{'FETCHROW'}->()->[0];
      $wrapDbh->{'FINISH'}->();
    }
    else {
      Logging::warning("Unable to get reseller id by its username '$reseller'!");

      # Couldn't find any sites for unknown reseller -> return
      return ();
    }
  }
  else {

    # PPCP uses resellerId = 0 to identify admin sites
    $resellerId = 0;
  }

  my $sql =
      "SELECT admin_user FROM siteinfo"
    . " NATURAL JOIN reseller"
    . " WHERE (reseller.reseller_id = '$resellerId')";
  my @siteAdmins;
  if ( $wrapDbh->{'EXECUTE'}->($sql) ) {
    my $ptrRow;
    while ( $ptrRow = $wrapDbh->{'FETCHROW'}->() ) {
      my @siteInfo = @$ptrRow;
      push @siteAdmins, $siteInfo[0];
    }
  } else {
    Logging::warning("Unable to get site admins for '$resellerId' reseller from the DB!");
  }

  $wrapDbh->{'FINISH'}->();

  Logging::debug( "Found " . scalar(@siteAdmins) . " site admins for '$resellerId' reseller: @siteAdmins" );

  return @siteAdmins;
}

sub getSitesForSiteAdmin {
  my ($siteAdmin) = @_;
 
  my $wrapDbh = _getDb()->getDbh();

  my $sql = "SELECT domain FROM siteinfo WHERE admin_user = '$siteAdmin'";

  my @sites;
  if ( $wrapDbh->{'EXECUTE'}->($sql) ) {
    my $ptrRow;
    while ( $ptrRow = $wrapDbh->{'FETCHROW'}->() ) {
      my @siteInfo = @$ptrRow;
      push @sites, $siteInfo[0];
    }
  }
  else {
    Logging::warning("Unable to get sites by site admin '$siteAdmin'!");
    return ();
  }
    
  $wrapDbh->{'FINISH'}->();
  
  Logging::debug( "Found " . scalar(@sites) . " sites for '$siteAdmin' site admin: @sites" );

  return @sites;
}

sub getResellerGeneralInfo {
  my $reseller = shift;

  Logging::trace("Getting general info about '$reseller' reseller...");

  my $sql =
      "SELECT reseller_id, username, email, password, fullname, enabled"
    . " FROM reseller_info WHERE (username = '$reseller')";

  my $wrapDbh = _getDb()->getDbh();
  my $ptrHash;
  if ( $wrapDbh->{'EXECUTE'}->($sql) ) {
    $ptrHash = $wrapDbh->{'FETCHHASH'}->();
  }
  $wrapDbh->{'FINISH'}->();
  if ( defined($ptrHash) && ref($ptrHash) =~ /HASH/ ) {
    return $ptrHash;
  }
  else {
    return;
  }
}

sub _getResellerIdByName {
  my ( $reseller ) = @_;
  return _getDb()->queryOneValue(
    "SELECT reseller_id FROM reseller_info WHERE username = '$reseller'",
    'reseller_id'
  );
}

sub getResellerLimits {
  my ( $reseller ) = @_;

  my $limits = {
    # units field of reseller_diskquota is ignored because 
    # it is always set to 'MB' however total_quota is in bytes
    # the same applies for other similar values
    'diskquota' => { 
      'table' => 'reseller_diskquota', 
      'enabled_column' => 'enabled_diskquota', 
      'value_column' => 'total_quota'
    },
    'users' => {
      'table' => 'reseller_users',
      'enabled_column' => 'enabled_users',
      'value_column' => 'total_maxusers'
    },
    'bandwidth' => {
      'table' => 'reseller_bandwidth',
      'enabled_column' => 'enabled_bandwidth',
      'value_column' => 'total_threshold'
    },
    'ip_based_sites' => {
      'table' => 'reseller_ipinfo',
      'enabled_column' => 'enabled_ipinfo_ip',
      'value_column' => 'total_ipbased'
    },
    'name_based_sites' => {
      'table' => 'reseller_ipinfo',
      'enabled_column' => 'enabled_ipinfo_nb',
      'value_column' => 'total_namebased' 
    }
  };
  
  return _getObjectLimits($limits, 'reseller_info', 'reseller_id', _getResellerIdByName($reseller));
}

sub getSiteLimits {
  my ( $siteId ) = @_;

  my $limits = {
    'diskquota' => { 
      'table' => 'diskquota', 
      'enabled_column' => 'enabled', 
      'value_column' => 'quota'
    },
    'users' => {
      'table' => 'users',
      'enabled_column' => 'enabled',
      'value_column' => 'maxusers'
    }
  };

  my %limits = _getObjectLimits($limits, 'siteinfo', 'site_id', $siteId);

  # bandwidth has no special limit enabled column,
  # so we have to get it in quite another way
  $limits{'bandwidth'} = _getSiteBandwidthLimit($siteId);

  return %limits;
}

sub _getSiteBandwidthLimit {
  my ( $siteId ) = @_; 

  my $result = _getDb()->queryOneValue(
    "SELECT threshold FROM bandwidth WHERE site_id = '$siteId'"
  );

   # for a bandwidth threshold '0' is a special value for unlimited
  if ($result eq '0') {
    return undef;
  } else {
    return $result;
  }
}

sub _getObjectLimits {
  my ( $limits, $mainTable, $tableKeyName, $tableKeyValue ) = @_;

  my @columnExprs;
  my @joinExprs;

  foreach my $limitName ( keys %{$limits}) {
    my $limit = $limits->{$limitName};

    my $table = $limit->{'table'};
    my $enabledColumn = $limit->{'enabled_column'};
    my $valueColumn = $limit->{'value_column'};

    my $sqlTableName = "t_$limitName";

    push(@columnExprs, "$sqlTableName.$enabledColumn as ${limitName}_enabled");
    push(@columnExprs, "$sqlTableName.$valueColumn as ${limitName}_value");

    push(@joinExprs, "INNER JOIN $table $sqlTableName ON $sqlTableName.$tableKeyName = $mainTable.$tableKeyName");
  }

  my $query = 
    'SELECT ' . join(', ', @columnExprs) . 
    " FROM $mainTable " . join(' ', @joinExprs) . 
    " WHERE $mainTable.$tableKeyName = '$tableKeyValue'";

  my $queryResult = _getDb()->queryOneRow($query);
 
  my %result;

  foreach my $limitName ( keys %{$limits} ) {
    if ( $queryResult->{$limitName . '_enabled'} == '1' ) {
      $result{$limitName} = $queryResult->{$limitName . '_value'};
    } else {
      $result{$limitName} = undef;
    }
  }

  return %result;
}

sub canResellerManageDNS {
  my ( $reseller ) = @_;

  return _getDb()->queryOneValue(
    "SELECT enabled_zone_mgmt FROM reseller_bind" .
    " INNER JOIN reseller_info ON reseller_bind.reseller_id = reseller_info.reseller_id" .
    " WHERE reseller_info.username = '$reseller'"
  );
}

#	my $sql =
#	  "SELECT reseller_id, username, email, password, fullname, enabled,"
#	  . "total_threshold, enabled_bandwidth, total_quota, enabled_diskquota,"
#	  . "total_ipbased, total_namebased, enabled_ipinfo_nb, enabled_ipinfo_ip,"
#	  . "total_maxusers, enabled_users, "
#	  . "enabled_zone_mgmt "
#	  . "FROM reseller_info "
#	  . "NATURAL LEFT JOIN reseller_bandwidth "
#	  . "NATURAL LEFT JOIN reseller_diskquota "
#	  . "NATURAL LEFT JOIN reseller_users "
#	  . "NATURAL LEFT JOIN reseller_ipinfo "
#	  . "NATURAL LEFT JOIN reseller_bind "
#	  . "WHERE (reseller_id = '$accountId')";

# temporary function to avoid current DumpComposer cPanel-ish behaviour
sub getSiteIdByName {
  my $site    = shift;
  my $sql     = "SELECT site_id FROM siteinfo" . " WHERE (domain = '$site')";
  my $wrapDbh = _getDb()->getDbh();
  my $siteId;
  if ( $wrapDbh->{'EXECUTE'}->($sql) ) {
    $siteId = $wrapDbh->{'FETCHROW'}->()->[0];
  }
  else {
    Logging::warning("Unable to get site id by its name '$site'!");
    return;
  }
  $wrapDbh->{'FINISH'}->();
  return $siteId;
}

sub getSpecificInfoForSite {
  my ($siteId, $infoFileName, $infoName) = @_;
  $infoName = $infoFileName unless $infoName;
  Logging::trace("Getting $infoName for '$siteId' site...");
  my %info = %{ _readConfigFile( $siteId, $infoFileName ) };
  my @infoTextDump = map { "$_ => $info{$_}\n" } keys %info;
  Logging::debug("$infoName for '$siteId' site is \n @infoTextDump");
  return %info;
}

sub getSiteIdByAdminName {
  my ( $adminUser ) = @_;
  return _getDb()->queryOneValue(
    "SELECT site_id FROM siteinfo WHERE admin_user = '$adminUser'"
  );
}

sub getSiteDomainByAdminName {
  my ( $adminUser ) = @_;
  return _getDb()->queryOneValue(
    "SELECT domain FROM siteinfo WHERE admin_user = '$adminUser'"
  );
}

sub getSiteIdByDomain {
  my ( $domain ) = @_;
  return _getDb()->queryOneValue(
    "SELECT site_id FROM siteinfo WHERE domain = '$domain'"
  );
}

sub getSiteContactEmailByAdminUser {
  my ( $adminUser ) = @_;
  return _getDb()->queryOneValue(
    "SELECT email FROM siteinfo WHERE admin_user = '$adminUser'"
  );
}

sub getIpInfoForSite {
  my ( $siteId ) = @_;
  return getSpecificInfoForSite($siteId, 'ipinfo', 'IP info');
}

sub getBindInfoForSite {
  my ( $siteId ) = @_;
  return getSpecificInfoForSite($siteId, 'bind', 'DNS info');
}

sub isDisabledSite {
  my $siteId = shift;
  Logging::trace("Getting disable status for '$siteId' site...");
  my $isDisabled = 0;
  if (-f getSiteinfoBySiteId($siteId) . "/disabled") {
    $isDisabled = 1;
  }
  Logging::debug( "'$siteId' site is " . ( ( $isDisabled == 0 ) ? 'enabled' : 'disabled' ) . '.' );
  return $isDisabled;
}

sub getAgentStatus {
  my $root = XmlNode->new('agent-status');

  unless ( checkPanelVersion() && defined AgentConfig::iconvBin() ) {
    my $item;
    if ( defined AgentConfig::iconvBin() ) {
      $item = XmlNode->new('wrong-platform');
      $item->setText('');
    }
    else {
      $item = XmlNode->new('wrong-platform');
      $item->setText('no iconv found on the source host');
    }

    $root->addChild($item);
  }

  return $root;
}

sub getPPCPVersion {

# TODO ddidenkov@ revise this check after PPCP source code analysis (the same thing is for scoup.pl)
# TODO ddidenkov@ discuss the found ways with current PPCP developers
# I see several ways in the PPCP's source code:
# 1. cat /usr/lib/opcenter/VERSION (returns '10.3.4-rhel.5ES.9' in my case). Ref: epl/eplapi/implementors/server/server.py, L35, getEPLVersionInfo
# 2. /bin/rpm -qf --qf %{VERSION}-%{RELEASE} /etc/rc.d/init.d/epld (returns the same '10.3.4-rhel.5ES.9'). Ref: epl/eplapi/licensing/licenselib.py, L115, get_version
# 3. /usr/bin/getapplversion.old . It reads the same /usr/lib/opcenter/VERSION file
# 4. python -c "import sys; sys.path.append('/usr/lib/ensim'); import eplapi.implementors.server.server; print eplapi.implementors.server.server.epl_version_release".
#    This way is used by web-interface to show CP version on the dashboard/Server Administrator page.
#    It reads some string generated during the release build.
  my $main_version_file_name = '/usr/lib/opcenter/VERSION';
  if ( -e $main_version_file_name ) {
    my $version = `cat $main_version_file_name`;
    chomp $version;
    if ( $version =~ /(\d+\.\d+(\.\d+)?)-.*/ ) {
      Logging::debug("Parsed '$1' product version number from '$version' string.");
      return $1;
    }
    else {
      Logging::debug("Couldn't get the product version number from '$version'");
    }
  }
  else {
    Logging::debug("The version file '$main_version_file_name' doesn't exist.");
  }

  {
    my $version =
`python -c "import sys; sys.path.append('/usr/lib/ensim'); import eplapi.implementors.server.server; print eplapi.implementors.server.server.epl_version_release"`;
    chomp $version;
    if ( $version =~ /(\d+\.\d+(\.\d+)?)-.*/ ) {
      Logging::debug("Parsed '$1' product version number from '$version' string.");
      return $1;
    }
    else {
      Logging::debug("Couldn't get the product version number from '$version'");
    }
  }

  my $rpm     = AgentConfig::rpmBin();
  my $version = `$rpm -qf --qf \%{VERSION} /etc/rc.d/init.d/epld 2>/dev/null`;
  if ( $version =~ /^(\d+\.\d+(\.\d+)?)$/ ) {
    Logging::debug("Parsed '$1' product version number from '$version' string.");
    return $1;
  }
  else {
    Logging::debug("Couldn't get the product version number from '$version'");
  }

  # there are older ways to determine the version below
  my @rpms_list = ( 'webppliance-version', 'webppliance-ipinfo' );
  for my $rpm_name (@rpms_list) {
    my $version = `$rpm -q $rpm_name 2>/dev/null`;
    if ( $version =~ /^$rpm_name-([^-]+)-[^-]+$/ ) {
      Logging::debug("Parsed '$1' product version number from '$version' string.");
      return $1;
    }
    else {
      Logging::debug("Couldn't get the product version number from '$version'");
    }
  }

  my @version_files_list = ( '/usr/bin/getapplversion', '/usr/bin/getapplversion.old' );
  for my $version_file_name (@version_files_list) {
    if ( -e $version_file_name ) {
      my $version = `$version_file_name`;
      chomp $version;
      if ( $version =~ /(\d+\.\d+(\.\d+)?)-.*/ ) {
        Logging::debug("Parsed '$1' product version number from '$version' string.");
        return $1;
      }
      else {
        Logging::debug("Couldn't get the product version number from '$version'");
      }
    }
    else {
      Logging::debug("The version file '$version_file_name' doesn't exist.");
    }
  }
  Logging::debug("Couldn't detect a PPCP.");
  return;
}

sub checkPanelVersion {
  my $version = getPPCPVersion();

  if ( $version =~ /^(\d+)\.(\d+)/ ) {
    Logging::debug(
      "The product version number '$version'" . " contains major($1) and minor($2) parts." );

    # Check version major and minor fields
    if ( $1 eq '10' && $2 eq '3' ) {
      return $version;
    }
    else {
      Logging::info("The PPCP version '$version' is not supported.");
      return;
    }
  }
  Logging::info("Couldn't determine major and minor version numbers from '$version'.");
  return;
}


my %ignoredUsers = (
  'root' => '1',
  'bin' => '1',
  'daemon' => '1',
  'adm' => '1',
  'lp' => '1',
  'sync' => '1',
  'shutdown' => '1',
  'halt' => '1',
  'mail' => '1',
  'uucp' => '1',
  'operator' => '1',
  'games' => '1',
  'gopher' => '1',
  'ftp' => '1',
  'news' => '1',
  'mysql' => '1',
  'nobody' => '1',
  'zope' => '1',
  'majordomo' => '1',
  'tomcat4' => '1',
  'apache' => '1',
  'ensimrootmail' => '1',
  'sshd' => '1',
  'smmsp' => '1',
);

sub getSiteUsersInfoMap {
  my ($domainName) = @_;

  my $passwdReader = Parser::makeSafeFileParser(getFilesystemByDomainName($domainName) . "/etc/passwd");
  my $shadowReader = Parser::makeSafeFileParser(getFilesystemByDomainName($domainName) . "/etc/shadow");

  my (%passwd, %shadow);

  if (defined($passwdReader)) {
    %passwd = %{$passwdReader->{'PARSE'}->('KEYSEPARATOR' => ':', 'VALUESEPARATOR' => ':')};
  } else {
    return undef;
  }

  if (defined($shadowReader)) {
    %shadow = %{$shadowReader->{'PARSE'}->('KEYSEPARATOR' => ':', 'VALUESEPARATOR' => ':')};
  }

  my %userParams;

  while (my ($username, $paramsPtr) = each %passwd) {
    if (defined($ignoredUsers{$username})) {
      next;
    }
    my @params = @{$paramsPtr};
    my $fullname = $params[3]; #see the 'passwd' format reference
    my $id = $params[1]; #see the 'passwd' format reference
    my $password;

    if (%shadow && defined($shadow{$username})) {
      my @shadowParams = @{$shadow{$username}};
      $password = $shadowParams[0]; #see the 'shadow' format reference
    }

    $userParams{$username} = { 'fullname' => $fullname, 'password' => $password, 'id' => $id };
  }

  return \%userParams;
}

sub _getProtectedDirectoryInfo {
  my ($htaccess, $usergroups, $users) = @_;
  my %htgroups = %{$usergroups};
  my %htusers = %{$users};
  my %pdirusers;
  my %result;

  open HTACCESS, $htaccess;
  binmode HTACCESS;
  while (<HTACCESS>) {
    if (/require\s+group\s+(.*)$/) {
      my @groups = split /\s+/, $1;
      my $group;
      foreach $group (@groups) {
        if (defined($htgroups{$group})) {
          my $htuser;
          foreach $htuser (@{$htgroups{$group}}) {
            if (defined($htusers{$htuser})) {
              $pdirusers{$htuser} = $htusers{$htuser};
            } else {
              $pdirusers{$htuser} = '';
            }
          }
        }
      }
    } elsif (/AuthName [\"]?([^\"]*)[\"]?$/) {
      $result{'title'} = $1;
    }
  }
  close HTACCESS;

  if (%pdirusers) {
    $result{'users'} = \%pdirusers;
    return \%result;
  }

  # at this point, it's a FrontPage's protected directory and to warn about it is nonsense;
  # instead, will just return undef and not include such protected directory in dump.
}

sub _getRelativeDirectory {
  my ($fullPath, $prefix, $suffix) = @_;

  if ($fullPath =~ /^\Q$prefix\E(.*)$suffix$/) {
        my $infix = $1;
        if ($infix =~ /^(.*)\/$/ and $infix ne '/') {
          return $1;
        }
        return $infix;
  }

  Logging::warning("Unable to get relative directory from path $fullPath given prefix $prefix and suffix $suffix");
}

sub _getProtectedDirectoriesOfAKind {
  my ($domainName, $htgroups, $htusers, $rootPath, $isCgi) = @_;

  my $htaccessList;
  my @result;
  my $find = AgentConfig::findBin();

  my $searchedFile = '.htaccess';
  open($htaccessList, "$find '$rootPath' -name $searchedFile -type f |");
  binmode $htaccessList;
  while (<$htaccessList>) {
    my $fullPath = $_;
    my $pdirInfo_ptr = _getProtectedDirectoryInfo($fullPath, $htgroups, $htusers) or next;
    $pdirInfo_ptr->{'cgi'} = $isCgi;
    $pdirInfo_ptr->{'name'} = _getRelativeDirectory($fullPath, $rootPath, $searchedFile);
    push @result, $pdirInfo_ptr;
  }
  close $htaccessList;

  return \@result;
}

sub getProtectedDirectories {
  my ( $domainName ) = @_;

  my %htusers;
  my %htgroups;

  my $htpasswdPath = "/home/virtual/$domainName/var/www/.htpasswd";
  my $htgroupPath = "/home/virtual/$domainName/var/www/.htgroup";
  if (-e $htpasswdPath) {
    my $htpasswdReader = Parser::makeSafeFileParser("/home/virtual/$domainName/var/www/.htpasswd");
    if (defined($htpasswdReader)) {
          %htusers = %{$htpasswdReader->{'PARSE'}->('KEYSEPARATOR'  => ':')};
    }
  }

  if (-e $htgroupPath) {
    my $htgroupReader = Parser::makeSafeFileParser("/home/virtual/$domainName/var/www/.htgroup");
    if (defined($htgroupReader)) {
      %htgroups = %{$htgroupReader->{'PARSE'}->('KEYSEPARATOR' => ':')};

      my $group;
      foreach $group (keys %htgroups) {
        my @members = split(/\s+/, $htgroups{$group});
        $htgroups{$group} = \@members;
      }
    }
  }

  # htmlPDirs and cgiPDirs each have the following structure:
  # [
  #   { 'name' => $directory,
  #     'title' => $AuthName from .htaccess file,
  #     'cgi' => whether it is cgi directory or regular html directory,
  #     'users' => {
  #       $username => $password,
  #       ...
  #     }
  #   }
  # ]
  my $wwwRoot = "/home/virtual/$domainName/var/www";
  my @htmlPDirs = @{_getProtectedDirectoriesOfAKind($domainName, \%htgroups, \%htusers, $wwwRoot . "/cgi-bin", "true")};
  my @cgiPDirs = @{_getProtectedDirectoriesOfAKind($domainName, \%htgroups, \%htusers, $wwwRoot . '/html', "false")};

  return @cgiPDirs, @htmlPDirs;
}

sub getLogrotateSettings {
  my ($domainName) = @_;
  my %settings;

  open(LOGROT, "/home/virtual/$domainName/etc/logrotate.conf") or return;
  binmode LOGROT;

  my $multiplier = 1;
  $settings{'period'} = 'monthly';

  while (<LOGROT>) {
        if (/^rotate\s+(\d+)*$/) {
          $settings{'number-of-rotated-files'} = $1;
        } elsif (/^size\s+(\d+)(.?)$/) {
          if ($2 eq 'M') {
                $multiplier = 1024*1024;
          } elsif ($2 eq 'k') {
                $multiplier = 1024;
          }
          $settings{'size'} = $1 * $multiplier;
        } elsif (/^(daily|weekly|monthly)$/) {
          $settings{'period'} = $1;
        }
  }

  unless (defined $settings{'number-of-rotated-files'}) {
        Logging::warning("Log rotation configuration is not complete.");
        return;
  }
  return %settings;
}

sub getWebUsers {
  my ($domainName) = @_;

  my %result;

  my $subdomainList = getListOfSubdomains($domainName);
  my %excludedUsers;
  for my $subdomain (values %$subdomainList) {
    my $documentRoot = $subdomain->{'document_root'};
    if ($documentRoot =~ /\/home\/([^\/]+)\/public_html/) {
      $excludedUsers{$1} = 1;
    }
  }
  my $usersList = getSiteUsersInfoMap($domainName);
  while (my ($userName, $userParams) = each(%$usersList)) {
    $result{$userName} = $userParams->{'password'} unless (defined $excludedUsers{$userName});
  }

  return \%result;
}

sub getMailAliases {
  my ($domainName, $ignoreAliases) = @_;

  my %ignoreAliasesHash;

  if (defined($ignoreAliases)) {
    %ignoreAliasesHash = map { $_ => 1  } @{$ignoreAliases};
  }
  
  my $configReader = Parser::makeFileParser(getFilesystemByDomainName($domainName) . '/etc/aliases');
  my %aliases = %{$configReader->{'PARSE'}->(
    'KEYSEPARATOR' => ':',
    'COMMENT' => '#',
    'VALUESEPARATOR' => ','
  )};

  my ($alias, $ptrActions);
  my %result;

  while (($alias, $ptrActions) = each %aliases) {

    # ignore default majordomo alias
    if ($alias eq 'majordomo') {
      next;
    }

    if (defined($ignoreAliasesHash{$alias})) {
      next;
    }

    my @actions = @{$ptrActions};
    my (@redirects, @localDelivery, @responders);

    @actions = map { trim($_); } @actions;

    foreach (@actions) {
      if (/^"\|responder\.sh\s+\S+\s+(\S+)"$/) {
        open INPUT, getFilesystemByDomainName($domainName) . $1;
        binmode INPUT;
        my $subject = <INPUT>;
        chomp $subject;
        my $body;
        while (<INPUT>) {
          $body .= $_;
        }
        close(INPUT);

        push @responders, { 'subject' => $subject, 'body' => $body };
      } elsif (/^([\w\.-]+)$/) {
        push @localDelivery, $1;
      } elsif (/^([\w@\.-]+)$/) {
        push @redirects, $1;
      } elsif (/^"\|\/usr\/lib\/majordomo\/wrapper$"/) {
        Logging::debug("Majordomo action ignored: $_");
      } elsif (/^:include:/) {
        Logging::debug("Majordomo action ignored: $_");
      } else {
        Logging::debug("Unknown action detected: $_");
      }
    }

    $result{$alias} = {
      'responders' => \@responders,
      'localDelivery' => \@localDelivery,
      'redirects' => \@redirects
    };
  }

  return \%result;
}

sub trim {
  my ($string) = @_; 
  $string =~ s/^\s+//;
  $string =~ s/\s+$//;
  return $string;
}

sub getMaillists {
  my ($domainName) = @_;
  my $maillistsFiles = _getMaillistsFiles($domainName);
  my %result;

  while (my ($maillistName, $files) = each %{$maillistsFiles}) {
    $result{$maillistName} = {
      'adminPassword' => _getMaillistAdminPassword($files->{'configFile'}),
      'members' => _getMaillistMembers($files->{'membersListFile'}),
      'owner' => _getMaillistOwner($domainName, $maillistName)
    };
  }

  return \%result; 
}

sub getMaillistsAliases {
  my ($domainName) = @_;
  my @maillists = keys(%{_getMaillistsFiles($domainName)});
  my @aliases;

  for my $maillist (@maillists) {
    push @aliases, (
      $maillist,
      "owner-$maillist",
      "$maillist-owner",
      "$maillist-request",
      "$maillist-outgoing",
      "$maillist-list",
      "$maillist-approval"
    );
  }

  return \@aliases;
}

# returns reference to a hash with:
# - key: name of a maillist
# - value: reference to a hash with keys 'configFile' and 'userListFile' and values - full path to corresponding maillist files
sub _getMaillistsFiles {
  my ($domainName) = @_;

  my $listsDir = getFilesystemByDomainName($domainName) . '/var/lib/majordomo/lists';

  unless (-d $listsDir) {
    Logging::debug("Domain $domainName doesn't contain maillists directory");
    return {};
  }
  
  my %lists;

  open(LISTCONFIGS, AgentConfig::findBin() . " $listsDir -maxdepth 1 -type f -name '*.config' |");
  binmode LISTCONFIGS;

  while(<LISTCONFIGS>) {
    chomp;
    
    my $configFile = $_;
    my ($listName, $membersListFile);
    
    if (/([^\/]+)\.config/) {
      $listName = $1;
    } else {
      Logging::warning("Failed to get maillist name from config file name '$_', ignoring maillist config");
      next;
    }
    
    if (/^(.+)\.config$/) {
      $membersListFile = $1;
    } else {
      Logging::warning("Failed to get maillist members file name from config file name '$_', ignoring maillist config");
      next;
    }

    $lists{$listName} = {
      'configFile' => $configFile,
      'membersListFile' => $membersListFile
    };
	}

	close LISTCONFIGS;

  return \%lists;
}

sub _getMaillistAdminPassword {
  my ($configFile) = @_;

  open(LISTCONFIG, $configFile);
  binmode LISTCONFIG;

  while (<LISTCONFIG>) {
    if (/admin_passwd\s=\s(.*)$/) {
      close(LISTCONFIG);
      return $1;
    }
  }

  close LISTCONFIG;
  return undef;
}

sub _getMaillistMembers {
  my ($membersListFile) = @_;

  my @result;

	open(LISTMEMBERS, $membersListFile);
	binmode LISTMEMBERS;

	while (<LISTMEMBERS>) {
		chomp;
    push @result, $_;
  }

  close LISTMEMBERS;

  return \@result;
}

sub _getMaillistOwner {
  my ($domainName, $maillistName) = @_;

  my $aliases = getMailAliases($domainName);

  while (my ($alias, $aliasInfo) = each %{$aliases}) {
    if ($alias eq $maillistName . '-owner') {
      if (@{$aliasInfo->{'localDelivery'}}) {
        return $aliasInfo->{'localDelivery'}->[0] . '@' . $domainName;
      } elsif (@{$aliasInfo->{'redirects'}}) {
        return $aliasInfo->{'redirects'}->[0];
      }
    }
  }

  return undef;
}

sub getProcmailrcAction {
    my ($procmailrcFile) = @_;
    my $action = "mark";  # default value

    open(INPUTPROCCFG, $procmailrcFile) || return $action;
    while (<INPUTPROCCFG>) {
        chomp;
        if (/^SPAM_FOLDER="(.+)"$/) {
            if ($1 eq "\$HOME/mail/Spam") {
                $action = "move";
            } elsif ($1 eq "/dev/null") {
                $action = "delete";
            }
            last;
        }
    }
    close INPUTPROCCFG;

    return $action;
}

sub getDnsZoneProperties {
  my ( $domainName ) = @_;
  my @records = ();
  my $dnsfile = "/var/named/db.".$domainName;

  # zone file is nicely canonicalized by named-checkzone -D
  open(DNS, "named-checkzone -jD $domainName $dnsfile |");

  while (<DNS>) {
    last if /^OK$/;

    # if named-checkzone result is invalid, bail out
    if (/^zone $domainName\/IN: (.*)/) {
      unless ($1 =~ /^loaded serial \d+( \(signed\))?$/) {
        Logging::warning("Zone file $dnsfile appears to be invalid: $_");
        close(DNS);
        return;
      }
    }

    # check if we found a resource record, "www 86400 IN A 10.0.2.15" for example 
    if (/^\S+\s+\d+\s+IN\s+(SOA|NS|A|CNAME|MX|PTR|TXT|SRV)\s+/o) {
      chomp;
      my @rec = split(/\s+/, $_);
      push @records, \@rec;
    }
  }

  close(DNS) || Logging::warning("$dnsfile has errors according to results of named-checkzone.");
  return @records;
}

1;

