# Copyright 1999-2014. Parallels IP Holdings GmbH. All Rights Reserved.
package Dumper;

use strict;
use warnings;

use AgentConfig;
use Logging;
use Parser;
use SimpleDbConnect;
use File::Spec;
use EnsimGuidGenerator;
use DumperUtils;

use constant USERNAME_MANGLE_MAX_ITERATIONS => 10000;

my $_db;
my $_clientDb;
my $_mysqlRootPassword;

sub _getDb() {
  unless (defined($_db)) {
    $_db = SimpleDbConnect->new('Pg', undef, undef, 'appldb', undef);
  }

  return $_db;
}

sub _getClientDb() {
  unless (defined($_clientDb)) {
    $_clientDb = SimpleDbConnect->new('mysql', getDbAdminCredentials(), 'mysql', undef);
  }

  return $_clientDb;
}

# @return list (<username> <password>)
sub getDbAdminCredentials {
  $_mysqlRootPassword ||= _getMysqlPassword(); 
  return ('root', $_mysqlRootPassword);
}

# This function uses internal PPCPL library for reading MySQL root password
sub _getMysqlPassword {
  my $python = AgentConfig::pythonBin();
  my $script =<< "EOS"
import sys
sys.path.append('/usr/lib/opcenter/mysql')
from mysqlbe import read_mysqlpass

sys.stdout.write(read_mysqlpass())
EOS
;
	unless (open(PY, "$python -c \"$script\" |")) {
		return '';
	}
	binmode PY;

	my $passwd = <PY>;
	close(PY);
	chomp $passwd;

	return $passwd;
}

sub _readConfigFile {
  my ( $siteId, $fileName ) = @_;

  my $configReader =
    Parser::makeSafeFileParser(getSiteinfoBySiteId($siteId) . "/current/" . $fileName);

  if ( defined $configReader ) {
    return $configReader->{'PARSE'}->(
      'KEYSEPARATOR' => '=',
      'COMMENT'      => '\[DEFAULT\]'
    );
  }
  else {
    return { "enabled" => 0 };
  }
}

sub getSiteinfoBySiteId {
  my ($siteId) = @_;

  return "/home/virtual/site$siteId/info";
}

sub getFilesystemByDomainName {
  my ($domainName) = @_;

  return "/home/virtual/$domainName";
}

sub getListOfSubdomains {
  my ($domainName) = @_;

  my $configListOfSubdomains;
  my $currentSubdomain;
  my $subcount=0;

  my $siteId = getSiteIdByDomain($domainName);

  # Parser::makeSafeFileParser("/etc/virtualhosting/subdomain/site" . $siteId) not supported
  # syntax of this config. So, will read config by hand...

  my ($subdomainsConfigFile);
  open($subdomainsConfigFile, "/etc/virtualhosting/subdomain/site" . $siteId) || return undef;
  while (<$subdomainsConfigFile>) {
    chomp;
    if (/^\[(.*)\]$/) {
        $currentSubdomain = $1;
        $subcount++ if ($currentSubdomain ne "DEFAULT");
        next;
    }
    next if (!defined($currentSubdomain));

    next if (/^#/ || /^$/);
    next if (! m/^(.*?)[ \t]*=[ \t]*(.*)$/ );
    $configListOfSubdomains->{"$currentSubdomain"}{"$1"} = "$2";

  }
  close($subdomainsConfigFile);

  if (defined($configListOfSubdomains->{"DEFAULT"})) {
    return undef unless $subcount;      # check if define only DEFAULT subdomain (and not have other subdomain(s))

    # fill now default values in all subdomain(s)
    foreach my $subdomainName ( keys %{$configListOfSubdomains} ) {
        foreach my $values_subdomain (keys %{$configListOfSubdomains->{"DEFAULT"}}) {
            if (!defined($configListOfSubdomains->{"$subdomainName"}{$values_subdomain})) {
                $configListOfSubdomains->{"$subdomainName"}{$values_subdomain}
                    = $configListOfSubdomains->{DEFAULT}{$values_subdomain};
            }
        }
    }
    delete($configListOfSubdomains->{DEFAULT});
  }

  return $configListOfSubdomains;
}


sub getResellers {
  Logging::trace('Getting resellers list...');

  # TODO ddidenkov@ revise WHERE clause
  #  Compare with old clause (taken from EnsimX.pl):
  # " WHERE (reseller_info.reseller_id <> 0) OR (site_id IS NOT NULL)";
  my $sql =
      "SELECT DISTINCT reseller_info.username"
    . " FROM reseller_info LEFT JOIN reseller"
    . " ON reseller_info.reseller_id = reseller.reseller_id"
    . " WHERE (reseller_info.reseller_id <> 0)";
  my $wrapDbh = _getDb()->getDbh();
  my @resellers;
  if ( $wrapDbh->{'EXECUTE'}->($sql) ) {
    my $ptrRow;
    while ( $ptrRow = $wrapDbh->{'FETCHROW'}->() ) {
      my @resellerInfo = @$ptrRow;
      push @resellers, $resellerInfo[0];
    }
  }

  $wrapDbh->{'FINISH'}->();
  Logging::debug( "Found " . scalar(@resellers) . " resellers: @resellers" );
  return @resellers;
}

sub getResellerNameForClientId($) {
  my ($clientId) = @_;

  my $resellerName = _getDb()->queryOneValue(
      "SELECT reseller_info.username "
    . "FROM reseller, reseller_info "
    . "WHERE reseller.site_id = '$clientId' and reseller.reseller_id = reseller_info.reseller_id"
  );

  return $resellerName; 
}

sub getVendorNameForClientId($) {
  my ($clientId) = @_;

  my $resellerName = getResellerNameForClientId($clientId);

  if (!defined($resellerName) or $resellerName eq '') {
    return 'admin';
  } else {
    return mangleResellerName($resellerName);
  }
}

sub getVendorGuidForClientId($) {
  my ($clientId) = @_;

  my $resellerName = getResellerNameForClientId($clientId);

  if (!defined($resellerName) or $resellerName eq '') {
    return EnsimGuidGenerator::getAdminGuid();
  } else {
    return EnsimGuidGenerator::getResellerGuid($resellerName);
  }
}

sub getClientIdsForReseller {

  # Could be 'undef' for admin
  my ($reseller) = @_;

  Logging::trace(
    "Getting client identifiers for '" . ( defined($reseller) ? $reseller : 'undef' ) . "' reseller..." );

 # temporary "resellerId by username" resolving to avoid current DumpComposer's cPanel-ish behaviour
  my $resellerId;
  my $wrapDbh = _getDb()->getDbh();
  if ( defined $reseller ) {
    my $sql = "SELECT reseller_id FROM reseller_info WHERE username = '$reseller'";
    if ( $wrapDbh->{'EXECUTE'}->($sql) ) {
      $resellerId = $wrapDbh->{'FETCHROW'}->()->[0];
      $wrapDbh->{'FINISH'}->();
    }
    else {
      Logging::warning("Unable to get reseller id by its username '$reseller'!");

      # Couldn't find any sites for unknown reseller -> return
      return ();
    }
  }
  else {

    # PPCP uses resellerId = 0 to identify admin sites
    $resellerId = 0;
  }

  my $sql =
      "SELECT domain FROM siteinfo"
    . " NATURAL JOIN reseller"
    . " WHERE (reseller.reseller_id = '$resellerId')";
  my @siteAdmins;
  if ( $wrapDbh->{'EXECUTE'}->($sql) ) {
    my $ptrRow;
    while ( $ptrRow = $wrapDbh->{'FETCHROW'}->() ) {
      my @siteInfo = @$ptrRow;
      push @siteAdmins, $siteInfo[0];
    }
  }

  $wrapDbh->{'FINISH'}->();

  Logging::debug( "Found " . scalar(@siteAdmins) . " site admins for '$resellerId' reseller: @siteAdmins" );

  return @siteAdmins;
}

sub getResellerGeneralInfo {
  my $reseller = shift;

  Logging::trace("Getting general info about '$reseller' reseller...");

  my $sql =
      "SELECT reseller_id, username, email, password, fullname, enabled"
    . " FROM reseller_info WHERE (username = '$reseller')";

  my $wrapDbh = _getDb()->getDbh();
  my $ptrHash;
  if ( $wrapDbh->{'EXECUTE'}->($sql) ) {
    $ptrHash = $wrapDbh->{'FETCHHASH'}->();
  }
  $wrapDbh->{'FINISH'}->();
  if ( defined($ptrHash) && ref($ptrHash) =~ /HASH/ ) {
    return $ptrHash;
  }
  else {
    return;
  }
}

sub _getResellerIdByName {
  my ( $reseller ) = @_;
  return _getDb()->queryOneValue(
    "SELECT reseller_id FROM reseller_info WHERE username = '$reseller'",
    'reseller_id'
  );
}

sub getResellerLimits {
  my ( $reseller ) = @_;

  my $limits = {
    # units field of reseller_diskquota is ignored because 
    # it is always set to 'MB' however total_quota is in bytes
    # the same applies for other similar values
    'diskquota' => { 
      'table' => 'reseller_diskquota', 
      'enabled_column' => 'enabled_diskquota', 
      'value_column' => 'total_quota'
    },
    'users' => {
      'table' => 'reseller_users',
      'enabled_column' => 'enabled_users',
      'value_column' => 'total_maxusers'
    },
    'bandwidth' => {
      'table' => 'reseller_bandwidth',
      'enabled_column' => 'enabled_bandwidth',
      'value_column' => 'total_threshold'
    },
    'ip_based_sites' => {
      'table' => 'reseller_ipinfo',
      'enabled_column' => 'enabled_ipinfo_ip',
      'value_column' => 'total_ipbased'
    },
    'name_based_sites' => {
      'table' => 'reseller_ipinfo',
      'enabled_column' => 'enabled_ipinfo_nb',
      'value_column' => 'total_namebased' 
    }
  };
  
  return _getObjectLimits($limits, 'reseller_info', 'reseller_id', _getResellerIdByName($reseller));
}

sub getSiteLimits {
  my ( $siteId ) = @_;

  my $limits = {
    'diskquota' => { 
      'table' => 'diskquota', 
      'enabled_column' => 'enabled', 
      'value_column' => 'quota'
    },
    'users' => {
      'table' => 'users',
      'enabled_column' => 'enabled',
      'value_column' => 'maxusers'
    }
  };

  my %limits = _getObjectLimits($limits, 'siteinfo', 'site_id', $siteId);

  # bandwidth has no special limit enabled column,
  # so we have to get it in quite another way
  $limits{'bandwidth'} = _getSiteBandwidthLimit($siteId);

  return %limits;
}

sub _getSiteBandwidthLimit {
  my ( $siteId ) = @_; 

  my $result = _getDb()->queryOneValue(
    "SELECT threshold FROM bandwidth WHERE site_id = '$siteId'"
  );

   # for a bandwidth threshold '0' is a special value for unlimited
  if ($result eq '0') {
    return undef;
  } else {
    return $result;
  }
}

sub _getObjectLimits {
  my ( $limits, $mainTable, $tableKeyName, $tableKeyValue ) = @_;

  my @columnExprs;
  my @joinExprs;

  foreach my $limitName ( keys %{$limits}) {
    my $limit = $limits->{$limitName};

    my $table = $limit->{'table'};
    my $enabledColumn = $limit->{'enabled_column'};
    my $valueColumn = $limit->{'value_column'};

    my $sqlTableName = "t_$limitName";

    push(@columnExprs, "$sqlTableName.$enabledColumn as ${limitName}_enabled");
    push(@columnExprs, "$sqlTableName.$valueColumn as ${limitName}_value");

    push(@joinExprs, "INNER JOIN $table $sqlTableName ON $sqlTableName.$tableKeyName = $mainTable.$tableKeyName");
  }

  my $query = 
    'SELECT ' . join(', ', @columnExprs) . 
    " FROM $mainTable " . join(' ', @joinExprs) . 
    " WHERE $mainTable.$tableKeyName = '$tableKeyValue'";

  my $queryResult = _getDb()->queryOneRow($query);
 
  my %result;

  foreach my $limitName ( keys %{$limits} ) {
    if ( $queryResult->{$limitName . '_enabled'} == '1' ) {
      $result{$limitName} = $queryResult->{$limitName . '_value'};
    } else {
      $result{$limitName} = undef;
    }
  }

  return %result;
}

sub canResellerManageDNS {
  my ( $reseller ) = @_;

  return _getDb()->queryOneValue(
    "SELECT enabled_zone_mgmt FROM reseller_bind" .
    " INNER JOIN reseller_info ON reseller_bind.reseller_id = reseller_info.reseller_id" .
    " WHERE reseller_info.username = '$reseller'"
  );
}

sub getSpecificInfoForSite {
  my ($siteId, $infoFileName, $infoName, %defaultSettings) = @_;

  $infoName = $infoFileName unless $infoName;
  Logging::trace("Getting '$infoName' for '$siteId' site...");
  my %info = %{ _readConfigFile( $siteId, $infoFileName ) };
  my @infoTextDump = map { "$_ => $info{$_}\n" } keys %info;
  Logging::trace("$infoName for '$siteId' site is \n @infoTextDump");
  while (my($setting, $value) = each %defaultSettings) {
    unless (exists $info{$setting} ) {
      $info{$setting} = $value;
      Logging::warning("Cannot find the parameter '$setting' in config file '$infoFileName'. Using the default value '$value'");
    }
  }
  return %info;
}

sub getSiteIdByDomain {
  my ( $domain ) = @_;

  return _getDb()->queryOneValue(
    "SELECT site_id FROM siteinfo WHERE domain = '$domain'"
  );
}

sub getSiteContactEmailByDomain {
  my ( $domain ) = @_;

  return _getDb()->queryOneValue(
    "SELECT email FROM siteinfo WHERE domain = '$domain'"
  );
}

sub getSiteAdminLoginByDomain {
  my ( $domain ) = @_;

  return _getDb()->queryOneValue(
    "SELECT admin_user FROM siteinfo WHERE domain = '$domain'"
  );
}

sub getIpInfoForSite {
  my ( $siteId ) = @_;

  return getSpecificInfoForSite($siteId, 'ipinfo', 'IP info');
}

sub getBindInfoForSite {
  my ( $siteId ) = @_;
  my %defaultSettings = (
     'dns_log'      => 0,
     'zone_mgmt'    => 0,
     'enabled'      => 0,
  );
  return getSpecificInfoForSite($siteId, 'bind', 'DNS info', %defaultSettings);
}

sub isDisabledSite {
  my $siteId = shift;

  Logging::trace("Getting disable status for '$siteId' site...");
  my $isDisabled = 0;
  if (-f getSiteinfoBySiteId($siteId) . "/disabled") {
    $isDisabled = 1;
  }
  Logging::debug( "'$siteId' site is " . ( ( $isDisabled == 0 ) ? 'enabled' : 'disabled' ) . '.' );
  return $isDisabled;
}

sub getAgentStatus {
  my $root = XmlNode->new('agent-status');

  unless ( checkPanelVersion() && defined AgentConfig::iconvBin() ) {
    my $item;
    if ( defined AgentConfig::iconvBin() ) {
      $item = XmlNode->new('wrong-platform');
      $item->setText('');
    }
    else {
      $item = XmlNode->new('wrong-platform');
      $item->setText('no iconv found on the source host');
    }

    $root->addChild($item);
  }

  return $root;
}

sub getPPCPVersion {

# TODO ddidenkov@ revise this check after PPCP source code analysis (the same thing is for scoup.pl)
# TODO ddidenkov@ discuss the found ways with current PPCP developers
# I see several ways in the PPCP's source code:
# 1. cat /usr/lib/opcenter/VERSION (returns '10.3.4-rhel.5ES.9' in my case). Ref: epl/eplapi/implementors/server/server.py, L35, getEPLVersionInfo
# 2. /bin/rpm -qf --qf %{VERSION}-%{RELEASE} /etc/rc.d/init.d/epld (returns the same '10.3.4-rhel.5ES.9'). Ref: epl/eplapi/licensing/licenselib.py, L115, get_version
# 3. /usr/bin/getapplversion.old . It reads the same /usr/lib/opcenter/VERSION file
# 4. python -c "import sys; sys.path.append('/usr/lib/ensim'); import eplapi.implementors.server.server; print eplapi.implementors.server.server.epl_version_release".
#    This way is used by web-interface to show CP version on the dashboard/Server Administrator page.
#    It reads some string generated during the release build.
  my $main_version_file_name = '/usr/lib/opcenter/VERSION';
  if ( -e $main_version_file_name ) {
    my $version = `cat $main_version_file_name`;
    chomp $version;
    if ( $version =~ /(\d+\.\d+(\.\d+)?)-.*/ ) {
      Logging::debug("Parsed '$1' product version number from '$version' string.");
      return $1;
    }
    else {
      Logging::debug("Couldn't get the product version number from '$version'");
    }
  }
  else {
    Logging::debug("The version file '$main_version_file_name' doesn't exist.");
  }

  {
    my $version =
`python -c "import sys; sys.path.append('/usr/lib/ensim'); import eplapi.implementors.server.server; print eplapi.implementors.server.server.epl_version_release"`;
    chomp $version;
    if ( $version =~ /(\d+\.\d+(\.\d+)?)-.*/ ) {
      Logging::debug("Parsed '$1' product version number from '$version' string.");
      return $1;
    }
    else {
      Logging::debug("Couldn't get the product version number from '$version'");
    }
  }

  my $rpm     = AgentConfig::rpmBin();
  my $version = `$rpm -qf --qf \%{VERSION} /etc/rc.d/init.d/epld 2>/dev/null`;
  if ( $version =~ /^(\d+\.\d+(\.\d+)?)$/ ) {
    Logging::debug("Parsed '$1' product version number from '$version' string.");
    return $1;
  }
  else {
    Logging::debug("Couldn't get the product version number from '$version'");
  }

  # there are older ways to determine the version below
  my @rpms_list = ( 'webppliance-version', 'webppliance-ipinfo' );
  for my $rpm_name (@rpms_list) {
    my $version = `$rpm -q $rpm_name 2>/dev/null`;
    if ( $version =~ /^$rpm_name-([^-]+)-[^-]+$/ ) {
      Logging::debug("Parsed '$1' product version number from '$version' string.");
      return $1;
    }
    else {
      Logging::debug("Couldn't get the product version number from '$version'");
    }
  }

  my @version_files_list = ( '/usr/bin/getapplversion', '/usr/bin/getapplversion.old' );
  for my $version_file_name (@version_files_list) {
    if ( -e $version_file_name ) {
      my $version = `$version_file_name`;
      chomp $version;
      if ( $version =~ /(\d+\.\d+(\.\d+)?)-.*/ ) {
        Logging::debug("Parsed '$1' product version number from '$version' string.");
        return $1;
      }
      else {
        Logging::debug("Couldn't get the product version number from '$version'");
      }
    }
    else {
      Logging::debug("The version file '$version_file_name' doesn't exist.");
    }
  }
  Logging::debug("Couldn't detect a PPCP.");
  return;
}

sub checkPanelVersion {
  my $version = getPPCPVersion();

  if ( $version =~ /^(\d+)\.(\d+)/ ) {
    Logging::debug(
      "The product version number '$version'" . " contains major($1) and minor($2) parts." );

    # Check version major and minor fields
    if ( $1 eq '10' && $2 eq '3' ) {
      return $version;
    }
    else {
      Logging::info("The PPCP version '$version' is not supported.");
      return;
    }
  }
  Logging::info("Couldn't determine major and minor version numbers from '$version'.");
  return;
}


my %ignoredUsers = (
  'root' => '1',
  'bin' => '1',
  'daemon' => '1',
  'adm' => '1',
  'lp' => '1',
  'sync' => '1',
  'shutdown' => '1',
  'halt' => '1',
  'mail' => '1',
  'uucp' => '1',
  'operator' => '1',
  'games' => '1',
  'gopher' => '1',
  'ftp' => '1',
  'news' => '1',
  'mysql' => '1',
  'nobody' => '1',
  'zope' => '1',
  'majordomo' => '1',
  'tomcat4' => '1',
  'apache' => '1',
  'ensimrootmail' => '1',
  'sshd' => '1',
  'smmsp' => '1',
);

sub getSiteUsersInfoMap {
  my ($domainName) = @_;

  my $passwdReader = Parser::makeSafeFileParser(getFilesystemByDomainName($domainName) . "/etc/passwd");
  my $shadowReader = Parser::makeSafeFileParser(getFilesystemByDomainName($domainName) . "/etc/shadow");

  my (%passwd, %shadow);

  if (defined($passwdReader)) {
    %passwd = %{$passwdReader->{'PARSE'}->('KEYSEPARATOR' => ':', 'VALUESEPARATOR' => ':')};
  } else {
    return undef;
  }

  if (defined($shadowReader)) {
    %shadow = %{$shadowReader->{'PARSE'}->('KEYSEPARATOR' => ':', 'VALUESEPARATOR' => ':')};
  }

  my %userParams;

  while (my ($username, $paramsPtr) = each %passwd) {
    if (defined($ignoredUsers{$username})) {
      next;
    }
    my @params = @{$paramsPtr};
    my $fullname = $params[3]; #see the 'passwd' format reference
    my $id = $params[1]; #see the 'passwd' format reference
    my $password;

    if (%shadow && defined($shadow{$username})) {
      my @shadowParams = @{$shadow{$username}};
      $password = $shadowParams[0]; #see the 'shadow' format reference
    }

    $userParams{$username} = { 'fullname' => $fullname, 'password' => $password, 'id' => $id };
  }

  return \%userParams;
}

sub _getProtectedDirectoryInfo {
  my ($htaccess, $usergroups, $users) = @_;

  my %htgroups = %{$usergroups};
  my %htusers = %{$users};
  my %pdirusers;
  my %result;

  open HTACCESS, $htaccess;
  binmode HTACCESS;
  while (<HTACCESS>) {
    if (/require\s+group\s+(.*)$/) {
      my @groups = split /\s+/, $1;
      my $group;
      foreach $group (@groups) {
        if (defined($htgroups{$group})) {
          my $htuser;
          foreach $htuser (@{$htgroups{$group}}) {
            if (defined($htusers{$htuser})) {
              $pdirusers{$htuser} = $htusers{$htuser};
            } else {
              $pdirusers{$htuser} = '';
            }
          }
        }
      }
    } elsif (/AuthName [\"]?([^\"]*)[\"]?$/) {
      $result{'title'} = $1;
    }
  }
  close HTACCESS;

  if (%pdirusers) {
    $result{'users'} = \%pdirusers;
    return \%result;
  }

  # at this point, it's a FrontPage's protected directory and to warn about it is nonsense;
  # instead, will just return undef and not include such protected directory in dump.
}

sub _getRelativeDirectory {
  my ($fullPath, $prefix, $suffix) = @_;

  if ($fullPath =~ /^\Q$prefix\E(.*)$suffix$/) {
        my $infix = $1;
        if ($infix =~ /^(.*)\/$/ and $infix ne '/') {
          return $1;
        }
        return $infix;
  }

  Logging::warning("Unable to get relative directory from path $fullPath given prefix $prefix and suffix $suffix");
}

sub _getProtectedDirectoriesOfAKind {
  my ($domainName, $htgroups, $htusers, $rootPath, $isCgi) = @_;

  my $htaccessList;
  my @result;
  my $find = AgentConfig::findBin();

  my $searchedFile = '.htaccess';
  open($htaccessList, "$find '$rootPath' -name $searchedFile -type f |");
  binmode $htaccessList;
  while (<$htaccessList>) {
    my $fullPath = $_;
    my $pdirInfo_ptr = _getProtectedDirectoryInfo($fullPath, $htgroups, $htusers) or next;
    $pdirInfo_ptr->{'cgi'} = $isCgi;
    $pdirInfo_ptr->{'name'} = _getRelativeDirectory($fullPath, $rootPath, $searchedFile);
    push @result, $pdirInfo_ptr;
  }
  close $htaccessList;

  return \@result;
}

sub getProtectedDirectories {
  my ( $domainName ) = @_;

  my %htusers;
  my %htgroups;

  my $htpasswdPath = "/home/virtual/$domainName/var/www/.htpasswd";
  my $htgroupPath = "/home/virtual/$domainName/var/www/.htgroup";
  if (-e $htpasswdPath) {
    my $htpasswdReader = Parser::makeSafeFileParser("/home/virtual/$domainName/var/www/.htpasswd");
    if (defined($htpasswdReader)) {
          %htusers = %{$htpasswdReader->{'PARSE'}->('KEYSEPARATOR'  => ':')};
    }
  }

  if (-e $htgroupPath) {
    my $htgroupReader = Parser::makeSafeFileParser("/home/virtual/$domainName/var/www/.htgroup");
    if (defined($htgroupReader)) {
      %htgroups = %{$htgroupReader->{'PARSE'}->('KEYSEPARATOR' => ':')};

      my $group;
      foreach $group (keys %htgroups) {
        my @members = split(/\s+/, $htgroups{$group});
        $htgroups{$group} = \@members;
      }
    }
  }

  # htmlPDirs and cgiPDirs each have the following structure:
  # [
  #   { 'name' => $directory,
  #     'title' => $AuthName from .htaccess file,
  #     'cgi' => whether it is cgi directory or regular html directory,
  #     'users' => {
  #       $username => $password,
  #       ...
  #     }
  #   }
  # ]
  my $wwwRoot = "/home/virtual/$domainName/var/www";
  my @htmlPDirs = @{_getProtectedDirectoriesOfAKind($domainName, \%htgroups, \%htusers, $wwwRoot . "/cgi-bin", "true")};
  my @cgiPDirs = @{_getProtectedDirectoriesOfAKind($domainName, \%htgroups, \%htusers, $wwwRoot . '/html', "false")};

  return @cgiPDirs, @htmlPDirs;
}

sub getLogrotateSettings {
  my ($domainName) = @_;

  my %settings;

  open(LOGROT, "/home/virtual/$domainName/etc/logrotate.conf") or return;
  binmode LOGROT;

  my $multiplier = 1;
  $settings{'period'} = 'monthly';

  while (<LOGROT>) {
        if (/^rotate\s+(\d+)*$/) {
          $settings{'number-of-rotated-files'} = $1;
        } elsif (/^size\s+(\d+)(.?)$/) {
          if ($2 eq 'M') {
                $multiplier = 1024*1024;
          } elsif ($2 eq 'k') {
                $multiplier = 1024;
          }
          $settings{'size'} = $1 * $multiplier;
        } elsif (/^(daily|weekly|monthly)$/) {
          $settings{'period'} = $1;
        }
  }

  unless (defined $settings{'number-of-rotated-files'}) {
        Logging::warning("Log rotation configuration is not complete.");
        return;
  }
  return %settings;
}

sub getWebUsers {
  my ($domainName) = @_;

  my %result;

  my $subdomainList = getListOfSubdomains($domainName);
  my %excludedUsers;
  for my $subdomain (values %$subdomainList) {
    my $documentRoot = $subdomain->{'document_root'};
    if ($documentRoot =~ /\/home\/([^\/]+)\/public_html/) {
      $excludedUsers{$1} = 1;
    }
  }
  my $usersList = getSiteUsersInfoMap($domainName);
  while (my ($userName, $userParams) = each(%$usersList)) {
    $result{$userName} = $userParams->{'password'} unless (defined $excludedUsers{$userName});
  }

  return \%result;
}

sub getMailboxForwarding {
  my ($domainName, $userName) = @_;

  my $detectEmailRegex = qr/^[^\\\|"#\s]/;
  my @forwardAddresses = ();

  my $forwardFilePath = File::Spec->catfile(getFilesystemByDomainName($domainName), 'home', $userName, '.forward');
  if (-e $forwardFilePath) {
    open(my $fh, $forwardFilePath);
    my $firstLine = <$fh>;
    close($fh);
    if (defined($firstLine)) {
      chomp($firstLine);
    
      if ($firstLine =~ $detectEmailRegex) { # non-special line beginning denotes email address
        my @possibleAddressesList = split(/[ ,]+/, $firstLine);
        foreach my $possibleAddress (@possibleAddressesList) {
          if ($possibleAddress =~ $detectEmailRegex) { # no reason to implement full RFC here
            push @forwardAddresses, $possibleAddress;
          }
        }
      }
    }
  }
  return @forwardAddresses;
}

sub getMailAliases {
  my ($domainName, $ignoreAliases) = @_;

  my %ignoreAliasesHash;

  if (defined($ignoreAliases)) {
    %ignoreAliasesHash = map { $_ => 1  } @{$ignoreAliases};
  }
  
  my $configReader = Parser::makeFileParser(getFilesystemByDomainName($domainName) . '/etc/aliases');
  my %aliases = %{$configReader->{'PARSE'}->(
    'KEYSEPARATOR' => ':',
    'COMMENT' => '#',
    'VALUESEPARATOR' => ','
  )};

  my ($alias, $ptrActions);
  my %result;

  while (($alias, $ptrActions) = each %aliases) {

    # ignore default majordomo alias
    if ($alias eq 'majordomo') {
      next;
    }

    if (defined($ignoreAliasesHash{$alias})) {
      next;
    }

    my @actions = @{$ptrActions};
    my (@redirects, @localDelivery, @responders);

    @actions = map { trim($_); } @actions;

    foreach (@actions) {
      if (/^"\|responder\.sh\s+\S+\s+(\S+)"$/) {
        if (open(INPUT, getFilesystemByDomainName($domainName) . $1)) {
          binmode INPUT;
          my $subject = <INPUT>;
          chomp $subject;
          my $body;
          while (<INPUT>) {
            $body .= $_;
          }
          close(INPUT);

          push @responders, { 'subject' => $subject, 'body' => $body };
        } else {
          Logging::info("Responder is enabled for $alias, but the corresponding message can't be read ($!). Responder won't be migrated.");
        }
      } elsif (/^([\w\.-]+)$/) {
        push @localDelivery, $1;
      } elsif (/^([\w@\.-]+)$/) {
        push @redirects, $1;
      } elsif (/^"\|\/usr\/lib\/majordomo\/wrapper$"/) {
        Logging::debug("Majordomo action ignored: $_");
      } elsif (/^:include:/) {
        Logging::debug("Majordomo action ignored: $_");
      } else {
        Logging::debug("Unknown action detected: $_");
      }
    }

    $result{$alias} = {
      'responders' => \@responders,
      'localDelivery' => \@localDelivery,
      'redirects' => \@redirects
    };
  }

  return \%result;
}

sub trim {
  my ($string) = @_; 
  $string =~ s/^\s+//;
  $string =~ s/\s+$//;
  return $string;
}

sub getMaillists {
  my ($domainName) = @_;

  my $maillistsFiles = _getMaillistsFiles($domainName);
  my %result;

  while (my ($maillistName, $files) = each %{$maillistsFiles}) {
    $result{$maillistName} = {
      'adminPassword' => _getMaillistAdminPassword($files->{'configFile'}),
      'members' => _getMaillistMembers($files->{'membersListFile'}),
      'owner' => _getMaillistOwner($domainName, $maillistName)
    };
  }

  return \%result; 
}

sub getMaillistsAliases {
  my ($domainName) = @_;

  my @maillists = keys(%{_getMaillistsFiles($domainName)});
  my @aliases;

  for my $maillist (@maillists) {
    push @aliases, (
      $maillist,
      "owner-$maillist",
      "$maillist-owner",
      "$maillist-request",
      "$maillist-outgoing",
      "$maillist-list",
      "$maillist-approval"
    );
  }

  return \@aliases;
}

# returns reference to a hash with:
# - key: name of a maillist
# - value: reference to a hash with keys 'configFile' and 'userListFile' and values - full path to corresponding maillist files
sub _getMaillistsFiles {
  my ($domainName) = @_;

  my $listsDir = getFilesystemByDomainName($domainName) . '/var/lib/majordomo/lists';

  unless (-d $listsDir) {
    Logging::debug("Domain $domainName doesn't contain maillists directory");
    return {};
  }
  
  my %lists;

  open(LISTCONFIGS, AgentConfig::findBin() . " $listsDir -maxdepth 1 -type f -name '*.config' |");
  binmode LISTCONFIGS;

  while(<LISTCONFIGS>) {
    chomp;
    
    my $configFile = $_;
    my ($listName, $membersListFile);
    
    if (/([^\/]+)\.config/) {
      $listName = $1;
    } else {
      Logging::warning("Failed to get maillist name from config file name '$_', ignoring maillist config");
      next;
    }
    
    if (/^(.+)\.config$/) {
      $membersListFile = $1;
    } else {
      Logging::warning("Failed to get maillist members file name from config file name '$_', ignoring maillist config");
      next;
    }

    $lists{$listName} = {
      'configFile' => $configFile,
      'membersListFile' => $membersListFile
    };
	}

	close LISTCONFIGS;

  return \%lists;
}

sub _getMaillistAdminPassword {
  my ($configFile) = @_;

  open(LISTCONFIG, $configFile);
  binmode LISTCONFIG;

  while (<LISTCONFIG>) {
    if (/admin_passwd\s=\s(.*)$/) {
      close(LISTCONFIG);
      return $1;
    }
  }

  close LISTCONFIG;
  return undef;
}

sub _getMaillistMembers {
  my ($membersListFile) = @_;

  my @result;

	open(LISTMEMBERS, $membersListFile);
	binmode LISTMEMBERS;

	while (<LISTMEMBERS>) {
		chomp;
    push @result, $_;
  }

  close LISTMEMBERS;

  return \@result;
}

sub _getMaillistOwner {
  my ($domainName, $maillistName) = @_;

  my $aliases = getMailAliases($domainName);

  while (my ($alias, $aliasInfo) = each %{$aliases}) {
    if ($alias eq $maillistName . '-owner') {
      if (@{$aliasInfo->{'localDelivery'}}) {
        return $aliasInfo->{'localDelivery'}->[0] . '@' . $domainName;
      } elsif (@{$aliasInfo->{'redirects'}}) {
        return $aliasInfo->{'redirects'}->[0];
      }
    }
  }

  return undef;
}

sub getProcmailrcAction {
    my ($procmailrcFile) = @_;

    my $action = "mark";  # default value

    open(INPUTPROCCFG, $procmailrcFile) || return $action;
    while (<INPUTPROCCFG>) {
        chomp;
        if (/^SPAM_FOLDER="(.+)"$/) {
            if ($1 eq "\$HOME/mail/Spam") {
                $action = "move";
            } elsif ($1 eq "/dev/null") {
                $action = "delete";
            }
            last;
        }
    }
    close INPUTPROCCFG;

    return $action;
}

sub getSpamassassinMaxThreads {
  # we don't claim this setting is migrated and therefore don't fight to retrieve it.
  # if unable to open both known spamassassin settings files (in this order) - count it unspecified.
  open(SETTINGS_FILE, '/etc/sysconfig/spamassassin') ||
    open(SETTINGS_FILE, '/etc/init.d/spamassassin') ||
      return undef;

  while (<SETTINGS_FILE>) {
    if (/^[[:blank:]]*SPAMDOPTIONS=".*-m[[:blank:]]*(\d*)/) {
      return $1;
    }
  }
  close(SETTINGS_FILE);

  return undef;
}

sub getDatabaseNames {
  my $domainName = shift @_;

  my %mysqlInfo = getSpecificInfoForSite(getSiteIdByDomain($domainName), 'mysql');
  my $defaultDbPrefix = $domainName;
  $defaultDbPrefix =~ s/\./_/g;
  my $dbPrefix = exists $mysqlInfo{'dbaseprefix'} ? $mysqlInfo{'dbaseprefix'} : $defaultDbPrefix;

  my $db = _getClientDb();
  my @userDatabases = $db->queryList("SHOW DATABASES LIKE '$dbPrefix%'");

  return @userDatabases;
}

sub getDatabaseUsers {
  my ($dbName, $excludeUsers) = @_;
  
  my $db = _getClientDb();

  $dbName =~ s/_/\\_/g;
  my (@users);

  my $sql = "SELECT DISTINCT User FROM db WHERE Db = '$dbName'";
  if (defined $excludeUsers && @{$excludeUsers}) {
    $sql .= " AND User NOT IN (" . join(", ", map {"'$_'"} @{$excludeUsers}) . ")";
  }

  @users = $db->queryList($sql);
  if (scalar(@users) == 0) { # bug 94641
    @users = $db->queryList("SELECT DISTINCT User FROM db WHERE Db LIKE '$dbName'");
  }
  return @users;
}

sub getDatabaseUserData {
  my ($userName) = @_;

  my $db = _getClientDb();
  my $sql = "SELECT password,host FROM user WHERE user='$userName'";
  my @rows = $db->queryTable($sql);

  my $password = undef;
  my @accessHosts = ();
  foreach my $row (@rows) {
    push @accessHosts, $row->{'host'};
    $password ||= $row->{'password'}; # first password wins, assuming they're all the same
  }
  return ($password, \@accessHosts);
}

sub _mangleDbUserName {
  my ($userName) = @_;

  my $mangledUserName = $userName;
  if ($userName eq 'admin') {
    my $db = _getClientDb();
    my $suffix = 0;
    for($suffix = 0; $suffix < USERNAME_MANGLE_MAX_ITERATIONS; $suffix += 1) {
      # we should check 16-symbol limit here, but it's met automatically
      $mangledUserName = $userName . '_' . $suffix;
      my $sql = "SELECT count(1) from db where user = '$mangledUserName';";
      if (($db->queryOneValue($sql)) < 1) {
        last;
      }
    }
    unless ($suffix < USERNAME_MANGLE_MAX_ITERATIONS) {
      Logging::error("Unable to mangle DB user name after '" . USERNAME_MANGLE_MAX_ITERATIONS . "' tries. Please contact support for assistance.");
      # leave last tried username here, because it's still better than 'admin' collision
    }
  }
  return $mangledUserName;
}

sub getEnabledWebstats {
  my ($domain) = @_;

  my $siteId = Dumper::getSiteIdByDomain($domain);

  my @enabled = ();
  for my $engine qw(webalizer awstats analog) {
    my %info = Dumper::getSpecificInfoForSite($siteId, $engine);
    if ('1' eq $info{'enabled'}) {
      push @enabled, $engine;
    }
  }

  return @enabled;
}

# TODO During migration it should go to dump.xml node 'phosting / preferences / sysuser / cron'
sub getClientCronJobs {
  my $domain = shift @_;

  my $siteId   = Dumper::getSiteIdByDomain($domain);
  my %info     = getSpecificInfoForSite($siteId, 'siteinfo');
  my $cronFile = "/var/spool/cron/$info{'admin'}";

  return DumperUtils::getFileLines($cronFile);
}

sub getDomainAliases {
  my ($domain) = @_;

  my $siteId = Dumper::getSiteIdByDomain($domain);
  my %info = Dumper::getSpecificInfoForSite($siteId, 'aliases');
  my $aliases = $info{'aliases'};
  # ['@li.as', 'metoo.test'] --> @li.as, metoo.test
  $aliases =~ s/[\[\]']//g;

  return split ', ', $aliases;
}

sub getMailResponders {
  my ($domain) = @_;

  my $siteId = Dumper::getSiteIdByDomain($domain);
  my $respondersDir = "/home/virtual/site$siteId/fst/usr/share/opcenter/responders";

  my @responders = ();
  unless (opendir (my $dir, $respondersDir)) {
    Logging::warning("Unable to open the autoresponders directory '$respondersDir'");
  } else {
    @responders = map { "$respondersDir/$_" } grep { /^responder*/ && -f "$respondersDir/$_" } readdir ($dir);
    closedir $dir;
  }
  return @responders;
}

sub getDuplicatedLogins {
  my $sql = 'SELECT admin_user, COUNT(*) AS duplicates FROM siteinfo GROUP BY admin_user HAVING COUNT(*) > 1';
  return _getDb()->queryList($sql);
}

sub getPowerToolsAvailability {
  my ($domain) = @_;

  my $siteId = Dumper::getSiteIdByDomain($domain);

  my $sql = "SELECT d.shortname FROM deployables d JOIN deployables_sites ds ON d.shortname = ds.shortname "
            . "WHERE ds.site_id = $siteId AND d.installed = TRUE AND ds.deployable = TRUE";
  return _getDb()->queryList($sql);
}

my $_namedStatus = undef;

sub getLocalNamedStatus {
  return $_namedStatus if defined ($_namedStatus);

  my $namedStatusCommand = '/sbin/service named status';

  if (-f '/sbin/service') {
    chomp(my $namedStatusWord = `/sbin/service named status`);
    Logging::trace("'$namedStatusCommand' returned '$namedStatusWord'");
    if ($namedStatusWord eq 'running') {
      $_namedStatus = 1;
    } else {
      $_namedStatus = 0;
    }
    return $_namedStatus;
  } else {
    return undef;
  }
}

# PHP handler is determined by site security level. 'high' will get 'cgi+suexec',
# 'medium' and 'low' - 'mod_php'. The only easily visible difference between those 
# groups of security levels is a 'jail' flag in 'apache' table, which is enough.
sub getSitePhpMode {
  my ($siteId) = @_;

  my $isJailed = undef;
  if (defined($siteId)) {
    my $sql = "SELECT jail from apache where site_id = '$siteId'";
    $isJailed = _getDb()->queryOneValue($sql);
  }

  if (defined($isJailed)) {
    if ($isJailed eq '1') {
      return 'cgi';
    } else {
      return 'module';
    }
  }

  return undef;
}

sub mangleResellerName {
  my ($resellerName) = @_;
  my $mangledResellerName = $resellerName;

  if ($resellerName eq 'admin') {
    my %resellers = map { $_ => 1 } getResellers();
    my $maxIterations = scalar(keys(%resellers)) + 2;
    for (my $suffix = 0; $suffix < $maxIterations; $suffix += 1) {
      $mangledResellerName = $resellerName . '_' . $suffix;
      last if not exists $resellers{$mangledResellerName};
    }
  }

  return $mangledResellerName;
}

# return value is in bytes
sub getDomainDiskQuotaUsed {
  my $domainName = shift;

  my @sitelookupCommand = ('/usr/local/bin/sitelookup', '-d', $domainName, 'wp_user,site_root');
  Logging::debug("Site lookup command: " . join(',', @sitelookupCommand));
  open FH, '-|', @sitelookupCommand;
  my $line = <FH>;
  close FH;

  if (defined($line)) {
    chomp $line;
    my ($unixUser, $siteRoot) = split(',', $line);
    my @quotaLines = _getQuotaReport($siteRoot, $unixUser);
    my $currentQuota = $quotaLines[1];

    if (defined($currentQuota)) {
      return $currentQuota * 1024;
    }
  }
  return undef;
}

sub _getQuotaReport {
  my ($directory, $unixUser) = @_;
  my (undef, undef, undef, $unixGroupId) = getpwnam($unixUser);
  my @quotareportCommand = ('/usr/local/bin/quota_report', '-d', $directory, '-g', $unixGroupId);
  Logging::debug("Site quota report command: " . join(',', @quotareportCommand));
  open FH, '-|', @quotareportCommand;
  my @quotaLines = <FH>;
  close FH;
  return @quotaLines;
}

1;
