#!/usr/bin/perl
# Copyright 1999-2015. Parallels IP Holdings GmbH. All Rights Reserved.

use vars qw($dumpDir $ptrArgs $workDir $storage);
use strict;

use AgentConfig;

use XmlNode;

my $dbName='hsphere';
my $dbHost='localhost';
my $dbType='Pg';
my $propertiesFile='/hsphere/local/home/cpanel/shiva/psoft_config/hsphere.properties';
#
# parse command line
#
my %arg_opts = ('--help|-h'=>'',
				'--get-status|-s'=>'',
				'--dump-accounts|-dc'=>'s',
				'--dump-domains|-dd'=>'s',
				'--dump-all|-da'=>'',
				'--get-content-list|-lc'=>'',
				'--no-content|-nc'=>'',
				'--no-compress|-nz'=>'',
				'--output|-o'=>'s',
				'--status-file|-sf'=>'s',
			   );

#
# global variables
#
my %exclIp;
my $kilobyte = 1024;
my $megabyte = $kilobyte*$kilobyte;

my $versionFile="/hsphere/local/home/cpanel/shiva/psoft_config/HS_VERSION";

my $ignoredUsers_ = {'root' => '1',
					 'bin' => '1',
					 'daemon' => '1',
					 'adm' => '1',
					 'lp' => '1',
					 'sync' => '1',
					 'shutdown' => '1',
					 'halt' => '1',
					 'mail' => '1',
					 'uucp' => '1',
					 'operator' => '1',
					 'games' => '1',
					 'gopher' => '1',
					 'ftp' => '1',
					 'news' => '1',
					 'mysql' => '1',
					 'nobody' => '1',
					 'zope' => '1',
					 'majordomo' => '1',
					 'tomcat4' => '1',
					 'apache' => '1',
					 'ensimrootmail' => '1',
					 'sshd' => '1',
					};

my %ignoredUsers = %{$ignoredUsers_};

#
# end global variables
#


#@@INCLUDE FILE="agent.include.pl"@@
if (-f 'agent.include.pl') {
  require 'agent.include.pl';
} else {
  require '../agent.include.pl';
}
#@@/INCLUDE@@

$ptrArgs = &getArguments (\@ARGV,\%arg_opts);

my ($outPath,$dumpFile,$statusFile);

$workDir = AgentConfig::cwd();

$dumpFile = $workDir.'/HsphereX.xml';
$statusFile = $workDir.'/HsphereX.status';
$dumpDir = $workDir.'/pma';

my $objDumpStatus = &makeDumpStatus($ptrArgs->{'status-file'}||$statusFile);

my $rootName = 'migration-dump';
my $dtdName = 'HsphereX.dtd';
my $dbUser = 'wwwuser';
my $dbPasswd = getDbPassword($propertiesFile);

my $wrapDbh = getDbConnect($dbType, $dbUser, $dbPasswd, $dbName,$dbHost);

if (exists $ptrArgs->{'get-status'}) {
  printAgentStatus();
  exit 0;
}

unless (ref($wrapDbh)=~/HASH/ && ref($wrapDbh->{'EXECUTE'})=~/CODE/) {
  die "Unable to connect to Hsphere DB. Giving up.";
}

#
# get MIME Base64 encoder
#
my $wrapBase64 = makeMIMEBase64();

if(exists $ptrArgs->{'get-status'}){
  printAgentStatus();
}elsif (exists ($ptrArgs->{'dump-all'}) ||
	exists ($ptrArgs->{'dump-accounts'}) ||
	exists ($ptrArgs->{'dump-domains'})) {

  my ($root,@accounts,@domains,$ptrAccounts,$ptrDomains, $value);
  initStorage();
  getDumpDir($dumpDir);

  if (exists $ptrArgs->{'no-compress'}) {
	setCompress(0);
  }

  &printToLog("Work dir: $workDir");
  &printToLog("Dump file: $dumpFile");
  &printToLog("Status file: ".$objDumpStatus->{'FILE'}->());


  if ($value = $ptrArgs->{'dump-accounts'}) {
	if ($value eq "-") {
	  $value = <STDIN>;
	  chomp $value;
	}
	@accounts = split(/\s*,\s*/,$value);
	$ptrAccounts = \@accounts;
  }

  if ($value = $ptrArgs->{'dump-domains'}) {
	if ($value eq "-") {
	  $value = <STDIN>;
	  chomp $value;
	}
	@domains = split(/\s*,\s*/,$value);
	$ptrDomains = \@domains;
  }
  #
  # generate a xml dump
  #

  $root = &getHsphereXDump($dumpDir, $ptrAccounts, $ptrDomains, $rootName);

  #
  # print dump to output
  #
  $storage->finish($root);
} elsif (exists $ptrArgs->{'get-content-list'}) {
  makeContentList();
} else {
  &printHelp();
}

exit 0;

#
# end main
#
#==============================================================
#
# subroutines
#

sub printAgentStatus {
  my $root = makeXmlNode('agent-status');

  unless (-e $versionFile && defined AgentConfig::iconvBin()) {
    my $item;
  	if (defined AgentConfig::iconvBin()) {
	  $item = makeXmlNode('wrong-platform', '');
	} else {
	  $item = makeXmlNode('wrong-platform', 'no iconv found on the source host');
	}

	$root->{'ADDCHILD'}->($item);
  }

  printXml($root, *STDOUT);
}

sub getHsphereXDump {
  my ($dumpDir, $ptrAccounts, $ptrDomains, $rootName) = @_;
  $rootName = 'migration-dump' unless $rootName;
  my ($item);
  my $root = XmlNode->new($rootName, 'attributes'=>{'agent-name', 'HsphereX'});

  if (ref($ptrDomains) =~ /ARRAY/ && (@{$ptrDomains} > 0)) {

	my $list = join(',', map {"'$_'"} @{$ptrDomains});
	my $sql = "SELECT id FROM domains WHERE name IN ($list)";

	$wrapDbh->{'EXECUTE'}->($sql);

	my $ptrRow;
	my %domainIds;
	while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
	  $domainIds{$ptrRow->[0]} = 1;
	}
	$wrapDbh->{'FINISH'}->();

	# prepeare ip
	printToLog('Reading IP information ...');
	readIpInfo();

	createSelectedDomains($root, \%domainIds);

  } elsif (ref($ptrAccounts) =~ /ARRAY/ && (@{$ptrAccounts} > 0)) {
	my %accountIds;

	my @accounts = @{$ptrAccounts};

	my $list = join(',', map {"'$_'"} @accounts);
	my $sql = "SELECT users.id FROM users,resellers WHERE users.username IN ($list) AND users.reseller_id=1 AND users.id=resellers.id";
	$wrapDbh->{'EXECUTE'}->($sql);

	my $ptrRow;
	while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
	  $accountIds{$ptrRow->[0]} = 1;
	}
# if admin is present - add him to %accountIds
	foreach my $id (@accounts) {
		if ($id eq 'admin') {
		$accountIds{1}=1;
		}
	}
	$wrapDbh->{'FINISH'}->();

	# prepeare ip
	printToLog('Reading IP information ...');
	readIpInfo();

	createSelectedAccounts($root, \%accountIds);

  } else {
	createAllAccounts($root);
  }

  return $root;
}

sub createAllAccounts {
  my ($parent) = @_;
  my ($item);

  my $ptrRow;
  my %accounts;

  my $sql = "SELECT COUNT(id) FROM resellers";

  $wrapDbh->{'EXECUTE'}->($sql)
	or die "Cannot retrieve resellers list from DB";

  $ptrRow = $wrapDbh->{'FETCHROW'}->();

  unless ($ptrRow == 0){

  $wrapDbh->{'FINISH'}->();

  $sql = "SELECT DISTINCT id FROM resellers";

  while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
	$accounts{$ptrRow->[0]} = 1;
  }
  }
  # add admin as reseller because he is not present in resellers;
  $accounts{1}=1;

  # prepeare ip
  printToLog('Reading IP information ...');
  readIpInfo();

  createSelectedAccounts($parent, \%accounts);
}

sub createSelectedAccounts {
  my ($parent, $accountsPtr) = @_;
  my %accounts = %{$accountsPtr};

  my $account;
  foreach $account (keys %accounts) {
	&createAccountNode($parent, $account);
  }
}

sub createSelectedDomains {
  my ($parent, $domainsPtr) = @_;
  my %domains = %{$domainsPtr};

  my $account = makeXmlNode('client');
  $account->{'ATTRIBUTE'}->('name','client');
  createPasswordNode($account, '', 'plain');

  my $status = XmlNode->new('status', 'children' => [XmlNode->new('enabled')]);
  $account->{'ADDCHILD'}->($status);

  my $domain;
  foreach $domain (keys %domains) {
	&createDomain($account, $domain);
  }

  $parent->{'ADDCHILD'}->($account);
}

sub createAccountNode {
  my ($parent, $accountId) = @_;
  my ($sql,$ptrHash,$item,$ptrRow);
  &printToLog("Creating account $accountId");

# Get login and password for admin account
  $sql = "SELECT username,password FROM users WHERE id=$accountId";

  ($wrapDbh->{'EXECUTE'}->($sql) && ($ptrHash = $wrapDbh->{'FETCHHASH'}->()))
	or die "Cannot retrieve info for account $accountId";
  my %info = %$ptrHash;

  $wrapDbh->{'FINISH'}->();

  $item = &makeXmlNode('client');

  &createPasswordNode($item, $info{'password'}, 'plain');

  my $status = XmlNode->new('status', 'children' => [XmlNode->new('enabled')]);
  $item->{'ADDCHILD'}->($status);

  if ($info{'username'} eq 'admin') {
  	$info{'username'} = $info{'username'}.1;
  }

  $item->{'ATTRIBUTE'}->('name',$info{'username'});

 # Get contact info for reseller
  $sql = "SELECT name,value FROM settings WHERE reseller_id=$accountId";
 $wrapDbh->{'EXECUTE'}->($sql)
	or die "Cannot retrieve contact info for reseller $accountId ";
  while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
  	$info{$ptrRow->[0]}=$ptrRow->[1];
  }
  $wrapDbh->{'FINISH'}->();

  $item->{'ATTRIBUTE'}->('contact',"$info{'ofname'} $info{'olname'}");
 # strip dots in phone
  $info{'phone'} =~s/\.//g;

  &createPinfoNode($item, 'phone',$info{'phone'});
  &createPinfoNode($item, 'address',$info{'address'});
  &createPinfoNode($item, 'city',$info{'city'});
  &createPinfoNode($item, 'state',$info{'state'});
  &createPinfoNode($item, 'zip',$info{'zip'});
  &createPinfoNode($item, 'country',$info{'country'});
  $info{'lang'}=~/(^.{5})/;
  $info{'lang'}= $1;
  $info{'lang'}=~ s/_/-/g;
  &createPinfoNode($item, 'locale',$info{'lang'}) if defined($info{'lang'});
  &createPinfoNode($item, 'email',$info{'email'});

  unless (exists $ptrArgs->{'no-content'}) {

  my %resLimits;

  $sql = "SELECT resource_amount.amount,parent_child.child_type FROM parent_child,resource_amount WHERE resource_amount.id=parent_child.child_id AND account_id=(SELECT account_id FROM user_account WHERE user_id=$accountId) AND (child_type=4003 OR child_type=3011)";

  if ($wrapDbh->{'EXECUTE'}->($sql)) {
  	while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
		$resLimits{$ptrRow->[1]} = $ptrRow->[0];
	}
  }
  $wrapDbh->{'FINISH'}->();
  if (exists ($resLimits{4003})) {
  	createLimitNode($item,'disk_space',$resLimits{4003}*$megabyte);
	}
  if (exists ($resLimits{3011})){
	createLimitNode($item,'max_traffic',$resLimits{3011}*$megabyte);
	}
  }

  my $ipPool = makeXmlNode('ip_pool');
  $sql = "SELECT DISTINCT dns_records.data,l_server_ips.r_type FROM l_server_ips,dns_records WHERE type='A' AND id IN (SELECT child_id FROM parent_child WHERE parent_id IN (SELECT child_id FROM parent_child WHERE child_type=8 AND parent_id IN (SELECT child_id FROM parent_child WHERE account_id IN (SELECT account_id FROM user_account WHERE user_id IN (SELECT id FROM users WHERE reseller_id=$accountId)) AND (child_type=2 OR child_type=34))) AND child_type=3002)  AND l_server_ips.ip=dns_records.data";

  my @ips;

  if ($wrapDbh->{'EXECUTE'}->($sql)) {
	  while($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
		  push @ips,$ptrRow->[0];
	  }
	  foreach my $ip (uniq(@ips)) {
		  $ipPool->{'ADDCHILD'}->(makeIpNode($ip));
	  }
  }
  $wrapDbh->{'FINISH'}->();
  $item->{'ADDCHILD'}->($ipPool);

  unless(exists $ptrArgs->{'no-content'}) {

  createAllDatabases($item,$accountId);

  }

  createAllDomainsForAccount($item, $accountId);
  $parent->{'ADDCHILD'}->($item);
}

sub createAllDomainsForAccount {
  my ($parent, $accountId) = @_;
  #select all real and service domains (child_type=2 and 34)
  my $sql = "SELECT domains.id FROM domains,parent_child WHERE domains.id = parent_child.child_id AND (parent_child.child_type=2 OR parent_child.child_type=34) AND parent_child.account_id IN (SELECT account_id FROM user_account,users WHERE user_account.user_id=users.id AND users.reseller_id=$accountId) ";

  $wrapDbh->{'EXECUTE'}->($sql);

  my $ptrRow;
  my @domains;
  while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
	push @domains,$ptrRow->[0];
  }

  my $domain;

  foreach $domain (@domains) {
  	createDomain($parent,$domain);
  }
  $wrapDbh->{'FINISH'}->();
}

sub createDomain {
  my ($parent, $domainId) = @_;

  my ($ptrRow,$sql,$ptrHash);
  # Check if domain is present at this host
 $sql="SELECT name FROM p_server WHERE id=(SELECT p_server_id FROM l_server WHERE id=(SELECT DISTINCT hostid FROM unix_user, parent_child WHERE parent_child.account_id = (SELECT account_id FROM domains, parent_child  WHERE domains.id = parent_child.child_id AND domains.id =$domainId) AND child_id = unix_user.id))";

  ($wrapDbh->{'EXECUTE'}->($sql) && ($ptrRow = $wrapDbh->{'FETCHROW'}->()));
  unless ($ptrRow->[0] eq getHostName()){
  	printToLog("Domain $domainId is not present at this host");
	return;
  }


  &printToLog("Creating domain $domainId");

  $sql = "SELECT name FROM domains WHERE id = $domainId";
 ($wrapDbh->{'EXECUTE'}->($sql) && ($ptrRow = $wrapDbh->{'FETCHROW'}->()))
        or die "Cannot retrieve domain info from DB";

  my $item = &makeXmlNode('domain');
  $item->{'ATTRIBUTE'}->('name',(@$ptrRow)[0]);

  my $domainName=(@$ptrRow)[0];

  $wrapDbh->{'FINISH'}->();

  # select ip address for domain
  $sql ="SELECT DISTINCT data FROM dns_records WHERE id IN (SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE child_type=8 AND parent_id=$domainId) AND child_type=3002)";
  ($wrapDbh->{'EXECUTE'}->($sql) && ($ptrRow = $wrapDbh->{'FETCHROW'}->()))
        or die "Cannot retrieve IP for domain $domainId from DB";

  $item->{'ADDCHILD'}->(makeIpNode((@$ptrRow)[0]));

  my $status = XmlNode->new('status', 'children' => [XmlNode->new('enabled')]);
  $item->{'ADDCHILD'}->($status);

  $parent->{'ADDCHILD'}->($item);

  if (exists $ptrArgs->{'no-content'}) {
	return;
  }

#
#	LIMITS
#
#
#	PERMISSIONS
#
#
# 	MAIL SYSTEM
#

  &createMailSystem($item,$domainId);

#
#	DNS
  &createDNSrecords($item,$domainId);
#
#	DATABASE
#
#
#	MAILING LISTS
#
#
#	TRAFFIC
#
#
#	STATUS
#
#
#	CERTIFICATE
#
#
#	DOMAINUSER
#
  $sql="SELECT name,last_name,company,phone,email,address1,city,state,postal_code,country FROM contact_info WHERE id=(SELECT ci_id FROM accounts WHERE id=(SELECT DISTINCT account_id FROM parent_child WHERE parent_id=$domainId AND (parent_type=2 OR parent_type=34)))";

  if (($wrapDbh->{'EXECUTE'}->($sql)) && ($ptrHash = $wrapDbh->{'FETCHHASH'}->())) {
  	createDomUserNode($item,$domainId,$ptrHash);
}

  $wrapDbh->{'FINISH'}->();

#
#	HOSTING
#
  $sql="SELECT child_id,child_type FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE parent_id=$domainId AND (parent_type=2 OR parent_type=34)AND child_type=9)";

  $wrapDbh->{'EXECUTE'}->($sql) or die "Cannot retrieve hosting info for domain $domainId ";

  my %hostingPrefs;

  while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
  	$hostingPrefs{$ptrRow->[1]} = $ptrRow->[0];
}
    createPhysHostingNode($item,$domainId,\%hostingPrefs,$domainName);

#  $parent->{'ADDCHILD'}->($item);
}


sub createPhysHostingNode {
	my ($root,$domId,$ptrHash,$domName)=@_;
	my ($sql,$ptrRow,%attr);
	my %hostingPrefs=%$ptrHash;
	my $item = makeXmlNode('phosting');

# SYSUSER

  $sql="SELECT login,password FROM unix_user WHERE id =(SELECT parent_id FROM parent_child WHERE parent_type=7 AND child_id=$domId)";

 ($wrapDbh->{'EXECUTE'}->($sql) && ($ptrRow = $wrapDbh->{'FETCHROW'}->()))
        or die "Cannot retrieve sysuser for domain $domId ";

  %attr = ('name'=> $ptrRow->[0],'password'=>$ptrRow->[1]);

  $wrapDbh->{'FINISH'}->();

  $sql="SELECT size_mb FROM quotas WHERE id=(SELECT child_id FROM parent_child WHERE child_type=4001 AND parent_id=(SELECT parent_id FROM parent_child WHERE parent_type=7 AND child_id=$domId))";

  if (($wrapDbh->{'EXECUTE'}->($sql))&&($ptrRow = $wrapDbh->{'FETCHROW'}->())){
  	%attr = (%attr,'quota' => $ptrRow->[0]);
  }

  $wrapDbh->{'FINISH'}->();

  my $sysUserNode = createSysUserNode(\%attr);
  $item->{'ADDCHILD'}->($sysUserNode);
# SCRIPTING
  createScriptingNode($item,\%hostingPrefs);
# FPUSER
  if (exists($hostingPrefs{'23'})){
  	$item->{'ATTRIBUTE'}->('fp','true');
  	createFpUserNode($item,$hostingPrefs{'23'},$domId);
  }
# WEBALIZER
  my (%statistics,$webstatDir);

  if (exists($hostingPrefs{'18'})) {
  $sql="select child_id,child_type from parent_child where parent_id=$hostingPrefs{'18'}";
        if ($wrapDbh->{'EXECUTE'}->($sql)) {
        while($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
        $statistics{$ptrRow->[1]}=$ptrRow->[0];
        }
    }
        $wrapDbh->{'FINISH'}->();

  if (exists($statistics{'28'})){
        $item->{'ATTRIBUTE'}->('webstat','true');
        $sql = "SELECT dir FROM apache_webalizer WHERE id=$statistics{'28'}";
        if (($wrapDbh->{'EXECUTE'}->($sql))&&($ptrRow = $wrapDbh->{'FETCHROW'}->())) {
        $webstatDir=$ptrRow->[0];
        }
        $wrapDbh->{'FINISH'}->();

  my $webstatRoot = "/home/hsphere/local/home/".getPath($domId)."/".$domName.$webstatDir;

  if (-d $webstatRoot) {

        my $webstatFile=makeDumpFile("$dumpDir/$domName\_webstat",$webstatRoot);
        $item->{'ATTRIBUTE'}->('cid_webstat',$webstatFile);
        }
   }

  }
# LOGROTATION
# ANONFTP
# PROTECTED DIR
# WEBUSER
# SUBDOMAIN
  createSubDomains($item,$domId);

# CONTENT
  my $docroot = "/home/hsphere/local/home/".getPath($domId)."/".$domName;
  my @exclude;

  push @exclude,'cgi-bin';
  push @exclude,$webstatDir;

  if (-d $docroot) {
        my $filename = makeDumpFile("$dumpDir/$domName\_docroot",$docroot,'',\@exclude);

	$item->{'ATTRIBUTE'}->('cid_docroot', $filename);

  }

  my $cgi = "/home/hsphere/local/home/".getPath($domId)."/".$domName."/cgi-bin";

  if (-d $cgi) {
  	my $filename = makeDumpFile("$dumpDir/$domName\_cgi",$cgi);

	$item->{'ATTRIBUTE'}->('cid_cgi',$filename);
  }

  $root->{'ADDCHILD'}->($item);
}

my %sysUsers;

sub createSysUserNode {
  my ($ptrHash)=@_;
  my ($username,$password,$item,$quota);
  my %myhash = %$ptrHash;

  $username = $myhash{'name'};
  $password = $myhash{'password'};
  $quota = $myhash{'quota'};

  if ($username) {
  	if ( exists( $sysUsers{$username} ) ) {
                        my $clon = 1;
                        while(exists($sysUsers{$username})){
                                if($username =~ /(.*)c(\d+)$/){
                                        $username = $1;
                                        $clon = $2;
                                        $clon++;
                                }
                                $username .= "c$clon";
                        }
                }
                $sysUsers{$username} = 1;

                $item = &makeXmlNode('sysuser');
                $item->{'ATTRIBUTE'}->('name',$username);
		$item->{'ATTRIBUTE'}->('quota',$quota*$megabyte);
                $item->{'ADDCHILD'}->( createPasswordNode($item,$password,'plain') );

  }
  return $item;
}

sub createScriptingNode {
	my ($parent,$ptrHash)=@_;
	my %hostingPrefs=%$ptrHash;

  my $scriptingNode = makeXmlNode('scripting');
  $scriptingNode->{'ATTRIBUTE'}->('ssi','true') if (exists($hostingPrefs{'13'}));
  $scriptingNode->{'ATTRIBUTE'}->('php','true') if (exists($hostingPrefs{'11'}));
  $scriptingNode->{'ATTRIBUTE'}->('cgi','true') if (exists($hostingPrefs{'10'}));
  $scriptingNode->{'ATTRIBUTE'}->('asp','true') if (exists($hostingPrefs{'27'}));
  $scriptingNode->{'ATTRIBUTE'}->('coldfusion','true') if (exists($hostingPrefs{'6500'}));
  $parent->{'ADDCHILD'}->($scriptingNode);

}

sub createSubDomains {
	my ($parent,$domainId)=@_;
	my (@subdomains,$sql,$ptrRow);

  $sql="SELECT child_id FROM parent_child WHERE child_type=31 AND parent_id=$domainId";

  unless($wrapDbh->{'EXECUTE'}->($sql)) {
  	return;
  }

  while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
  push @subdomains,$ptrRow->[0];
  }
 foreach my $subId (@subdomains) {
 	createSubDomainNode($parent,$subId);
 }
}

sub createSubDomainNode {
	my ($parent,$subDomainId)=@_;
	my ($sql,$ptrRow);
  $sql="SELECT child_id,child_type,name FROM parent_child,domains WHERE parent_id=(SELECT child_id FROM parent_child WHERE child_type=9 AND parent_id=$subDomainId) AND domains.id=$subDomainId";

$wrapDbh->{'EXECUTE'}->($sql)
      or die "Cannot retrieve subdomain $subDomainId";

  my %hostingPrefs;

  while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
  	$hostingPrefs{$ptrRow->[1]} = $ptrRow->[0];
}
  $wrapDbh->{'FINISH'}->();

  $sql = "SELECT name FROM domains WHERE id=$subDomainId";
  ($wrapDbh->{'EXECUTE'}->($sql) && ($ptrRow = $wrapDbh->{'FETCHROW'}->()))
  	or die "Cannot get subdomain name $subDomainId";
  my @subDomainName=split('\.',$ptrRow->[0]);

  $wrapDbh->{'FINISH'}->();

  my $item=makeXmlNode('subdomain');

  $item->{'ATTRIBUTE'}->('name',$subDomainName[0]);
  createScriptingNode($item,\%hostingPrefs);
 #content

  my $fullSubDomainName = join ('.',@subDomainName);
  my $docroot = "/home/hsphere/local/home/".getPath($subDomainId)."/".$fullSubDomainName;
  my $exclude = "cgi-bin";
  if (-d $docroot) {
        my $filename = makeDumpFile("$dumpDir/$fullSubDomainName\_docroot",$docroot,'',$exclude);

	$item->{'ATTRIBUTE'}->('cid_docroot', $filename);

  }

  my $cgi = "/home/hsphere/local/home/".getPath($subDomainId)."/".$fullSubDomainName."/cgi-bin";

  if (-d $cgi) {
  	my $filename = makeDumpFile("$dumpDir/$fullSubDomainName\_cgi",$cgi);

	$item->{'ATTRIBUTE'}->('cid_cgi',$filename);
  }

  $parent->{'ADDCHILD'}->($item);
}

sub createFpUserNode {
	my ($parent,$fpUserId,$domainId)=@_;
	my ($sql,$ptrRow);
	my $item = makeXmlNode('fpuser');

  $sql="SELECT login,password FROM frontpage WHERE id=$fpUserId";

  unless ($wrapDbh->{'EXECUTE'}->($sql)) {
   $sql = "SELECT login,password FROM unix_user WHERE id =(SELECT parent_id FROM parent_child WHERE parent_type=7 AND child_id=$domainId)";

  ($wrapDbh->{'EXECUTE'}->($sql) && ($ptrRow = $wrapDbh->{'FETCHROW'}->()))
        or die "Cannot retrieve sys user for  $domainId ";

  $item->{'ATTRIBUTE'}->('name',$ptrRow->[0]);
  createPasswordNode($item,$ptrRow->[1],'plain');
  $parent->{'ADDCHILD'}->($item);
  return;

  }

  $ptrRow = $wrapDbh->{'FETCHROW'}->();

  $item->{'ATTRIBUTE'}->('name',$ptrRow->[0]);
  createPasswordNode($item,$ptrRow->[1],'plain');
  $parent->{'ADDCHILD'}->($item);
}

sub makeIpNode( $ )
{
	my $ip = shift;

	my $type = 'shared';
	if ($exclIp{$ip}) {
		$type = 'exclusive';
	}

	my $ip_node = makeXmlNode('ip');
	$ip_node->{'ADDCHILD'}->(makeXmlNode('ip-type',    $type));
	$ip_node->{'ADDCHILD'}->(makeXmlNode('ip-address', $ip));
	return $ip_node;
}

sub createMailSystem {
  my ($parent,$domId)=@_;

  my (%mailTypes,$ptrRow,$item,$sql);
  
  $item = makeXmlNode('mailsystem');

  my $status = XmlNode->new('status', 'children' => [XmlNode->new('enabled')]);
  $item->{'ADDCHILD'}->($status);

  # Check if mailservice is present for domain
  $sql = "SELECT * FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE child_type=1000 AND parent_id=$domId) AND child_type=1001)" ;

  unless ($wrapDbh->{'EXECUTE'}->($sql)){
     return;
  }

  $wrapDbh->{'FINISH'}->();

  $sql="SELECT catch_all FROM mail_domain WHERE id=(SELECT child_id FROM parent_child WHERE parent_id =(SELECT child_id FROM parent_child WHERE child_type=1000 AND parent_type=2 AND parent_id=$domId) AND child_type=1001)";

  if ($wrapDbh->{'EXECUTE'}->($sql)){
    $ptrRow = $wrapDbh->{'FETCHROW'}->();

    $item->{'ADDCHILD'}->('catch-all',(@$ptrRow)[0]);

    $wrapDbh->{'FINISH'}->();
  }

  #Get all mailboxes

  $sql = "SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE child_type=1000 AND parent_id=$domId) AND child_type=1001) AND (child_type =1002)";

  my (@ids,$id);

  if ($wrapDbh->{'EXECUTE'}->($sql)) {

    while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
  	  push @ids,$ptrRow->[0];
    }

    $wrapDbh->{'FINISH'}->();

  }

  # Add postmaster mailbox

  $sql = "SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE child_type=1000 AND parent_id=$domId) AND child_type=1001";

  if (($wrapDbh->{'EXECUTE'}->($sql)) && ($ptrRow = $wrapDbh->{'FETCHROW'}->())) {

  	push @ids,$ptrRow->[0];

    $wrapDbh->{'FINISH'}->();

  }

  foreach $id (@ids) {
    createMailUserNode($item,$domId,$id);
  }

  $parent->{'ADDCHILD'}->($item);

}

sub createMailUserNode {
  my ($parent,$domainId,$mailboxId)=@_;

  my $mailUserNode=makeXmlNode('mailuser');

  my $sql="SELECT email,password FROM mailboxes WHERE id=$mailboxId";

  my ($ptrHash,$ptrRow);
  ($wrapDbh->{'EXECUTE'}->($sql) && ($ptrHash = $wrapDbh->{'FETCHHASH'}->()))
	or die "Cannot retrieve info for mailuser $mailboxId";
  my %info = %$ptrHash;

  $wrapDbh->{'FINISH'}->();

  $mailUserNode->{'ATTRIBUTE'}->('name',$info{'email'});

  my $mailName=$info{'email'};

  &createPasswordNode($mailUserNode,$info{'password'},'plain');

#  MAILBOX
  my $mailBoxNode=makeXmlNode('mailbox');
	#Mdir or Mbox
  $mailBoxNode->{'ATTRIBUTE'}->('type','mdir');
	#quota
  $sql="SELECT amount FROM resource_amount WHERE id=(SELECT child_id FROM parent_child WHERE child_type=1008 AND parent_id=$mailboxId)";

  if (($wrapDbh->{'EXECUTE'}->($sql) && ($ptrRow = $wrapDbh->{'FETCHROW'}->()))){

  $mailUserNode->{'ATTRIBUTE'}->('mailbox-quota', (@$ptrRow)[0]*$megabyte);

  $wrapDbh->{'FINISH'}->();

}

	#cid
  my $pathToMdir = getPathToMdir($domainId,$mailName);
  my $filename;

  if (-d $pathToMdir) {
  	$filename = makeDumpFile ("$dumpDir/$mailName\@$domainId\_mbox",$pathToMdir);
  }

  $mailBoxNode->{'ATTRIBUTE'}->('cid',$filename);

  $mailUserNode->{'ADDCHILD'}->($mailBoxNode);

  #ALIAS
  createMailAlias($mailUserNode,$domainId,$mailName);

  #FORWARD
  createMailForward($mailUserNode,$domainId,$mailName);
  #RESPONDER
  createMailResponder($mailUserNode,$domainId,$mailName);

  $parent->{'ADDCHILD'}->($mailUserNode);

}

sub createMailResponder {
  my ($parent,$domId,$mailName)=@_;
  my ($sql,$ptrRow);

  $sql="SELECT count(*) FROM responders WHERE id IN (SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE child_type=1000 AND parent_id=$domId) AND child_type=1001) AND (child_type =1005))";

  ($wrapDbh->{'EXECUTE'}->($sql) && ($ptrRow = $wrapDbh->{'FETCHROW'}->()))
        or die "Cannot retrieve amout of responders for $domId ";
  if ($ptrRow->[0] == 0) {
    return;
  }

  $wrapDbh->{'FINISH'}->();

  $sql="SELECT local_email,foreign_email,subject,message FROM responders WHERE id IN (SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE child_type=1000 AND parent_id=$domId) AND child_type=1001) AND (child_type =1005))";

  $wrapDbh->{'EXECUTE'}->($sql) or die "Cannot retrieve mail responders for $domId ";

  my $autorespondersNode = makeXmlNode('autoresponders');
  $autorespondersNode->{'ATTRIBUTE'}->('enabled', 'true');

  while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
    if ($ptrRow->[0] eq $mailName) {
      createAutoResponderNode($autorespondersNode,$ptrRow->[1],$ptrRow->[2],$ptrRow->[3]);
    }
    $parent->{'ADDCHILD'}->($autorespondersNode);
  }

}

sub createMailAlias {
  my ($parent,$domId,$mailName)=@_;
  my ($sql,$ptrRow,%aliases,$temp);

  $sql="SELECT count(*) FROM mail_aliases WHERE id IN (SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE child_type=1000 AND parent_id=$domId) AND child_type=1001) AND (child_type =1006))";

  ($wrapDbh->{'EXECUTE'}->($sql) && ($ptrRow = $wrapDbh->{'FETCHROW'}->()))
        or die "Cannot retrieve amout of aliases for $domId ";
  if ($ptrRow->[0] == 0) {
    return;
  }

  $wrapDbh->{'FINISH'}->();

  $sql="SELECT email_local,email_foreign FROM mail_aliases WHERE id IN (SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE child_type=1000 AND parent_id=$domId) AND child_type=1001) AND (child_type =1006))";

  $wrapDbh->{'EXECUTE'}->($sql) or die "Cannot retrieve mail aliases for $domId ";

  while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
  	$aliases{$ptrRow->[0]} = $ptrRow->[1];
  }
  foreach $temp (keys %aliases) {
  	if ($aliases{$temp} eq $mailName) {
		my $item = makeXmlNode('alias',$temp);
		$parent->{'ADDCHILD'}->($item);
	}
  }
  $wrapDbh->{'FINISH'}->();
}

sub createMailForward {
  my ($parent,$domId,$mailName)=@_;
  my ($sql,%forwards,$ptrRow,$temp,@dest,$i);

  $sql="SELECT count(*) FROM mail_forwards WHERE id IN (SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE child_type=1000 AND parent_id=$domId) AND child_type=1001) AND (child_type =1004))";
  ($wrapDbh->{'EXECUTE'}->($sql) && ($ptrRow = $wrapDbh->{'FETCHROW'}->()))
        or die "Cannot retrieve amout of forwards for $domId ";
  if ($ptrRow->[0] == 0) {
    $parent->{'ATTRIBUTE'}->('mailgroup-enabled','false');
    return;
  }
  $wrapDbh->{'FINISH'}->();

  $sql="SELECT email_local,email_foreign FROM mail_forwards WHERE id IN (SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE child_type=1000 AND parent_id=$domId) AND child_type=1001) AND (child_type =1004))";

  $wrapDbh->{'EXECUTE'}->($sql) or die "Cannot retrieve mail forwards for $domId ";

  while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
  	$forwards{$ptrRow->[0]} = $ptrRow->[1];
  }

  my $item;

  foreach $temp (keys %forwards) {
  	if ($temp eq $mailName) {
  	    @dest=split(/\;/,$forwards{$temp});
	   	if(scalar(@dest)==1) {
			#todo: make redirect true
			$parent->{'ATTRIBUTE'}->('mailgroup-enabled','false');
			$item = makeXmlNode('redirect',(@dest)[0]);
			$parent->{'ADDCHILD'}->($item);
		} else {
			$parent->{'ATTRIBUTE'}->('mailgroup-enabled','true');
			foreach $i (@dest) {
			  $item = makeXmlNode('mailgroup-member',$i);
			  $parent->{'ADDCHILD'}->($item);
			}
		}
	}
	$parent->{'ATTRIBUTE'}->('mailgroup-enabled','false');
  }
  $wrapDbh->{'FINISH'}->();

}

sub createAutoResponderNode {
  my ($parent,$foreignEmail,$subject,$message)=@_;
  my $item=makeXmlNode('autoresponder');
  $item->{'ATTRIBUTE'}->('name','autoresponder');
  $item->{'ATTRIBUTE'}->('redirect',$foreignEmail) if defined($foreignEmail);

  my $encodedSubj = $wrapBase64->{'ENCODE'}->($subject);

  $item->{'ATTRIBUTE'}->('subject',$encodedSubj);

  my $value = $wrapBase64->{'ENCODE'}->($message);
  my $textNode = makeXmlNode ('text',$value);
  $item->{'ADDCHILD'}->($textNode);

  $parent->{'ADDCHILD'}->($item);
}

sub createDNSrecords {
  my ($root,$domId)=@_;
  my ($sql,$ptrHash,$dnsrec,$ptrRow);

  my $item = makeXmlNode('dns');
  
  # get A records for domain.
  $sql="SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE child_type=8 AND parent_id=$domId) AND child_type=3002";

  my (@arecords,$arecord);
  if ($wrapDbh->{'EXECUTE'}->($sql)){
    while ($ptrRow = $wrapDbh->{'FETCHROW'}->()){
  	  push @arecords,$ptrRow->[0];
    }
    $wrapDbh->{'FINISH'}->();
    foreach $arecord (@arecords) {
  	  createRecord($item,$arecord);
    }
  }

  # get CNAME records for domain.
  $sql="SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE child_type=1000 AND parent_id=$domId) AND child_type=3006";

  my (@crecords,$crecord);
  if ($wrapDbh->{'EXECUTE'}->($sql)) {

    while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
  	  push @crecords,$ptrRow->[0];
    }
    
    foreach $crecord (@crecords) {
  	  createRecord($item,$crecord);
    }

    $wrapDbh->{'FINISH'}->();
  }
  
  # get MX records for domain
  $sql="SELECT child_id FROM parent_child WHERE parent_id=(SELECT child_id FROM parent_child WHERE child_type=1000 AND parent_id=$domId) and child_type=1007";

   my (@mrecords,$mrecord);
   if ($wrapDbh->{'EXECUTE'}->($sql)) {

     while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
  	   push @mrecords,$ptrRow->[0];
     }

     foreach $mrecord (@mrecords) {
       createRecord($item,$mrecord);
     }

     $wrapDbh->{'FINISH'}->();
  }

  # get NS records for domain
  $sql = "SELECT name,master,slave1 FROM dns_zones WHERE id = (SELECT child_id FROM parent_child WHERE parent_id=$domId and child_type=3001)";

   my (@nsrecords,$nsrecord);
   if ($wrapDbh->{'EXECUTE'}->($sql)) {
     $ptrHash = $wrapDbh->{'FETCHHASH'}->();
     my %info = %$ptrHash;
     push @nsrecords,$info{'master'};
     push @nsrecords,$info{'slave1'};
     foreach $nsrecord (@nsrecords) {
       createNsRecord($item,$info{'name'},$nsrecord);
     }
     $wrapDbh->{'FINISH'}->();
   }
   $root->{'ADDCHILD'}->($item);
}

sub createNsRecord {
  my ($root,$domain,$recId)=@_;
  my $item = makeXmlNode('dnsrec');
  my $sql = "SELECT name FROM l_server WHERE id=$recId";
  if ($wrapDbh->{'EXECUTE'}->($sql)){
    my $ptrRow=$wrapDbh->{'FETCHROW'}->();
    $item->{'ATTRIBUTE'}->('type','NS');
    $item->{'ATTRIBUTE'}->('src',$domain);
    $item->{'ATTRIBUTE'}->('dst',$ptrRow->[0]);
    $wrapDbh->{'FINISH'}->();
  }
  $root->{'ADDCHILD'}->($item);
}

sub createRecord {
  my ($root,$recId)=@_;
  my ($item,$sql,$ptrRow);

  $sql = "SELECT count(*) FROM dns_records WHERE id=$recId";
  ($wrapDbh->{'EXECUTE'}->($sql) && ($ptrRow = $wrapDbh->{'FETCHROW'}->()))
	or die "Cannot retrieve amount of record $recId";
  if ($ptrRow->[0] == 0) {
  	return;
  }

  $item = makeXmlNode('dnsrec');
  $sql = "SELECT * FROM dns_records WHERE id=$recId";
  my $ptrHash;

  ($wrapDbh->{'EXECUTE'}->($sql) && ($ptrHash = $wrapDbh->{'FETCHHASH'}->()))
	or die "Cannot retrieve info for record $recId";
  my %info = %$ptrHash;

  $item->{'ATTRIBUTE'}->('type',$info{'type'});
  $item->{'ATTRIBUTE'}->('src',$info{'name'});
  $item->{'ATTRIBUTE'}->('dst',$info{'data'});

  if ($info{'type'} eq 'MX') {
    $item->{'ATTRIBUTE'}->('opt',$info{'pref'});
  }
  $root->{'ADDCHILD'}->($item);

}

sub readIpInfo {
  my $sql = "SELECT DISTINCT ip AS data,r_type FROM l_server_ips";
  if ($wrapDbh->{'EXECUTE'}->($sql)) {
    while (my $ptrHash = $wrapDbh->{'FETCHHASH'}->()) {
      my $ip = $ptrHash->{'data'};
      $exclIp{$ip} = ($ptrHash->{'r_type'} == 8);
    }
  }
  $wrapDbh->{'FINISH'}->();
}

sub createAllDatabases {
  my ($parent,$accountId)=@_;
  my ($sql,$ptrRow,$item,$database,@databases,@realdbs,$realdb);

  $sql="SELECT db_name FROM mysqldb WHERE parent_id IN (SELECT child_id FROM parent_child WHERE account_id IN (SELECT account_id FROM user_account,users WHERE user_account.user_id=users.id AND users.reseller_id=$accountId) AND child_type=6000)";

  if ($wrapDbh->{'EXECUTE'}->($sql)) {
  	while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
  		push @databases,$ptrRow->[0];
	}
  }
  $wrapDbh->{'FINISH'}->();

  foreach $database (@databases){
   	next unless (-d "/var/lib/mysql/".$database );
	push @realdbs,$database;
  }

  if (scalar(@realdbs) == 0) {
  	return;
  }

  $item = makeXmlNode ('domain');
  my $dbDomainName = "databases-migrated-fromHsphere".$accountId.".fake-domain.net";
  $item ->{'ATTRIBUTE'}->('name',$dbDomainName);
  $sql = "SELECT DISTINCT dns_records.data,l_server_ips.r_type FROM l_server_ips,dns_records WHERE type='A' AND id IN (SELECT child_id FROM parent_child WHERE parent_id IN (SELECT child_id FROM parent_child WHERE child_type=8 AND parent_id IN (SELECT child_id FROM parent_child WHERE account_id IN (SELECT account_id FROM user_account WHERE user_id IN (SELECT id FROM users WHERE reseller_id=$accountId)) AND (child_type=2 OR child_type=34))) AND child_type=3002)  AND l_server_ips.ip=dns_records.data";

  if ($wrapDbh->{'EXECUTE'}->($sql)) {
    $ptrRow= $wrapDbh->{'FETCHROW'}->();
    $item->{'ADDCHILD'}->(makeIpNode($ptrRow->[0]));
  }
  $wrapDbh->{'FINISH'}->();

  foreach $realdb (@realdbs) {
  	createDbNode($item,$realdb);
  }
  $parent->{'ADDCHILD'}->($item);
}

sub createDbNode {
  my ($parent,$dbName)=@_;
  my $script = '/hsphere/shared/scripts/mysql-db-users';

  my $item = makeXmlNode ('database');
  my ($sql,$ptrRow,$dbUserNode,%dbAccess);

  $item->{'ATTRIBUTE'}->('name',$dbName);
  $item->{'ATTRIBUTE'}->('type','mysql');
  $item->{'ATTRIBUTE'}->('version', Db::MysqlUtils::getVersion());

  my ($dbUser,@dbUsers);
  my $users = `$script $dbName`;
  @dbUsers = join("\n",$users);
  foreach $dbUser (@dbUsers) {
    chomp $dbUser;
    $dbUserNode = makeXmlNode('dbuser');
    $dbUserNode ->{'ATTRIBUTE'}->('name',$dbUser);
    $sql = "SELECT password FROM mysql_users WHERE login = '$dbUser'";
    if (($wrapDbh->{'EXECUTE'}->($sql)) && ($ptrRow = $wrapDbh->{'FETCHROW'}->())){
      createPasswordNode($dbUserNode,$ptrRow->[0],'plain');
    }
    $dbAccess{$dbUser}=$ptrRow->[0];
    $wrapDbh->{'FINISH'}->();

  }

  my $dumpDbFile = makeDumpDb ($dbName,'mysql',$dbUsers[0],$dbAccess{$dbUsers[0]},'localhost');

  $item->{'ADDCHILD'}->($dbUserNode);
  $item->{'ATTRIBUTE'}->('cid',$dumpDbFile);
  $parent->{'ADDCHILD'}->($item);
}

sub createPasswordNode {
  my ($parent, $password, $type) = @_;

  my $item = &makeXmlNode('password', $password);
  $item->{'ATTRIBUTE'}->('type', $type);
  $parent->{'ADDCHILD'}->($item);
}

sub createPinfoNode {
  my ($parent, $attname, $attvalue) = @_;

  my $item = &makeXmlNode('pinfo',$attvalue);
  $item->{'ATTRIBUTE'}->('name',$attname) if defined($attname) ;
  $parent->{'ADDCHILD'}->($item);
}

sub createLimitNode {
  my ($parent, $name, $value, $unlimited) = @_;

  if ($unlimited) {
	return;
  }

  my $item = &makeXmlNode("limit", $value);
  $item->{'ATTRIBUTE'}->('name', $name);
  $parent->{'ADDCHILD'}->($item);
}

sub getPath {
  my ($domainId) = @_;
  my ($ptrRow);
  my $sql ="SELECT username FROM users WHERE id=(SELECT user_id FROM user_account WHERE account_id=(SELECT DISTINCT account_id FROM parent_child WHERE parent_id=$domainId AND (parent_type=2 OR parent_type=34 OR parent_type=31 )))";
  ($wrapDbh->{'EXECUTE'}->($sql) && ($ptrRow = $wrapDbh->{'FETCHROW'}->()))
        or die "Cannot retrieve user info from DB";
  my $username = (@$ptrRow)[0];
  $wrapDbh->{'FINISH'}->();
  return $username;
}

sub getDbPassword {
  my $file = shift || '/hsphere/local/home/cpanel/shiva/psoft_config/hsphere.properties';
  my $password = `cat $file | grep ^DB_PASSWORD | awk '{print \$3}'`;
  chomp $password;
  return $password;
}

sub getHostName {
  my $hostname = `hostname`;
  chomp $hostname;
  return $hostname;
}

sub createDomUserNode {
  my ($parent,$domId,$ptrHash)=@_;
  my ($ptrRow,%info);

  %info=%$ptrHash;

  my $item = makeXmlNode('domainuser');
  my $sql = "SELECT password FROM users WHERE id=(SELECT user_id FROM user_account WHERE account_id=(SELECT DISTINCT account_id FROM parent_child WHERE parent_id=$domId AND (parent_type=2 OR parent_type=34)))";

  if (($wrapDbh->{'EXECUTE'}->($sql)) && ($ptrRow = $wrapDbh->{'FETCHROW'}->())){
    createPasswordNode($item,$ptrRow->[0],'plain');
  }

  $wrapDbh->{'FINISH'}->();

  $item->{'ATTRIBUTE'}->('contact',$info{'name'}.$info{'last_name'});
  $item->{'ATTRIBUTE'}->('status','true');

  createPinfoNode($item,'country',$info{'country'});
  createPinfoNode($item,'zip',$info{'postal_code'});
  createPinfoNode($item,'city',$info{'city'});
  createPinfoNode($item,'address',$info{'address1'});
  createPinfoNode($item,'email',$info{'email'});

  $info{'phone'} =~s/\.//g;

  createPinfoNode($item,'phone',$info{'phone'});
  createPinfoNode($item,'company',$info{'company'});
  createPinfoNode($item,'state',$info{'state'});

  $parent->{'ADDCHILD'}->($item);

}

sub getPathToMdir {
  my ($domId,$mailName)=@_;
  my ($sql,$ptrRow,$domainName);
  $sql = "SELECT name FROM domains WHERE id=$domId";
  if (($wrapDbh->{'EXECUTE'}->($sql))&&($ptrRow = $wrapDbh->{'FETCHROW'}->())) {
    $domainName=$ptrRow->[0];
  }
  $wrapDbh->{'FINISH'}->();

  my $path = "/home/hsphere/local/var/vpopmail/domains/".$domainName."/".$mailName."/Maildir";
  unless (-d $path) {
    $path = "/home/hsphere/local/var/vpopmail/domains/0/".$domainName."/".$mailName."/Maildir";
  }

  return $path;

 }

sub printHelp {

  print <<"HELP";

Usage:
$0 <options>

Options:
-s |--get-status           get status of the agent
-dc|--dump-accounts=<list> a coma separated list of resellers to dump
-dd|--dump-domains=<list>  a coma separated list of domains to dump
-da|--dump-all             make a full dump

-lc|--get-content-list     get list of content files
-nc|--no-content           do not make content files
-nz|--no-compress          do not compress content files

-h |--help                 this help

HELP
}
