# Copyright 1999-2015. Parallels IP Holdings GmbH. All Rights Reserved.
package ContentDumper;

use strict;

use AgentConfig;
use Logging;
use Dumper;
use File::Basename;
use File::Path;
use Imapd2Md;
use XmlNode;
use Fcntl qw/O_RDONLY O_BINARY/;
use File::Spec;
use File::Temp qw/tempdir/;

use constant MAILBOX_MBX => 1;
use constant MAILBOX_MBOX => 2;

use constant MAILBOX_MAGIC_LENGTH => 5;

use constant BIND_BINARY_ROOT => '/usr/lib/opcenter/bind';

sub getPHostingContent {
  my ($base, $domainName) = @_;

  Logging::trace("Getting phosting content for domain '$domainName' ... ");

  my $wwwRoot = "/var/www/html";
  my $cgiRoot = "/var/www/cgi-bin";

  my @children;
  my $wwwContent = _packUpSinglePhostingDirectory($base, $domainName, $wwwRoot, "docroot", _makeExcludedHtaccessList($domainName, $wwwRoot));
  push @children, $wwwContent if (ref($wwwContent) =~ /XmlNode/);
  my $cgiContent = _packUpSinglePhostingDirectory($base, $domainName, $cgiRoot, "cgi", _makeExcludedHtaccessList($domainName, $cgiRoot));
  push @children, $cgiContent if (ref($cgiContent) =~ /XmlNode/);

  if ( @children ) {
    my $contentNode = XmlNode->new('content',
      'children' => \@children
    );
    return $contentNode;
  }

  Logging::debug("No phosting content found for domain '$domainName'.");
}

sub getSubdomainPHostingContent {
  my ($base, $domainName, $pHostingNode, $subdomainName, $wwwdir, $cgidir) = @_;
  my @children;

  Logging::trace("Getting SubdomainPhosting content for domain '$domainName' with subdomain '$subdomainName' ... ");

  if (defined($cgidir) && $cgidir !~ m/^\s*$/)
  {
    my $cgiContent = _packUpSingleSubdomainDirectory($base, $domainName, $subdomainName, $cgidir, "cgi");
        # can use '_makeExcludedHtaccessList($domainName, $wwwdir)' for fill last argument (@exclude_list),
        # but subdomains not using Protect Directories at webpanel management.
        # So, keep it empty.

    if (ref($cgiContent) =~ /XmlNode/) {
        push @children, $cgiContent;
        $pHostingNode->setAttribute('cgi_bin_mode', "www-root");     # this is realy require for Plesk, for access to "cgi-bin"
                                                                     # directories for ftpusers/owners of subdomains.
                                                                     # And yes, it's UGLY hack for this place, but it simplest way.
    }
  }

  my $wwwContent = _packUpSingleSubdomainDirectory($base, $domainName, $subdomainName, $wwwdir, "docroot",
    ($cgidir eq "$wwwdir/cgi-bin" ? "./cgi-bin" : undef ) );        # same thing about _makeExcludedHtaccessList() see above
  push @children, $wwwContent if (ref($wwwContent) =~ /XmlNode/);

  if ( @children ) {
    my $contentNode = XmlNode->new('content',
      'children' => \@children
    );
    return $contentNode;
  }

  Logging::debug("No Subdomain phosting content found for domain '$domainName' subdomain '$subdomainName' .");
}


sub getAnonFtpContent {
  my ($base, $domainName) = @_;

  my @children;
  my $pubContent = _packUpSinglePhostingDirectory($base, $domainName, "/var/ftp/pub", "ftp_pub");
  push @children, $pubContent if (ref($pubContent) =~ /XmlNode/);
  my $incomingContent = _packUpSinglePhostingDirectory($base, $domainName, "/var/ftp/uploads", "ftp_incoming");
  push @children, $incomingContent if (ref($incomingContent) =~ /XmlNode/);

  if (@children) {
    my $contentNode = XmlNode->new('content',
      'children' => \@children
    );
    return $contentNode;
  } else {
    Logging::debug("No anonymous FTP content found for domain '$domainName'.");
    return undef;
  }
}

sub getWebUserSiteContent {
  my ($base, $domainName, $webUserName) = @_;

  my $contentNode = _packUpSingleWebUserDirectory($base, $domainName, $webUserName, "/home/$webUserName/public_html", "webuser_home");
  if (ref($contentNode) =~ /XmlNode/) {
    $contentNode->setAttribute('offset', "web_users/$webUserName");
    return XmlNode->new('content',
      'children' => [$contentNode]
    );
  }

  Logging::debug("No site content found for web user $webUserName at domain $domainName");
}

sub getMailboxType($) {
  my ($mailboxFilename) = @_;
  sysopen(MAILBOX, $mailboxFilename, O_RDONLY | O_BINARY) or return undef;
  my $magic = '';
  my $bytesRead = read(MAILBOX, $magic, MAILBOX_MAGIC_LENGTH);
  close(MAILBOX);

  if( $bytesRead != MAILBOX_MAGIC_LENGTH ) {
    return undef;
  }

  if ($magic eq 'From ') {
    return MAILBOX_MBOX;
  } elsif ($magic eq '*mbx*') {
    return MAILBOX_MBX;
  } else {
    return undef;
  }
}

# beware: caller is responsible for cleanup
sub _getTempDirectory {
  my $baseTempDirectory = File::Spec->catdir(AgentConfig::getSharedDir(), 'tmp');
  if (! -d $baseTempDirectory) {
    mkdir($baseTempDirectory, 0077) or die "Can't create base temporary directory '$baseTempDirectory': $!";
  }
  return tempdir(DIR => $baseTempDirectory);
}

sub getMailBoxContent {
  my ($base, $domainName, $userName, $userInfo) = @_;

  # Inbox is stored in mailbox format in var/spool/mail
	my $mailHome = Dumper::getFilesystemByDomainName($domainName) . "/home/$userName";
  my $convertedMailDir = _getTempDirectory();
  unless (-d $convertedMailDir) {
    die "Cannot create folder to convert mail messages!";
  }

  my $inboxDirectory = Dumper::getFilesystemByDomainName($domainName) . "/var/spool/mail/";
  my $inboxFile = $inboxDirectory . $userName;
  Logging::debug("Dumping mailbox contents: '$inboxFile'");
  my $mb2mdPath = dirname(__FILE__) . '/shared_legacy/mb2md.pl';
  if (-f $inboxFile) {
    # mb2md.pl was taken from http://batleth.sapienti-sat.org/projects/mb2md/mb2md-3.20.pl.gz
    `/usr/bin/perl $mb2mdPath -s $inboxFile -d $convertedMailDir`;
  }

  # all other IMAP-directories (except Inbox) are stored in IMAP format in user's home directory
  my @mail;
  my $mailBoxList = "$mailHome/.mailboxlist";
  if (-f $mailBoxList) {
    open MAILBOXLIST, "<$mailBoxList";
    binmode MAILBOXLIST;
    while (<MAILBOXLIST>) {
      chomp;
      if (-f "$mailHome/$_") {
        push @mail, $_;
      }
    }
  }
  if (@mail) {
    foreach my $mailFolder(@mail) {
      Logging::debug("Converting $userName\@$domainName mail folder '$mailFolder'");
      my $sourceMailbox = File::Spec->catfile($mailHome, $mailFolder); 
      my $destPath = $mailFolder;
      $destPath =~ s/^mail//g;
      $destPath =~ s/^\///g;
      $destPath =~ s/\./_/g;
      $destPath =~ s/\//\./g;
      $destPath = ".$destPath";

      my $mailboxType = getMailboxType($sourceMailbox);
      if (defined($mailboxType) and $mailboxType == MAILBOX_MBX) {
        my $conversionResult = Imapd2Md::convertit( $sourceMailbox, "$convertedMailDir/$destPath" );
        if ($conversionResult) {
          Logging::info("Mailbox '$sourceMailbox' converted unsuccessfully: '$conversionResult'");
        }
      } elsif (defined($mailboxType) and $mailboxType == MAILBOX_MBOX) {
        `/usr/bin/perl $mb2mdPath -s '$sourceMailbox' -d $convertedMailDir/$destPath`; 
      } else {
        Logging::debug("Mailbox '$sourceMailbox' is neither mbx not mbox, skipping");
      }
      system("touch", "$convertedMailDir/$destPath/maildirfolder") if -e "$convertedMailDir/$destPath";
    }
  }

  my %options;
  $options{'directory'} = $convertedMailDir;
  $options{'checkEmptyDir'} = 1;
  $options{'include_hidden_files'} = 1;
  my $domainId = Dumper::getSiteIdByDomain($domainName);

  my $contentFileName = $base->{namecreator}->getMailnameDstFile($userName, $domainName, $domainId);
  my $mailboxNode = $base->addContent('mailbox', $contentFileName, %options);
  if (defined($convertedMailDir)) {
    unless (rmtree($convertedMailDir)) {
      Logging::warning("Unable to remove temporary dir '$convertedMailDir' after migration of mailbox '$userName\@$domainName'. Please remove it manually.");
    }
  }
  if (ref($mailboxNode) =~ /XmlNode/) {
    return [ XmlNode->new('content', 'children' => [$mailboxNode]) ];
  }

  Logging::debug("No content found for $userName\@$domainName maildir.");
  return undef;
}

sub getDnsZoneSoa {
  my ($domainName) = (@_);

  my $listSoaBinary = File::Spec->catfile(BIND_BINARY_ROOT, 'list_soa');
  if (open(SOA, "$listSoaBinary $domainName |")) {
    my %soa = ();
    while (my $line = <SOA>) {
      chomp($line);
      my ($name, $value) = split(' ', $line);
      $soa{$name} = $value;
    }
    close(SOA);
    return \%soa;
  } else {
    Logging::warning("Cannot get SOA for domain '$domainName': $!");
    return;
  }
}

sub getDnsZoneRecords {
  my ($domainName) = (@_);

  my $listDnsRecordsBinary = File::Spec->catfile(BIND_BINARY_ROOT, 'list_domain');
  if (open(DNSREC, "$listDnsRecordsBinary -all $domainName |")) {
    my @records = ();
    while (my $line = <DNSREC>) {
      chomp($line);
      my ($name, $type, @remain) = split(' ', $line);
      push @records, [$name, $type, @remain];
    }
    close(DNSREC);
    return @records;
  } else {
    Logging::warning("Cannot get resource records for domain '$domainName': $!");
    return;
  }
}

# In PPCPL, a .htaccess file becomes owned by root after site admin protects
# appropriate directory using CP. Such protected directories we report to Plesk
# separately. If we don't exclude them here, Plesk has problems importing such site -
# unable to remove these .htaccess files at restoration stage.
sub _makeExcludedHtaccessList {
  my ($domainName, $jailedPath) = @_;

  my $rootPath = Dumper::getFilesystemByDomainName($domainName) . $jailedPath;

  my @result;
  my $find = AgentConfig::findBin();
  my $searchedFile = '.htaccess';
  my $htaccessList;
  open($htaccessList, "$find '$rootPath' -name $searchedFile -type f -user root -printf \"./%P\n\" |");
  binmode $htaccessList;
  while (<$htaccessList>) {
    chomp $_;
    push @result, $_;
  }
  close $htaccessList;

  return \@result;
}

sub _packUpSinglePhostingDirectory {
  my ($base, $domainName, $jailedPath, $contentType, $excludedPaths) = @_;
  my $domainId = Dumper::getSiteIdByDomain($domainName);

  my $contentFileName = $base->{namecreator}->getPhostingDstFile($contentType, $domainName, $domainId);
  return _packUpSingleDirectory($base, $domainName, $jailedPath, $contentType, $contentFileName, $excludedPaths);
}

sub _packUpSingleSubdomainDirectory {
  my ($base, $domainName, $subdomainName, $jailedPath, $contentType, $excludedPaths) = @_;
  my $domainId = Dumper::getSiteIdByDomain($domainName);

  my $contentFileName = $base->{namecreator}->getSubdomainDstFile($contentType, $domainName, $subdomainName, $domainId);
  return _packUpSingleDirectory($base, $domainName, $jailedPath, $contentType, $contentFileName, $excludedPaths);
}

sub _packUpSingleWebUserDirectory {
  my ($base, $domainName, $webUserName, $jailedPath, $contentType, $excludedPaths) = @_;
  my $domainId = Dumper::getSiteIdByDomain($domainName);

  my $contentFileName = $base->{namecreator}->getWebUserDstFile($domainName, $webUserName, $domainId);
  return _packUpSingleDirectory($base, $domainName, $jailedPath, $contentType, $contentFileName, $excludedPaths);
}

# $excludedPaths must be specified relative to $jailedPath, and start with "./",
# otherwise tar won't match them and will include them into tarball.
# Examples of $contentFileName:
#   * $base->{namecreator}->getPhostingDstFile('cgi', $domainName)
#   * $base->{namecreator}->getSubdomainDstFile('docroot', $domainName, $subdomainName )
# full list of content filename creator functions can see at common/.../shared/Storage/ContentNameCreator.pm
sub _packUpSingleDirectory {
  my ($base, $domainName, $jailedPath, $contentType, $contentFileName, $excludedPaths) = @_;

  my $fullPath = Dumper::getFilesystemByDomainName($domainName) . "$jailedPath";
  my $contentNode;

  if (-d $fullPath) {
    Logging::debug("Dumping directory contents: '$fullPath'");
    my %options;
    $options{'checkEmptyDir'} = 1;
    $options{'directory'} = $fullPath;
    $options{'include_hidden_files'} = 1;
    $options{'exclude'} = $excludedPaths if ( ref($excludedPaths) =~ /ARRAY/ and @{$excludedPaths} );
    $contentNode = $base->addContent($contentType, $contentFileName, %options);
  }
  return $contentNode;
}

1;
