#!/usr/bin/perl
# Copyright 1999-2015. Parallels IP Holdings GmbH. All Rights Reserved.

use strict;
use Fcntl qw(O_RDONLY);

use XmlNode;

use vars qw(
             $dumpDir
             $workDir
             $ptrArgs
             $storage);
#
# Constants
#
my $kilobyte = 1024;
my $defaultraqdomains = 'defaultraqdomains';

my $vsitesDir = "/home/sites";
my $spoolDir = "/var/spool/mail";

#
# parse command line
#
my %arg_opts = ('--help|-h'=>'',
		'--get-status|-s'=>'',
		'--dump-accounts|-dc'=>'s',
		'--dump-domains|-dd'=>'s',
		'--dump-all|-da'=>'',
		'--get-content-list|-lc'=>'',
		'--no-content|-nc'=>'',
		'--no-compress|-nz'=>'',
		'--output|-o'=>'s',
		'--status-file|-sf'=>'s',
	       );

my %UIDB;
my %shadow;

#@@INCLUDE FILE="agent.include.pl"@@
if(-f 'agent.include.pl'){
  require 'agent.include.pl';
}else{
  require '../agent.include.pl';
}
#@@/INCLUDE@@

$ptrArgs = getArguments(\@ARGV, \%arg_opts);

my ($outPath,$dumpFile,$statusFile);

$workDir = AgentConfig::cwd();

$dumpFile = $workDir . '/dump.xml';
$statusFile = $workDir . '/dumping-status.xml';
$dumpDir = $workDir . '/pma';

my $objDumpStatus = &makeDumpStatus($ptrArgs->{'status-file'}||$statusFile);

#
# get MIME Base64 encoder
#
my $wrapBase64 = makeMIMEBase64();

my $version = getVersion();
if ($version) {
  eval "require Cobalt::Vsite";
  eval "require Cobalt::Stats";
  eval "require Cobalt::Group";
  eval "require Cobalt::Fpx";
  eval "require Cobalt::Ftp";
  eval "require Cobalt::List";
  eval "require Cobalt::User";
  eval "require Cobalt::Vacation";
  eval "require Cobalt::Email";
}

if (exists $ptrArgs->{'get-status'}) {
  if ($version) {
    printAgentStatus();
  } else {
	printWrongPlatform();
  }
  exit 0;
}

if(exists ($ptrArgs->{'dump-all'}) ||
   exists ($ptrArgs->{'dump-accounts'}) ||
   exists ($ptrArgs->{'dump-domains'})) {

  my ($root, @accounts, @domains, $ptrAccounts, $ptrDomains, $value);
  initStorage();
  getDumpDir($dumpDir);

  if (exists $ptrArgs->{'no-compress'}) {
	setCompress(0);
  }

  printToLog("Work dir: $workDir");
  printToLog("Dump file: $dumpFile");
  printToLog("Status file: " . $objDumpStatus->{'FILE'}->());

  if ($value = $ptrArgs->{'dump-accounts'}) {
  	if ($value eq "-") {
	  $value = <STDIN>;
	  chomp $value;
	}
    @accounts = split(/\s*,\s*/, $value);
    $ptrAccounts = \@accounts;
  }

  if ($value = $ptrArgs->{'dump-domains'}) {
  	if ($value eq "-") {
	  $value = <STDIN>;
	  chomp $value;
	}
    @domains = split(/\s*,\s*/,$value);
    $ptrDomains = \@domains;
  }
#
# generate a xml dump
#
  $root = getRaQ2Dump($ptrAccounts, $ptrDomains);
#
# print dump to output
#
  $storage->finish($root);
} elsif (exists $ptrArgs->{'get-content-list'}) {
  makeContentList();
} else {
  printHelp();
}

exit 0;

#
# end main
#
#==============================================================
#
# subroutines
#

sub openUIDB {
  tie(%UIDB, 'DB_File', '/var/cobalt/uidb', Fcntl::O_RDONLY, 0600);
}

sub closeUIDB {
  untie(%UIDB);
}

sub parseShadow {
  open SHADOW, "/etc/shadow";
  while (<SHADOW>) {
	my @userInfo = split /:/;
	if ($userInfo[1] ne '*') {
	  $shadow{$userInfo[0]} = $userInfo[1];
	}
  }
  close SHADOW;
}

sub getRaQ2Dump {
  my ($ptrAccounts, $ptrDomains) = @_;

  openUIDB();
  parseShadow();

  my $root = XmlNode->new('RaQ2dump', 'attributes'=>{'agent-name'=>'RaQ2'});
  my $client = dumpClient();

  if (ref($ptrDomains) =~ /ARRAY/) {
    $objDumpStatus->{'COUNTS'}->(0, scalar(@{$ptrDomains}));
    $objDumpStatus->{'PRINT'}->();
	dumpDomains($client, $ptrDomains);
  } else {
    my @domains = map {$_->[1]} Cobalt::Vsite::vsite_list();

    $objDumpStatus->{'COUNTS'}->(1, scalar(@domains));
    $objDumpStatus->{'ACCOUNT'}->('RaQ domains');
    $objDumpStatus->{'PRINT'}->();

    dumpDomains($client, \@domains);
  }

  $root->{'ADDCHILD'}->($client);

  closeUIDB();

  return $root;
}

sub dumpClient {
  my $clientNode = makeXmlNode('client');
  $clientNode->{'ATTRIBUTE'}->('name', 'raqdomains');
  $clientNode->{'ATTRIBUTE'}->('contact', 'RaQ domains');
  createPasswordNode($clientNode, '', 'plain');

  return $clientNode;
}

sub dumpDomains {
  my ($parent, $ptrDomains) = @_;

  my (%domainsInfo, $domain);
  foreach $domain (Cobalt::Vsite::vsite_list()) {
	$domainsInfo{$domain->[1]} = $domain;
  }

  foreach $domain (@{$ptrDomains}) {
    $objDumpStatus->{'DOMAIN'}->($domain);
    $objDumpStatus->{'PRINT'}->();

	if (exists $domainsInfo{$domain}) {
	  dumpDomain($parent, $domainsInfo{$domain});
	} elsif (exists $domainsInfo{'www.'.$domain}) {
	  dumpDomain($parent, $domainsInfo{'www.'.$domain});
	}
  }
}

sub dumpDomain {
  my ($parent, $domainInfo) = @_;

  my $www = 'false';
  if ($domainInfo->[1] =~ /^www./) {
	$domainInfo->[1] =~ s/^www.//;
	$www = 'true';
  }

  my $domainNode = makeXmlNode('domain');
  $parent->{'ADDCHILD'}->($domainNode);

  $domainNode->{'ATTRIBUTE'}->('ip', $domainInfo->[0]);
  $domainNode->{'ATTRIBUTE'}->('name', $domainInfo->[1]);
  $domainNode->{'ATTRIBUTE'}->('www', $www);

  # NO MORE INFO REQUIRED IN '--no-content' MODE
  if (exists $ptrArgs->{'no-content'}) {
	return;
  }

  my $domainHandle = $domainInfo->[2];

  my $stat_period = Cobalt::Stats::stats_get_www_report($domainHandle);
  if ($stat_period eq 'weekly' or $stat_period eq 'daily') {
	$domainNode->{'ATTRIBUTE'}->('stat-period', $stat_period);
  }

  if (-d "$vsitesDir/$domainHandle/web") {
	my $dump = makeDumpFile("$dumpDir/$domainHandle.web", "$vsitesDir/$domainHandle/web");
	if ($dump) {
	  $domainNode->{'ATTRIBUTE'}->('cid', $dump);
	}
  }

  dumpDomainLimits($domainNode, $domainHandle);
  dumpDomainPermissions($domainNode, $domainHandle);
  dumpAnonFtp($domainNode, $domainHandle);
  dumpUsers($domainNode, $domainInfo->[1], $domainHandle);
  dumpMailLists($domainNode, $domainInfo->[1], $domainHandle);
}

sub dumpDomainLimits {
  my ($parent, $domainHandle) = @_;

  my $disk_quota = (Cobalt::Group::group_get_quota($domainHandle))[1];
  if ($disk_quota && $disk_quota != 0) {
	my $limitNode = makeXmlNode('domain-limit', $disk_quota*$kilobyte);
	$limitNode->{'ATTRIBUTE'}->('name', 'quota');
	$parent->{'ADDCHILD'}->($limitNode);
  }

  my $user_quota = $UIDB{'maxUsers-' . $domainHandle};
  if ($user_quota && $user_quota != 0) {
	my $limitNode = makeXmlNode('domain-limit', $user_quota);
	$limitNode->{'ATTRIBUTE'}->('name', 'users');
	$parent->{'ADDCHILD'}->($limitNode);
  }
}

sub dumpDomainPermissions {
  my ($parent, $domainHandle) = @_;

  createPermNode($parent, 'domain', 'email', $UIDB{'emailDomain-' . $domainHandle} eq 'on');
  createPermNode($parent, 'domain', 'shell', $UIDB{'shell-' . $domainHandle} eq 'on');
  createPermNode($parent, 'domain', 'cgi', Cobalt::Vsite::vsite_get_cgis($domainHandle));
  createPermNode($parent, 'domain', 'ssi', Cobalt::Vsite::vsite_get_ssi($domainHandle));
  createPermNode($parent, 'domain', 'frontpage', Cobalt::Fpx::fpx_get_web($domainHandle));
}

sub dumpAnonFtp {
  my ($parent, $domainHandle) = @_;

  my @anonftp = Cobalt::Ftp::ftp_get_anonymous($domainHandle);

  if ($anonftp[0] eq '1') {
	my $anonFtpNode = makeXmlNode('anonftp');

	my $usersLimit = $anonftp[1] || $UIDB{'maxFtpUsers-' . $domainHandle} || 10;
	my $usersLimitNode = makeXmlNode('anonftp-limit', $usersLimit);
	$usersLimitNode->{'ATTRIBUTE'}->('name', 'users');
	$anonFtpNode->{'ADDCHILD'}->($usersLimitNode);

	my $incomingQuota = $anonftp[2] || 20;
	my $incomingQuotaNode = makeXmlNode('anonftp-limit', $incomingQuota*$kilobyte);
	$incomingQuotaNode->{'ATTRIBUTE'}->('name', 'incoming-quota');
	$anonFtpNode->{'ADDCHILD'}->($incomingQuotaNode);

	if (-d "$vsitesDir/$domainHandle/ftp/incoming") {
	  my $dump = makeDumpFile("$dumpDir/$domainHandle.ftp.incoming", "$vsitesDir/$domainHandle/ftp/incoming");
	  if ($dump) {
		$anonFtpNode->{'ATTRIBUTE'}->('cid', $dump);
	  }
	}
	if (-d "$vsitesDir/$domainHandle/ftp") {
	  my $dump = makeDumpFile("$dumpDir/$domainHandle.ftp", "$vsitesDir/$domainHandle/ftp", ".", "incoming");
	  if ($dump) {
		$anonFtpNode->{'ATTRIBUTE'}->('cid_incoming', $dump);
	  }
	}
	$parent->{'ADDCHILD'}->($anonFtpNode);
  }
}

sub dumpUsers {
  my ($parent, $domainName, $domainHandle) = @_;

  my @users = Cobalt::List::list_get_mem($domainHandle . '-users');

  my @admins = Cobalt::Vsite::vsite_admin_list($domainHandle);
  my @admins = sort @admins;
  my $adminName = @admins[0] if @admins > 0;

  my $user;
  foreach $user (@users) {
	dumpUser($parent, $domainHandle, $domainName, $user, $user eq $adminName);
  }

  unless ($adminName) {
	dumpSyntheticAdmin($parent, $domainHandle, $domainName);
  }
}

sub dumpUser {
  my ($parent, $domainHandle, $domainName, $user, $admin) = @_;

  my ($name, $fullname, $quota, , $shell) = Cobalt::User::user_show($user);

  my $userNode = makeXmlNode($admin ? 'admin' : 'user');

  if ($fullname) {
	$userNode->{'ATTRIBUTE'}->('fullname', $fullname);
  }

  $userNode->{'ADDCHILD'}->(makeSysUserNode($name, $shadow{$user}, 'encrypted'));

  my $quotaNode = makeXmlNode('user-limit', $quota*$kilobyte);
  $quotaNode->{'ATTRIBUTE'}->('name', 'quota');
  $userNode->{'ADDCHILD'}->($quotaNode);

  createPermNode($userNode, 'user', 'shell', $shell eq '/bin/bash');
  createPermNode($userNode, 'user', 'admin', $admin);
  createPermNode($userNode, 'user', 'frontpage', Cobalt::Fpx::fpx_get_web($user) == 1);

  if (-d "$vsitesDir/$domainHandle/users/$name/web") {
	my $dump = makeDumpFile("$dumpDir/$name"."@"."$domainHandle.web",
							"$vsitesDir/$domainHandle/users/$name/web");
	if ($dump) {
	  $userNode->{'ATTRIBUTE'}->('cid', $dump);
	}
  }

  dumpMail($userNode, $domainName, $domainHandle, $parent, $user);

  $parent->{'ADDCHILD'}->($userNode);
}

sub dumpSyntheticAdmin {
  my ($parent, $domainHandle, $domainName) = @_;

  my $name = createUserNameFromDomain($domainName);

  my $adminNode = makeXmlNode('admin');
  $adminNode->{'ATTRIBUTE'}->('fullname', $domainName . " admin");
  $adminNode->{'ADDCHILD'}->(makeSysUserNode($name, '', 'plain'));

  my $adminMailNode = makeXmlNode('mail');
  $adminMailNode->{'ATTRIBUTE'}->('mailname', $name);

  $adminNode->{'ADDCHILD'}->($adminMailNode);

  $parent->{'ADDCHILD'}->($adminNode);
}

sub dumpMail {
  my ($parent, $domainName, $domainHandle, $domainNode, $user) = @_;

  my $mailNode = makeXmlNode('mail');
  $mailNode->{'ATTRIBUTE'}->('mailname', $user);

  my @forwards = Cobalt::List::alias_get_vacationless($user);
  my $forward;
  foreach $forward (@forwards) {
	$mailNode->{'ADDCHILD'}->(makeXmlNode('forward', $forward));
  }

  my @aliases = Cobalt::Email::mail_virtuser_get_byuser($user);
  @aliases = map { s/(.+)\@.*$/$1/; $_; } @aliases;
  @aliases = grep !/^$user$/, @aliases;

  my ($alias, %uniquealiases);
  foreach $alias (@aliases) {
	$alias =~ s/^\./_/gi;
	$alias =~ s/\.$/_/gi;
	$uniquealiases{$alias} = 1;
  }

  foreach $alias (keys %uniquealiases) {
	if ($alias eq '@' . $domainName) {
	  $domainNode->{'ATTRIBUTE'}->('catch-all', $user . '@' . $domainName);
	} elsif ($alias =~ /@/) {
	  printToLog("Unable to parse mailalias '$alias', skipping");
	} else {
	  $mailNode->{'ADDCHILD'}->(makeXmlNode('alias', $alias));
	}
  }

  my $msg = $wrapBase64->{'ENCODE'}->(Cobalt::Vacation::vacation_get_message($user));
  my $autoresponderNode = makeXmlNode('autoresponder', $msg);
  my $autoresponderState = Cobalt::Vacation::vacation_get_on($user);
  $autoresponderNode->{'ATTRIBUTE'}->('status', $autoresponderState == '1' ? 'on' : 'off');

  if ($msg || $autoresponderState) {
	$mailNode->{'ADDCHILD'}->($autoresponderNode);
  }

  if (-f "$spoolDir/$user") {
	my $dump = makeDumpFile("$dumpDir/$user"."@"."$domainHandle.mail", $spoolDir, $user);
	if ($dump) {
	  $mailNode->{'ATTRIBUTE'}->('cid_inbox', $dump);
	}
  }

  $parent->{'ADDCHILD'}->($mailNode);
}

sub dumpMailLists {
  my ($parent, $domainName, $domainHandle) = @_;

  my @lists = Cobalt::List::list_sites_lists($domainHandle);

  my $list;
  foreach $list (@lists) {
	dumpMailList($parent, $domainHandle, $domainName, $list);
  }
}

sub dumpMailList {
  my ($parent, $domainHandle, $domainName, $list) = @_;

  my @members = Cobalt::List::list_get_mem($list . '_' . $domainHandle);

  my $mailListNode = makeXmlNode('maillist');
  $mailListNode->{'ATTRIBUTE'}->('name', $list . '_' . $domainHandle);

  my $member;
  foreach $member (@members) {
	if ($member =~ /@/) {
	  $mailListNode->{'ADDCHILD'}->(makeXmlNode('recipient', $member));
	} else {
	  $mailListNode->{'ADDCHILD'}->(makeXmlNode('recipient', $member . '@' . $domainName));
	}
  }

  $parent->{'ADDCHILD'}->($mailListNode);
}

sub createPermNode {
  my ($parent, $prefix, $name, $value) = @_;

  if ($value) {
	my $permNode = makeXmlNode($prefix . '-permission');
	$permNode->{'ATTRIBUTE'}->('name', $name);
	$parent->{'ADDCHILD'}->($permNode);
  }
}

sub printAgentStatus {
  my $root = makeXmlNode('agent-status');
  printXml($root, *STDOUT);
}

sub printWrongPlatform {
  my $outFh = *STDOUT;

  my $root = makeXmlNode('agentstatus');
  my $item = makeXmlNode('wrong-platform',
					   defined AgentConfig::iconvBin() ? '' : 'no iconv found on the source host');
  $root->{'ADDCHILD'}->($item);
  printXml($root, $outFh, 'agentstatus.dtd');
  return;
}

sub getVersion {
  my %bldHash = ('2700'=>'RaQ1',
				 '2799'=>'RaQ2',
				 '2800'=>'RaQ2',
				 '3000'=>'RaQ3',
				 '3001'=>'RaQ4',
				 '3100'=>'RaQ4',
				 '3500'=>'RaQXTR',
				 '3599'=>'RaQXTR',
				);

  my ($build, $version);

  if ((-T '/etc/build') && open (CONF, "</etc/build")) {
    while (<CONF>) {
      if (/(\d{4,4})/) {
		my $buildCode = $1;
		if (exists($bldHash{$1})) {
		  $build = $1;
		  $version = $bldHash{$1};
		  last;
		}
      }
    }
    close (CONF);
  }

  if ($version eq 'RaQ2') {
	return "RaQ2";
  }
}

sub printHelp {

  print <<"HELP";

Usage:
  $0 <options>

Options:
  -s |--get-status           get status of the agent
  -dc|--dump-accounts=<list> a coma separated list of resellers to dump
  -dd|--dump-domains=<list>  a coma separated list of customers to dump
  -da|--dump-all             make a full dump

  -lc|--get-content-list     get list of content files
  -nc|--no-content           do not make content files
  -nz|--no-compress          do not compress content files

  -h |--help                 this help

HELP
}
