#!/usr/bin/perl
# Copyright 1999-2015. Parallels IP Holdings GmbH. All Rights Reserved.

# use for dump of user's mysql databases
use vars qw($mysqlUserUser $mysqlUserServer $mysqlUserPw $storage);


use vars qw($dumpDir $ptrArgs $workDir);
use strict;

use AgentConfig;

use XmlNode;

use Status;

#
# Constants
#
my $megabyte = 1024*1024;
my $kilobit = 1024*8;

my $defaultraqdomains = 'defaultraqdomains';

#
# parameters for connection to database
#
my $dbUser = 'admin';
my $dbName = 'cobalt';
my $dbHost = 'localhost';
my $dbType = 'Pg';
#
# end parameters for connection to database
#

my %vSitePerms = ('cgi'=>'cgi','ssi'=>'ssi', 'ssl'=>'ssl',
		  'fpx'=>'frontpage','apop'=>'apop',
		  'php'=>'php','casp'=>'casp');
my %vSiteLimits = ('quota'=>'space', 'maxusers'=>'account');
my %userPerms = ('admin'=>'admin','shell'=>'shell',
		  'fpx'=>'frontpage','apop'=>'apop',
		);
my %userLimits = ('quota'=>'space');

#
# parse command line
#
my %arg_opts = ('--help|-h'=>'',
		'--config|-c'=>'s',
		'--get-status|-s'=>'',
		'--dump-accounts|-dc'=>'s',
		'--dump-domains|-dd'=>'s',
		'--dump-all|-da'=>'',
		'--get-content-list|-lc'=>'',
		'--no-content|-nc'=>'',
		'--no-compress|-nz'=>'',
		'--output|-o'=>'s',
		'--status-file|-sf'=>'s',
	       );

my $defaultdomainname = `hostname -f`;
chomp $defaultdomainname;
my $dbdomainname = "databases.from.$defaultdomainname";

my $defaultipaddr = `ifconfig eth0 | perl -ne "if(/inet addr:([\\d.]+)/) { print \\\$1; }"`;
chomp $defaultipaddr;

#@@INCLUDE FILE="agent.include.pl"@@
if(-f 'agent.include.pl'){
  require 'agent.include.pl';
}else{
  require '../agent.include.pl';
}
#@@/INCLUDE@@

$ptrArgs = &getArguments (\@ARGV,\%arg_opts);

my ($outPath,$dumpFile,$statusFile);

$workDir = AgentConfig::cwd();

$dumpFile = $workDir.'/dump.xml';
$statusFile = $workDir.'/dumping-status.xml';
$dumpDir = $workDir.'/pma';

my $objDumpStatus = &makeDumpStatus($ptrArgs->{'status-file'}||$statusFile);


#
# get MIME Base64 encoder
#
my $wrapBase64 = makeMIMEBase64();

my (%RaQ4Config);
my $rootName = 'RaQ4dump';
my $dtdName = 'RaQ4.dtd';

unless($ptrArgs->{'config'} = &loadMainConfig($ptrArgs->{'config'},\%RaQ4Config)){
  if(exists $ptrArgs->{'get-status'}){
    printAgentStatus();
    exit 0;
  }else{

    if(exists ($ptrArgs->{'dump-all'}) ||
       exists ($ptrArgs->{'dump-accounts'}) ||
       exists ($ptrArgs->{'dump-domains'})){

      my $root = XmlNode->new($rootName);

      if(ref($root)=~/HASH/ && ref($root->{'PRINT'})=~/CODE/){
		my $outFh = &openOutput($ptrArgs->{'output'});
		&printXml($root,$outFh,$dtdName);
		&closeOutput();
      }
    }

    die "Error: Raq config file '/etc/build' is not found\n";
  }
}


#
# get db connect
#

my $wrapDbh = getDbConnect($dbType, $dbUser, $RaQ4Config{'dbpassword'},
			   $dbName, $dbHost);
unless (ref($wrapDbh) =~ /HASH/ && ref($wrapDbh->{'EXECUTE'}) =~ /CODE/) {
  die "Error: can not connect to cobalt database\n";
}





#
# global variables
#
my ($ptrShadow,     ## poiner to hash of /etc/shadow
    $dirMajordomo   ## Majordomo home dir
   );

sub lazyInitPtrShadow {
  unless(ref($ptrShadow)=~/HASH/){
	my ($fileParser);
    $fileParser = &makeFileParser('/etc/shadow');
    $ptrShadow = $fileParser->{'PARSE'}->('KEYSEPARATOR'=>':', 'VALUESEPARATOR'=>':');
  }
}

my %usersInfo;

sub lazyInitUser {
  my ($userName, $dropAdmin) = @_;

  unless (exists $usersInfo{$userName}) {
	my ($sql, $userInfo);

	lazyInitPtrShadow();

	$sql = "SELECT * FROM users WHERE name='$userName'";
	$wrapDbh->{'EXECUTE'}->($sql);
	$userInfo = $wrapDbh->{'FETCHHASH'}->();
	$userInfo->{'quota'} *= $megabyte if $userInfo->{'quota'} =~ /^\d+$/;

	if ($dropAdmin) {
	  undef $userInfo->{'admin'};
	}

	$wrapDbh->{'FINISH'}->();

	$userInfo->{'shadow'} = $ptrShadow->{$userName}->[0];

	$usersInfo{$userName} = $userInfo;
  }
}

my %vsites;
my %cli2dom;

#
# end global variables
#


if(exists $ptrArgs->{'get-status'}){
  printAgentStatus();
}elsif(exists ($ptrArgs->{'dump-all'}) ||
       exists ($ptrArgs->{'dump-accounts'}) ||
       exists ($ptrArgs->{'dump-domains'})){

  my ($root,@accounts,@domains,$ptrAccounts,$ptrDomains, $value);
  initStorage();
  getDumpDir($dumpDir);

  if (exists $ptrArgs->{'no-compress'}) {
	setCompress(0);
  }

  &printToLog("Work dir: $workDir");
  &printToLog("Dump file: $dumpFile");
  &printToLog("Status file: ".$objDumpStatus->{'FILE'}->());


  if ($value = $ptrArgs->{'dump-accounts'}){
  	if ($value eq "-") {
		$value = <STDIN>;
		chomp $value;
	}
    @accounts = split(/\s*,\s*/,$value);
    $ptrAccounts = \@accounts;
  }

  if ($value = $ptrArgs->{'dump-domains'}){
  	if ($value eq "-") {
		$value = <STDIN>;
		chomp $value;
	}
    @domains = split(/\s*,\s*/,$value);
    $ptrDomains = \@domains;
  }
#
# generate a xml dump
#

  $root = &getRaQ4Dump($dumpDir, $ptrAccounts, $ptrDomains, $rootName);

#
# print dump to output
#
  $storage->finish($root);
}elsif(exists $ptrArgs->{'get-content-list'}){
  makeContentList();
}else{
  &printHelp();
}

exit 0;

#
# end main
#
#==============================================================
#
# subroutines
#

sub getRaQ4Dump {
  my ($dumpDir, $ptrAccounts, $ptrDomains, $rootName) = @_;

  my (%unvsites,$sql,$list,$ptrRow,$name,$domain,$item,$client,$clientNode);

  my $root = &makeXmlNode($rootName);

  $root->{'ATTRIBUTE'}->('build',$RaQ4Config{'build'});
  $root->{'ATTRIBUTE'}->('agent-name', 'RaQ4');

  unless(ref($wrapDbh)=~/HASH/ && ref($wrapDbh->{'EXECUTE'})=~/CODE/){
    $root->{'CONTENT'}->("Error: there is no connect to database");
    &printToError("Error: unable to connect to SQL database");
    return $root;
  }

  $sql = "select name,fqdn,hostname,domain from vsite";

  if($wrapDbh->{'EXECUTE'}->($sql)){
    while( $ptrRow = $wrapDbh->{'FETCHROW'}->()){
	  my $val;
      if($ptrRow->[1]){
		$val = $ptrRow->[1];
      }else{
		$val = $ptrRow->[2] . '.' .$ptrRow->[3];
      }

	  $unvsites{$val} = $ptrRow->[0];
	  $vsites{$ptrRow->[0]} = $val;
    }
  }
  $wrapDbh->{'FINISH'}->();

  delete $vsites{'default'};

  $vsites{'dbvsites'} = $dbdomainname;
  $unvsites{$dbdomainname} = 'dbvsites';

#
# generate clients - first admin from domain's users
#

  $sql = "SELECT v.name, MIN(u.name) FROM vsite v, users u ".
    "WHERE u.vsite=v.name AND (u.admin='t') GROUP by v.name";

  if ($wrapDbh->{'EXECUTE'}->($sql)) {
    while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
      ($name, $client) = @{$ptrRow};

	  # postgres sometimes returns only one row containing only NULLs. Worked around
	  unless (defined ($name) && defined ($client)) {
		next;
	  }

      $cli2dom{$client} = [];
	  push @{$cli2dom{$client}}, $name;
    }
  }
  $wrapDbh->{'FINISH'}->();

#
# generate default client
#
  my @vsitesList;
  $sql = "SELECT name FROM vsite";
  if ($wrapDbh->{'EXECUTE'}->($sql)) {
	while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
	  my ($vsitename) = @{$ptrRow};
	  push @vsitesList, (@{$ptrRow})[0];
	}
  }
  $wrapDbh->{'FINISH'}->();
  push @vsitesList, 'dbvsites';

  my $vsiteName;
  foreach $vsiteName (@vsitesList) {
	$sql = "SELECT COUNT(*) FROM users WHERE vsite = '" . $vsiteName . "' AND users.admin='t'";
	$wrapDbh->{'EXECUTE'}->($sql);
	my $count = (@{$wrapDbh->{'FETCHROW'}->()})[0];
	if ($count == 0 && $vsiteName ne 'default') {
	  push @{$cli2dom{$defaultraqdomains}}, $vsiteName;
	}
	$wrapDbh->{'FINISH'}->()
  }
#
# end generate clients
#

  if (ref($ptrAccounts) =~ /ARRAY/) {
    my $domainsCount = 0;
    my $clientsCount = 0;

    foreach $name(@{$ptrAccounts}) {
      if (defined $cli2dom{$name}) {
        $clientsCount++;
        $domainsCount += scalar(@{$cli2dom{$name}});
      }
    }
    $objDumpStatus->{'COUNTS'}->($clientsCount, $domainsCount);
    $objDumpStatus->{'PRINT'}->();

    foreach $name (@{$ptrAccounts}) {
      if (defined $cli2dom{$name}) {
        my $client = makeClientNode($name, 0, 1);
        foreach my $domid (@{$cli2dom{$name}}) {
          my $item = makeDomainNode($domid, $vsites{$domid}, 0, $name);
          $client->{'ADDCHILD'}->($item);
        }
        $root->{'ADDCHILD'}->($client);
      }
    }
  } elsif (ref($ptrDomains) =~ /ARRAY/) {
    $objDumpStatus->{'COUNTS'}->(0, scalar(@{$ptrDomains}));
    $objDumpStatus->{'PRINT'}->();

	my $client = makeClientNode($defaultraqdomains, 1, 0);

	foreach my $domain (@{$ptrDomains}) {
	  my $vSiteName = $unvsites{$domain} || $unvsites{'www.'.$domain};
	  my $domainNode = makeDomainNode($vSiteName, $domain, 1, $defaultraqdomains);
	  $client->{'ADDCHILD'}->($domainNode);
	}

	$root->{'ADDCHILD'}->($client);

  } else {
	my $domainsCount = 0;
	my $clientsCount = 0;

    foreach $name(keys %cli2dom){
      if (defined $cli2dom{$name}){
        $clientsCount++;
        $domainsCount += scalar(@{$cli2dom{$name}});
      }
    }
    $objDumpStatus->{'COUNTS'}->($clientsCount, $domainsCount);
    $objDumpStatus->{'PRINT'}->();

	foreach $name (keys %cli2dom) {
	  my $client = makeClientNode($name, 0, 1);
	  my $domid;
	  foreach $domid (@{$cli2dom{$name}}) {
		my $item = makeDomainNode($domid, $vsites{$domid}, 0, $name);
		$client->{'ADDCHILD'}->($item);
	  }

	  $root->{'ADDCHILD'}->($client);
	}
  }

  return $root;
}

sub makeClientNode {
  my ($client, $dropAdmins, $countClientInStatus) = @_;

  my $clientNode = makeXmlNode('client');
  $clientNode->{'ATTRIBUTE'}->('name', $client);

  if ($countClientInStatus) {
    $objDumpStatus->{'ACCOUNT'}->($client);
    $objDumpStatus->{'PRINT'}->();
  }

  if ($client eq $defaultraqdomains) {
	# here info is generated
	$clientNode->{'ADDCHILD'}->(makePasswordNode('', 'plain'));
  } else {
	# here info is taken from the corresponding admin
	lazyInitUser($client, $dropAdmins);
	my $userInfo = $usersInfo{$client};

	$clientNode->{'ATTRIBUTE'}->('contact', $userInfo->{'fullname'})
	  if $userInfo->{'fullname'};

	my $domName = $vsites{$cli2dom{$client}->[0]};
	$domName =~ s/^www\.//;

	$clientNode->{'ATTRIBUTE'}->('email', $client . '@' . $domName);

	$clientNode->{'ADDCHILD'}->(makePasswordNode($userInfo->{'shadow'}, 'encrypted'));
  }

  return $clientNode;
}


sub makeDomainNode {
  my ($vSiteName, $vSiteDomain, $dropAdmins, $clientName) = @_;

  my ($root,$sql,$ptrRow,$ptrHash,%vSite);
  my ($www,$dumpFile,$item,$key,$attrName);

  printToLog("Dumping $vSiteDomain domain...");
  $objDumpStatus->{'DOMAIN'}->($vSiteDomain);
  $objDumpStatus->{'PRINT'}->();

  if($vSiteDomain=~s/^www\.//){
    $www = 'true';
  }else{
    $www = 'false';
  }

  $root = &makeXmlNode('domain');

  $root->{'ATTRIBUTE'}->('name',$vSiteDomain);
  $root->{'ATTRIBUTE'}->('www',$www);

  if ($vSiteDomain eq $dbdomainname) {
	return makeDbDomainNode($root);
  }

  $sql = "select * from vsite where name='$vSiteName'";
  unless($wrapDbh->{'EXECUTE'}->($sql)){
    return $root;
  }
  $ptrHash = $wrapDbh->{'FETCHHASH'}->();
  %vSite = %{$ptrHash};
  $wrapDbh->{'FINISH'}->();

#RaQ3
#('emaildomain',
# 'name', 'suspend', 'type','ipaddr','hostname', 'domain','fqdn',
#! 'gid', 'webdomain',  ## only in RaQ3
#! 'bwlimit',           ## only in RaQ3
# 'maxusers','quota','ftpquota','ftpusers',
# 'ftp','cgi','ssi','ssl','fpx','apop','shell',
# 'userlist_range','userlist_sort',
# 'user_quota','user_fpx','user_apop','user_shell','user_namegen',
# 'modify')

#RaQ4
#('name','suspend','type','ipaddr','hostname','domain','fqdn',
#! 'volume',            ## new in RaQ4
# 'maxusers','quota','ftpquota','ftpusers',
# 'ftp', 'cgi','ssi', 'ssl','fpx','apop','shell',
#! 'php','casp',        ## new in RaQ4
# 'userlist_range','userlist_sort',
# 'user_quota','user_fpx','user_apop','user_shell','user_namegen',
# 'user_casp',
# 'modify')
#

  $vSite{'quota'} *= $megabyte if $vSite{'quota'}=~/^\d+$/; ## translate quota to bytes

  if(exists($vSite{'webdomain'})){
    $root->{'ATTRIBUTE'}->('hosting',
			   $vSite{'webdomain'}=~/on/?'true':'false');
  }

  my $ip = $vSite{'ipaddr'};
  $root->{'ATTRIBUTE'}->('ip',$ip);
  if($vSite{'suspend'}=~/on/){
    $root->{'ATTRIBUTE'}->('state','false');
  }

  $dumpFile = &makeDumpFile("$dumpDir/$vSiteDomain.web",  ## name of dump file
			    "$RaQ4Config{'webPrefix'}/$vSiteName/web"  ## dir to dump
			   );
  if ($dumpFile){
    $root->{'ATTRIBUTE'}->('cid',$dumpFile);
  }

  while(($key,$attrName)=each(%vSiteLimits)){
    if($vSite{$key}){
      $root->{'ADDCHILD'}->(makeLimitNode($attrName,$vSite{$key}));
    }
  }

  if($vSite{'bwlimit'} eq 'on'){
    $sql="SELECT bwlimit FROM bw WHERE name in ('$ip','default') ORDER BY name";
    if($wrapDbh->{'EXECUTE'}->($sql)){
      $ptrRow = $wrapDbh->{'FETCHROW'}->();
	  $root->{'ADDCHILD'}->(makeLimitNode('traffic',$ptrRow->[0] * $kilobit));
    }
    $wrapDbh->{'FINISH'}->();
  }

  while(($key,$attrName)=each(%vSitePerms)){
    if($vSite{$key}){
      $item = &makePermissionNode($attrName,$vSite{$key});
      if(ref($item)=~/HASH/){
               $root->{'ADDCHILD'}->($item);
      }
    }
  }

  if($vSite{'ftp'}=~/on/ || $vSite{'ftp'} eq 't'){
    $item = &makeAnonFtpNode($vSiteName,$vSiteDomain,\%vSite);
    if(ref($item)=~/HASH/){
      $root->{'ADDCHILD'}->($item);
    }
  }

  addDomainUsers($root,$vSiteName,$vSiteDomain,\%vSite, $dropAdmins, $clientName);

  # fabricate the user
  if ($clientName eq $defaultraqdomains) {

	my $name = createUserNameFromDomain($vSiteDomain);

	my $adminNode = makeXmlNode('admin');
	$adminNode->{'ATTRIBUTE'}->('name', $name);
	$adminNode->{'ATTRIBUTE'}->('state', 'true');

	$adminNode->{'ADDCHILD'}->(makeSysUserNode($name, randomPasswd(), 'plain'));
	my $mailNode = makeXmlNode('mail');
	$mailNode->{'ATTRIBUTE'}->('mailname', $name . '@' . $vSiteDomain);

	my $mailboxNode = makeXmlNode('mailbox');
	$mailboxNode->{'ATTRIBUTE'}->('type', 'mbox');
	$mailNode->{'ADDCHILD'}->($mailboxNode);

	$adminNode->{'ADDCHILD'}->($mailNode);

	$root->{'ADDCHILD'}->($adminNode);
  }

  &addMailLists($root,$vSiteName,$vSiteDomain,\%vSite);

  if($vSite{'ssl'}=~/on/ || $vSite{'ssl'} eq 't'){
    my $certsDir = "$RaQ4Config{'webPrefix'}/$vSiteName/certs";
    if(-d $certsDir){
	  my $content = readFile($certsDir . "/certificate");
	  if ($content) {
		my $itemCert =  &makeXmlNode('certificate');
		$root->{'ADDCHILD'}->($itemCert);

		$itemCert->{'ADDCHILD'}->(makeXmlNode('certificate-data', $content));

		$content = readFile($certsDir . "/key");
		if ($content) {
		  $item = &makeXmlNode('private-key',$content);
		  $itemCert->{'ADDCHILD'}->($item);
		}
      }
    }
  }
  my @aliases;
  push @aliases, $vSiteDomain;

  if ($www ==  'true') {
    $vSiteDomain = 'www.'.$vSiteDomain; # need for search domain aliases
  }

  my $httpdConf = "/etc/httpd/conf/httpd.conf";
  my ($found);
  open (CONF, "< $httpdConf");
  while(<CONF>) {
    if ($_=~/^ServerName\s+$vSiteDomain/) {
      $found = 1;
      next;
    }

    if ($found and $_=~/\<\/VirtualHost\>/) {
      last;
    }

    if ($found){
      if ($_=~/^ServerAlias\s+(.*)/){
        @aliases = split(/ /, $1);
        $found = undef;
        last;
      }
    }
  }
  close(CONF);

  foreach my $alias (@aliases) {
    makeDomainAliasNode($root, $alias);
  }

  return $root;
}

sub validConnection {
  return ref($_[0]) =~ /HASH/ && ref($_[0]->{'EXECUTE'}) =~ /CODE/;
}

sub getUserDbConnect2 {
  my ($password) = @_;

  return getDbConnect('mysql', "root", $password, "mysql", "localhost", 1);
}

sub getUserDbConnect {
  my $usermysql;

  printToLog("Trying to connect to user's DB without password...");
  $usermysql = getUserDbConnect2("");
  return ("", $usermysql) if validConnection($usermysql);

  printToLog("Trying to connect to user's DB with standard password...");
  $usermysql = getUserDbConnect2("cobalt-mysql");
  return ("cobalt-mysql", $usermysql) if validConnection($usermysql);

  printToLog("Restarting mysql with ACL disabled...");
  `/etc/rc.d/init.d/mysql stop`;
  `safe_mysqld --skip-grant-tables --skip-networking&`;

  printToLog("Trying to connect to user's DB without password...");
  $usermysql = getUserDbConnect2("");
  return ("", $usermysql) if validConnection($usermysql);
}

sub makeDbDomainNode {
  my ($root) = @_;

  return if AgentConfig::mysqlBin() eq "";

  my ($usermysqlpasswd, $usermysql) = getUserDbConnect();
  $root->{'ATTRIBUTE'}->('ip', $defaultipaddr);

  return unless validConnection($usermysql);

  my $sql = "SHOW DATABASES";
  my @dbs;
  if ($usermysql->{'EXECUTE'}->($sql)) {
	while (my $r = $usermysql->{'FETCHROW'}->()) {
	  my $db = $r->[0];
	  printToLog($db);
	  next if $db eq "mysql" or $db eq "test";
	  push @dbs, $db;
	}
  }
  $usermysql->{'FINISH'}->();

  foreach my $db (@dbs) {
	my $dbnode = XmlNode->new("database", "attributes" => { "name" => $db, "type" => "mysql" });
	$dbnode->{'ATTRIBUTE'}->('cid', makeDumpDb($db, "mysql", "root", $usermysqlpasswd));
    $dbnode->{'ATTRIBUTE'}->('version', Db::MysqlUtils::getVersion());

	my %dbusers;

	$sql = "SELECT user.Host, user.User, user.Password FROM user, db ".
	  " WHERE db.Select_priv = 'Y' AND db.Host = user.Host AND db.Db = '$db' ".
	  " AND db.User = user.User AND user.User <> ''";
	if ($usermysql->{'EXECUTE'}->($sql)) {
	  while (my $h = $usermysql->{'FETCHHASH'}->()) {
		unless (exists $dbusers{$h->{'User'}}) {
		  $dbusers{$h->{'User'}} = {};
		  $dbusers{$h->{'User'}}->{'accesshost'} = [];
		}
		push @{$dbusers{$h->{'User'}}->{'accesshost'}}, $h->{'Host'};

		unless (exists $dbusers{$h->{'User'}}->{'password'} or $h->{'Password'} eq '') {
		  $dbusers{$h->{'User'}}->{'password'} = $h->{'Password'};
		}
	  }
	}
	$usermysql->{'FINISH'}->();

	foreach my $dbuser (keys(%dbusers)) {
	  my $dbusernode = XmlNode->new("dbuser",
							   "attributes" => { "name" => $dbuser },
							   "children" => [makePasswordNode($dbusers{$dbuser}->{'password'})]);
	  foreach my $host (@{$dbusers{$dbuser}->{'accesshost'}}) {
		$dbusernode->{'ADDCHILD'}->(XmlNode("accesshost", "content" => $host));
	  }
	  $dbnode->{'ADDCHILD'}->($dbusernode);
	}
	$root->{'ADDCHILD'}->($dbnode);
  }

  return $root;
}


sub addMailLists {

  my ($root,$vSiteName,$vSiteDomain,$ptrvSite)=@_;

  my($line,$file);

  unless($dirMajordomo && -d $dirMajordomo){
    if(-T '/etc/aliases.majordomo'){
      my $fileParser = &makeFileParser('/etc/aliases.majordomo');
      my $ptrHash = $fileParser->{'PARSE'}->('KEYSEPARATOR'=>':');
      foreach $line (values %{$ptrHash}){
		if ($line=~/\|\s*(\/\S+)\/wrapper\s/){
		  $dirMajordomo = $1;
		  last;
		}
      }
    }
    unless ($dirMajordomo && -d $dirMajordomo){
      $dirMajordomo = '/usr/local/majordomo';
      unless(-d $dirMajordomo){
		$dirMajordomo = undef;
      }
    }
  }

  my $domainsLists = "$dirMajordomo/$vSiteDomain/lists";

  unless(-d $dirMajordomo && -d "$dirMajordomo/$vSiteDomain"
	&& -d $domainsLists){
    return;
  }

  my (@lists,$list,$item);

  opendir(LISTS,$domainsLists);
  while ($file = readdir(LISTS)) {
    if ((-f $file) && ($file =~ /^(.*)\.config$/)) {
	  push @lists, $1;
	}
  }
  closedir(LISTS);

  foreach $list (@lists){
    $item = &makeMaliListNode($list,"$domainsLists/$list");

    if(ref($item)=~/HASH/){
      $root->{'ADDCHILD'}->($item);
    }
  }
}

sub makeMailListNode {
  my ($listName,$listPath)=@_;
  my ($root,$key,$value,$adminPassword,$item,%config);
  my $configPath = "$listPath.config";

#
# parse maillist config
#
  unless(-T $configPath){
    return undef;
  }
  unless(open(CONFIG,"<$configPath")){
    return undef;
  }
  binmode CONFIG;
  while(<CONFIG>){
    chomp;
    s/^\s+//;
    s/\s+$//;
    s/#.*$//;
    next unless $_;
    ($key,$value)=split(/\s*=\s*/,$_);
    if(defined($value)){
      $config{$key}=$value;
    }
  }
  close (CONFIG);
  $root = &makeXmlNode('maillist');
  $item = &makePasswordNode($config{'admin_passwd'}||'', 'plain');
  $root->{'ADDCHILD'}->($item);

#
# permissions
#
  my $subscrib=$config{'subscribe_policy'}||$config{'unsubscribe_policy'};
  if($subscrib=~/open/){
    $item= &makePermissinNode('mlsubscribe');
    if(ref($item)=~/HASH/){
      $root->{'ADDCHILD'}->($item);
    }
  }
  unless($config{'restrict_post'}){
    $item= &makePermissinNode('mlspam');
    if(ref($item)=~/HASH/){
      $root->{'ADDCHILD'}->($item);
    }
  }
#
# add recipients
#
  if(open(LIST,$listPath)){
  	binmode LIST;
    while(<LIST>){
      chomp;
      $item = &makeXmlNode('recipient',$_);
      $root->{'ADDCHILD'}->($item);
    }
    close(LIST);
  }

  return $root;
}

sub addDomainUsers {
  my ($root, $vSiteName, $vSiteDomain, $ptrvSite, $dropAdmins, $adminName)=@_;

  my (%users,$sql,$ptrRow,$item,$userName,%user,$ptrHash);
  $sql="SELECT name,admin FROM users WHERE vsite='$vSiteName' ORDER BY name";
  if($wrapDbh->{'EXECUTE'}->($sql)){
    while($ptrRow=$wrapDbh->{'FETCHROW'}->()){
     $users{$ptrRow->[0]}=$ptrRow->[1];
    }
  }
  $wrapDbh->{'FINISH'}->();

  lazyInitPtrShadow();

  foreach $userName (keys %users) {
	lazyInitUser($userName, $dropAdmins);
	next unless exists $usersInfo{$userName};
    unless(exists $usersInfo{$userName}->{'name'}){ #bug 93728
      $usersInfo{$userName}->{'name'} = $userName;
      $usersInfo{$userName}->{'fullname'} = $userName;
    }

	%user = %{$usersInfo{$userName}};

	my $catchAll;
	($item, $catchAll) = makeUserNode($vSiteName, $vSiteDomain, \%user, $adminName);
	if (ref($item) =~ /HASH/) {
	  $root->{'ADDCHILD'}->($item);
	}

	if ($catchAll) {
	  $root->{'ATTRIBUTE'}->('catch-all', $catchAll);
	}
  }
}

sub makeUserNode {
  my($vSiteName, $vSiteDomain, $ptrUserHash, $adminName) = @_;

  my($root,$userName,$item,$key,$attrName);

  if ($ptrUserHash->{'name'} eq $adminName) {
	$root = makeXmlNode('admin');
  } else {
	$root = makeXmlNode('user');
  }

  $userName = $ptrUserHash->{'name'};
  $root->{'ATTRIBUTE'}->('name', $userName);

# RaQ3
# 'name','type','vsite','fullname','altname',
# 'uid','suspend','admin','quota',
# 'fpx','apop','shell',
# 'aliases','forward','vacation',
# 'vacationmsg', ## only in RaQ3 (<vacationDir>/<name>.msg)
# 'modify'

#RaQ4
# 'name','type','vsite','fullname','altname',
# 'uid','suspend','admin','quota',
# 'fpx','apop','shell',
# 'aliases','forward','vacation',
# 'modify'



  if ($ptrUserHash->{'suspend'} =~ /on/) {
    $root->{'ATTRIBUTE'}->('state', 'false');
  }

  $item = makeXmlNode('fullname', $ptrUserHash->{'fullname'});
  $root->{'ADDCHILD'}->($item);

  my $userNode;
  if ($ptrUserHash->{'shadow'}) {
	  $userNode = makeSysUserNode($userName, $ptrUserHash->{'shadow'}, 'encrypted');
  } else {
	  $userNode = makeSysUserNode($userName, '', 'plain');
  }

  if (ref($item) =~ /HASH/) {
    $root->{'ADDCHILD'}->($userNode);
  }

  while (($key, $attrName) = each(%userLimits)) {
    if ($ptrUserHash->{$key}) {
      $root->{'ADDCHILD'}->(makeLimitNode($attrName,$ptrUserHash->{$key}));
    }
  }

  while (($key, $attrName) = each(%userPerms)) {
    if ($ptrUserHash->{$key}) {
      $root->{'ADDCHILD'}->(makePermissionNode($attrName, $ptrUserHash->{$key}));
    }
  }

  my $catchAll;
  ($item, $catchAll) = makeMailNode($vSiteName, $vSiteDomain, $ptrUserHash);
  $root->{'ADDCHILD'}->($item);

  return ($root, $catchAll);
}

sub makeMailNode {
  my($vSiteName,$vSiteDomain,$ptrUserHash) = @_;
  my($item,$autoresponder,$path,$fileName);

  my $userName = $ptrUserHash->{'name'};
  my $root = &makeXmlNode('mail');
  my $mailName = "$userName\@$vSiteDomain";
  my $catchAll;

  $root->{'ATTRIBUTE'}->('mailname',$mailName);

  if($ptrUserHash->{'forward'} && ($ptrUserHash->{'forward'} ne 'off')){
    # sometimes forward address contains dot at the end
    my $faddr = $ptrUserHash->{'forward'};
    $faddr =~ s/\.$//;
    $item = &makeXmlNode('forward',$faddr);
    $root->{'ADDCHILD'}->($item);
  }

  if($ptrUserHash->{'aliases'}){
    foreach my $alias (split(' ',$ptrUserHash->{'aliases'})){
	  if ($alias =~ /^@(.*)$/) { #catch-all address
		$catchAll = $mailName;
	  } else { #regular alias
		if ($alias ne $userName) {
		  $item = &makeXmlNode('alias',$alias);
		  $root->{'ADDCHILD'}->($item);
		}
	  }
    }
  }
  unless($autoresponder = $ptrUserHash->{'vacationmsg'}){
    $path= "$RaQ4Config{'webPrefix'}/$userName.msg";
    if(-T $path){
      if(open(MSG,"<$path")){
	  	binmode MSG;
		$autoresponder=<MSG>;
		close (MSG);
      }
    }
  }
  if($autoresponder){
    $item = &makeXmlNode('autoresponder',$autoresponder);
    if($ptrUserHash->{'vacation'}=~/off/){
      $item->{'ATTRIBUTE'}->('state','false');
    }
    $root->{'ADDCHILD'}->($item);
  }

  my ($inbox_dump, $dump);

  my @extraParams = ('-h');
  $path = "$RaQ4Config{'spoolDir'}/$userName";
  if (-e $path) {
    $inbox_dump = makeDumpFile("$dumpDir/$userName.mbox",
			     $RaQ4Config{'spoolDir'},
			     $userName,
			     undef,
			     \@extraParams);
  }
  my $mailboxDir = $RaQ4Config{'webPrefix'} . "/$vSiteDomain/users/$userName/mail";
  if (-d $mailboxDir) {
    $dump = makeDumpFile("$dumpDir/$userName"."@"."$vSiteDomain.mail", $mailboxDir,
                         undef, undef, \@extraParams);
  }

  if ($inbox_dump || $dump) {
    $item = makeXmlNode('mailbox');
    $item->{'ATTRIBUTE'}->('type', 'mbox');
    $item->{'ATTRIBUTE'}->('cid_inbox', $inbox_dump) if $inbox_dump;
    $item->{'ATTRIBUTE'}->('cid', $dump) if $dump;
    $root->{'ADDCHILD'}->($item);
  }

  return ($root, $catchAll);
}

sub makeAnonFtpNode {
  my ($name,$domain,$ptrHash) = @_;

  unless(ref($ptrHash)=~/HASH/){
    return undef;
  }

  my($dumpFile,$key,$attrName,$item);

  my $root = &makeXmlNode('anonftp');

#
# dump ftp content
#
  $dumpFile = &makeDumpFile("$dumpDir/$domain.ftp",  ## name of dump file
			    "$RaQ4Config{'webPrefix'}/$name/ftp",  ## dir to dump
			    '.',       ## dump all
			    'incoming' ## exclude 'incoming'
			       );
  $root->{'ATTRIBUTE'}->('cid',$dumpFile) if $dumpFile;

  $dumpFile = &makeDumpFile("$dumpDir/$domain.ftp.incoming",  ## name of dump file
			    "$RaQ4Config{'webPrefix'}/$name/ftp",  ## dir to dump
			    'incoming' ## dump 'incoming'
			   );
  $root->{'ATTRIBUTE'}->('cid_incoming',$dumpFile) if $dumpFile;

#
# end dump ftp content
#


  if($ptrHash->{'ftpquota'}) {
	$root->{'ADDCHILD'}->(makeLimitNode('space', $ptrHash->{'ftpquota'} * $megabyte));
  }
  if($ptrHash->{'ftpusers'}) {
	$root->{'ADDCHILD'}->(makeLimitNode('account', $ptrHash->{'ftpusers'}));
  }

  return $root;
}

sub makeLimitNode {
  my($name,$value) = @_;
  unless (defined($value)) {
	die 'makeLimitNode: undefined $value';
  }
  my $root = makeXmlNode('limit', $value);
  $root->{'ATTRIBUTE'}->('name',$name);
  return $root;
}

sub makePermissionNode {
  my($name, $value) = @_;
  my $root = makeXmlNode('permission');
  $root->{'ATTRIBUTE'}->('name', $name);

  if ($value eq 'off' or $value eq 'f') {
	$root->{'ATTRIBUTE'}->('value', 'off');
  } elsif ($value eq 'on' or $value eq 't') {
	$root->{'ATTRIBUTE'}->('value', 'on');
  } else {
	die "makePermissionNode: not a boolean $value: " . $value;
  }
  return $root;
}

sub printAgentStatus {
  my $root = makeXmlNode('agent-status');

  unless (exists($RaQ4Config{'build'}) && defined AgentConfig::iconvBin()) {
    my $item;
	if (defined AgentConfig::iconvBin()) {
	  $item = makeXmlNode('wrong-platform', '');
	} else {
	  $item = makeXmlNode('wrong-platform', 'no iconv found on the source host');
	}

	$root->{'ADDCHILD'}->($item);
  }

  printXml($root, *STDOUT);
}

sub loadMainConfig {
  my $pathToConf = shift||'/etc/cmu/cmu.conf';
  my $ptrHash = shift;
  my %bldHash = ('2700'=>'RaQ1',
		'2799'=>'RaQ2',
		'2800'=>'RaQ2',
		'3000'=>'RaQ3',
		'3001'=>'RaQ4',
		'3100'=>'RaQ4',
		'3500'=>'RaQXTR',
		'3599'=>'RaQXTR',

		);
  if((-T '/etc/build') && open(CONF,"</etc/build")){
    my ($buildCode);
    while(<CONF>){
      if(/(\d{4,4})/){
		$buildCode=$1;
		if (exists($bldHash{$buildCode})){
		  $ptrHash->{'buildCode'} = $buildCode;
		  $ptrHash->{'build'} = $bldHash{$buildCode};
		  last;
		}
      }
    }
    close (CONF);
  }
  if((-T '/etc/cobalt/.meta.id') && open(CONF,"</etc/cobalt/.meta.id")){
  	binmode CONF;

    my $content = <CONF>;
    chomp $content;
    $content=~s/^\s+//;
    $content=~s/\s+$//;

    $ptrHash->{'dbpassword'} = $content;

    close(CONF);
  }
  if((-T $pathToConf) && open(CONF,"<$pathToConf")){
  	binmode CONF;
    while(<CONF>){
      chomp;
      next unless $_;
      next if /^#/;
      if(/\s*(\S+)\s*=\s*(.+)/){
		$ptrHash->{$1} = $2;
      }
    }
    close(CONF);
  }else{
	$ptrHash->{'webPrefix'} = '/home/sites';
	$ptrHash->{'spoolDir'} = '/var/spool/mail';
  }

  return $pathToConf;
}

sub getMailDomainNames {
 my $localConfig = shift || "/etc/mail/local-host-names";

 my %domains;

 open (LOCAL, "< $localConfig");

 while(<LOCAL>) {
  chomp;
  $domains{$_} = 1;
 }

 close(LOCAL);

 return \%domains;
}

sub makeDomainAliasNode {
  my ($parent, $alias) = @_;

  my $mailAliases = getMailDomainNames();

  my $mail;

  $mail = (exists $mailAliases->{$alias}) ? 'true' : 'false';

  if (exists $mailAliases->{'mail.'.$alias}) {
    $mail = 'true';
    $alias = 'mail.'.$alias;
  }

  my $aliasNode = XmlNode->new('domain-alias', 'attributes' => {'name' => $alias, 'mail' => $mail, 'web' => 'true'}, 'children' => [Status::make('enabled')]);
  $parent->{'ADDCHILD'}->($aliasNode);
}

sub printHelp {

  print <<"HELP";

Agent of migration from RaQ4,XTR to Plesk

Usage:
  $0 <options>

Options:
  -c |--config=<path>        path to 'cmu.conf'

  -s |--get-status           get status of the agent
  -dc|--dump-accounts=<list> a coma separated list of resellers to dump
  -dd|--dump-domains=<list>  a coma separated list of customers to dump
  -da|--dump-all             make a full dump

  -lc|--get-content-list     get list of content files
  -nc|--no-content           do not make content files
  -nz|--no-compress          do not compress content files

  -h |--help                 this help

HELP
}
