#!/usr/bin/perl
# Copyright 1999-2015. Parallels IP Holdings GmbH. All Rights Reserved.

# use for dump of user's mysql databases
use vars qw($mysqlUserUser $mysqlUserServer $mysqlUserPw $storage);

use vars qw|$dumpDir
			$ptrArgs
			$pleskVersion
			$pleskMajorVersion
			$pleskMinorVersion
			$pleskMinorMinorVersion
			$workDir|;

use strict;
use Status;

use AgentConfig;

use XmlNode;
#
# maps to translate xmlName => fieldName
#

my %xmlPermissions = ('create_domains'=>1,
		      'create_domains'=>1,
		      'manage_phosting'=>1,
		      'manage_quota'=>1,
		      'manage_subdomains'=>1,
		      'change_limits'=>1,
		      'manage_dns'=>1,
		      'manage_log'=>1,
		      'manage_crontab'=>1,
		      'manage_anonftp'=>1,
		      'manage_webapps'=>1,
		      'manage_sh_access'=>1,
		      'manage_maillists'=>1,
		      'make_dumps'=>1,
		      'manage_drweb'=>1,
		      'make_phosting'=>1,
		      'manage_quota'=>1,
			  'manage_not_chroot_shell'=>1,
			  'manage_ftp_password'=>1,
		     );


my %domainLimits25 = ('max_wu' => 1,
					  'max_db' => 1,
					  'max_box' => 1,
					  'mbox_quota' => 1,
					  'max_redir' => 1,
					  'max_mg' => 1,
					  'max_resp' => 1);

my %typeOfField = ('expiration'=>'timestamp');

my %clientsAttribute = ('name'=>'login',
		       'contact'=>'pname',
		       );

my %clientsInfo = ('company'=>'cname',
		  'phone'=>'phone',
		  'fax'=>'fax',
		  'address'=>'address',
		  'city'=>'city',
		  'state'=>'state',
		  'zip'=>'pcode',
		  'country'=>'country',
		  'locale'=>'locale',
		  'email'=>'email',
		  );

my %domainUserAttribute = ('contact'=>'personalName');
my %domainUserInfo = ('company'=>'conpanyName',
		  'phone'=>'phone',
		  'fax'=>'fax',
		  'address'=>'address',
		  'city'=>'city',
		  'state'=>'state',
		  'zip'=>'pcode',
		  'email'=>'email',
		  'country'=>'country',);

my %hostingAttribute = ('https'=>'ssl',
			'fp'=>'fp',
			'fpssl'=>'fp_ssl',
			'fpauth'=>'fp_enable',
			'webstat'=>'webstat',
		       );

my %hostingScripting = ('ssi'=>'ssi',
			'php'=>'php',
			'cgi'=>'cgi',
			'perl'=>'perl',
			'asp'=>'asp',
			'python'=>'python',
			'coldfusion'=>'coldfusion',
			'asp_dot_net'=>'asp_dot_net',
		       );

my %webUserScripting = ('ssi'=>'ssi',
			'php'=>'php',
			'cgi'=>'cgi',
			'perl'=>'perl',
			'asp'=>'asp',
			'python'=>'python',
			'asp_dot_net'=>'asp_dot_net',
		       );

my %subDomainScripting = ('ssi'=>'ssi',
			'php'=>'php',
			'cgi'=>'cgi',
			'perl'=>'perl',
			'asp'=>'asp',
			'python'=>'python',
			'coldfusion'=>'coldfusion',
			'asp_dot_net'=>'asp_dot_net',
		       );

my %locale_map = ('bg' => 'bg', 'hk' => 'zh-HK',
			'ru' => 'ru-RU', 'en' => 'en-US',
			'nl' => 'nl-NL', 'br' => 'pt-BR',
			'it' => 'it-IT', 'tr' => 'tr-TR',
			'es' => 'es-ES', 'pl' => 'pl-PL',
			'ca' => 'ca-ES', 'jp' => 'ja-JP',
			'tw' => 'zh-TW', 'fi' => 'fi-FI',
			'cn' => 'zh-CN', 'ko' => 'ko-KR',
			'fr' => 'fr-FR', 'pt' => 'pt-PT',
			'de' => 'de-DE', 'lt' => 'lt-LT'
				);

#
# end maps
#

#
# GLOBAL
#

my $defaultIp;
my %exclIp;

#
# parse command line
#
my %arg_opts = ('--help|-h'=>'',
		'--config|-c'=>'s',
		'--get-status|-s'=>'',
		'--dump-accounts|-dc'=>'s',
		'--dump-domains|-dd'=>'s',
		'--dump-all|-da'=>'',
		'--get-content-list|-lc'=>'',
		'--no-content|-nc'=>'',
		'--no-compress|-nz'=>'',
		'--output|-o'=>'s',
		'--status-file|-sf'=>'s',
	       );

#@@INCLUDE FILE="agent.include.pl"@@
if(-f 'agent.include.pl'){
  require 'agent.include.pl';
}else{
  require '../agent.include.pl';
}
#@@/INCLUDE@@

$ptrArgs = getArguments(\@ARGV, \%arg_opts);

my ($outPath, $dumpFile, $statusFile);

$workDir = AgentConfig::cwd();

$dumpFile = $workDir.'/dump.xml';
$statusFile = $workDir.'/dumping-status.xml';
$dumpDir = $workDir.'/pma';
$pleskVersion = undef;
$pleskMajorVersion = undef;
$pleskMinorVersion = undef;
$pleskMinorMinorVersion = undef;

my $objDumpStatus = &makeDumpStatus($ptrArgs->{'status-file'}||$statusFile);

#
# get MIME Base64 encoder
#
my $wrapBase64 = makeMIMEBase64();

my (%PsaConfig,%domainsInfo);
my $rootName = 'migration-dump';
my $dtdName = 'plesk.dtd';

AgentConfig::init();

unless($ptrArgs->{'config'} = &loadMainConfig(\%PsaConfig)){
  if(exists $ptrArgs->{'get-status'}){
    printAgentStatus();
    exit 0;
  }else{

    if(exists ($ptrArgs->{'dump-all'}) ||
       exists ($ptrArgs->{'dump-accounts'}) ||
       exists ($ptrArgs->{'dump-domains'})){

      my $root=&makeEmptyDump($rootName);

      if(ref($root)=~/HASH/ && ref($root->{'PRINT'})=~/CODE/){
		my $outFh = &openOutput($ptrArgs->{'output'});
		&printXml($root,$outFh,$dtdName);
		&closeOutput();
      }
    }

    die "Error: Psa config file 'admin.conf' is not found\n";
  }
}
#
# get db connect
#

my $wrapDbh = getDbConnect('mysql', 'admin', $PsaConfig{'password'}, 'slash', 'localhost');
unless (ref($wrapDbh) =~ /HASH/ && ref($wrapDbh->{'EXECUTE'}) =~ /CODE/) {
  die "Error: can not connect to Slash database\n";
}

if(exists $ptrArgs->{'get-status'}){
  printAgentStatus();
}elsif(exists ($ptrArgs->{'dump-all'}) ||
       exists ($ptrArgs->{'dump-accounts'}) ||
       exists ($ptrArgs->{'dump-domains'})){

  my ($root,@accounts,@domains,$ptrAccounts,$ptrDomains, $value);
  initStorage();
  getDumpDir($dumpDir);

  if (exists $ptrArgs->{'no-compress'}) {
	setCompress(0);
  }

  &printToLog("Work dir: $workDir");
  &printToLog("Dump file: $dumpFile");
  &printToLog("Status file: ".$objDumpStatus->{'FILE'}->());

  if ($value = $ptrArgs->{'dump-accounts'}){
  	if ($value eq "-") {
		$value = <STDIN>;
		chomp $value;
	}
    @accounts = split(/\s*,\s*/,$value);
  }
  $ptrAccounts = \@accounts;

  if ($value = $ptrArgs->{'dump-domains'}){
  	if ($value eq "-") {
		$value = <STDIN>;
		chomp $value;
	}
    @domains = split(/\s*,\s*/,$value);
  }
  $ptrDomains = \@domains;
#
# generate a xml dump
#

  $root = &getSlashDump($dumpDir, $ptrAccounts, $ptrDomains, $rootName);

#
# print dump to output
#
  $storage->finish($root);
}elsif(exists $ptrArgs->{'get-content-list'}){
  makeContentList();
}else{
  &printHelp();
}


exit 0;

#
# end main
#
#==============================================================
#
# subroutines
#

sub getSlashDump {
  my ($dumpDir, $ptrAccounts, $ptrDomains, $rootName) = @_;

  my ($sql,$list,$ptrRow,$name,$domain,$item,
      %domains,%clients,$ptrClients,$fakeClient);

  my $root = XmlNode->new($rootName, 'attributes'=>{'agent-name'=>'Slash'});

  unless(ref($wrapDbh)=~/HASH/ && ref($wrapDbh->{'EXECUTE'})=~/CODE/){
    $root->{'CONTENT'}->("Error: there is no connect to database");
    &printToError("Error: unable to connect to SQL database");
    return $root;
  }


#
# prepare list of domains
#
  if (ref($ptrDomains)=~/ARRAY/ && (@{$ptrDomains}>0)){

    $fakeClient= &makeXmlNode('client');

    $list = join(',',map {"'$_'"} @{$ptrDomains});
    $sql = "SELECT d.id,d.name,d.cl_id,c.login FROM domains d, clients c ".
      "WHERE d.name in ($list) AND d.cl_id = c.id";
    if($wrapDbh->{'EXECUTE'}->($sql)){
      while( $ptrRow = $wrapDbh->{'FETCHROW'}->()){
		$domains{$ptrRow->[0]}=[$ptrRow->[1],$ptrRow->[2]];
		$clients{$ptrRow->[2]}=[$ptrRow->[3],0];
      }
    }
    $wrapDbh->{'FINISH'}->();
  }

  if (ref($ptrAccounts)=~/ARRAY/ && (@{$ptrAccounts}>0)) {
	$list = join (',', map {"'$_'"} @{$ptrAccounts});

	$sql = "SELECT c.id, c.login, d.id, d.name FROM clients c LEFT JOIN domains d ".
	  "ON c.id = d.cl_id WHERE c.login in ($list)";

	if ($wrapDbh->{'EXECUTE'}->($sql)) {
	  while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
		$clients{$ptrRow->[0]} = [$ptrRow->[1], 0];
		$domains{$ptrRow->[2]} = [$ptrRow->[3], $ptrRow->[0]];
	  }
	}
	$wrapDbh->{'FINISH'}->();
  }

  unless (keys %clients) {
	$sql = "SELECT id, pname FROM clients";
	$wrapDbh->{'EXECUTE'}->($sql);
	while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
	  $clients{$ptrRow->[0]} = [$ptrRow->[1],1];
	}
	$wrapDbh->{'FINISH'}->();
  }
#
# end prepare list of domains
#

#
# prepare list of clients
#

  if(ref($ptrAccounts)=~/ARRAY/ && (@{$ptrAccounts}>0)){

    $fakeClient = undef;

    $list = join(',',map {"'$_'"} @{$ptrAccounts});
    $sql = "SELECT id, login FROM clients WHERE login in ($list)";
    if($wrapDbh->{'EXECUTE'}->($sql)){
      while( $ptrRow = $wrapDbh->{'FETCHROW'}->()){
		$clients{$ptrRow->[0]}=[$ptrRow->[1],1];
      }
    }
    $wrapDbh->{'FINISH'}->();
  }
#
# end prepare list of clients
#

#
# domains/clients count
#
  my ($domainsCount, $clientsCount);

  if (ref($ptrDomains) =~ /ARRAY/ && (@{$ptrDomains} > 0)) {
	$domainsCount = scalar(@{$ptrDomains});
	$clientsCount = 1;
  } elsif (ref($ptrAccounts) =~ /ARRAY/ && (@{$ptrAccounts} > 0)) {
	my $list = join(",", keys %clients);
	$sql = "SELECT COUNT(*) FROM domains WHERE parentDomainId = 0 AND cl_id in ($list)";
	$wrapDbh->{'EXECUTE'}->($sql);
	$ptrRow = $wrapDbh->{'FETCHROW'}->();
	$wrapDbh->{'FINISH'}->();

	$domainsCount = $ptrRow->[0];
	$clientsCount = scalar(@{$ptrAccounts});
  } else {
	$sql = "SELECT COUNT(*) FROM domains WHERE parentDomainId = 0";
	$wrapDbh->{'EXECUTE'}->($sql);
	$ptrRow = $wrapDbh->{'FETCHROW'}->();
	$wrapDbh->{'FINISH'}->();

	$domainsCount = $ptrRow->[0];

	$sql = "SELECT COUNT(*) FROM clients";
	$wrapDbh->{'EXECUTE'}->($sql);
	$ptrRow = $wrapDbh->{'FETCHROW'}->();
	$wrapDbh->{'FINISH'}->();

	$clientsCount = $ptrRow->[0];
  }
#
# end domains/clients count
#
  $objDumpStatus->{'COUNTS'}->($clientsCount, $domainsCount);

  # Prepare IP information
  printToLog('Read IP information ...', 1);
  readIpInfo($ptrClients);
  printToLog(' OK');

#
# loop by client
#
  my($clientName,$full,$clientID);
  foreach $clientID (sort {$a <=> $b} keys %clients){

    ($clientName,$full)=@{$clients{$clientID}};

    &printToLog("Client '$clientName' is started");

    $objDumpStatus->{'ACCOUNT'}->($clientName);
    $objDumpStatus->{'PRINT'}->();

    $item = &makeClientNode($clientID,$full,\%domains,$fakeClient);

    if(ref($item)=~/HASH/ && ref($item->{'PRINT'})=~/CODE/){
      $root->{'ADDCHILD'}->($item);
      &printToLog("Client '$clientName' is successfully dumped");
    }else{
      &printToLog("Client '$clientName' is not dumped");
    }
  }

#
# end loop by client
#

  $objDumpStatus->{'ACCOUNT'}->('');
  $objDumpStatus->{'DOMAIN'}->('');
  $objDumpStatus->{'COUNTS'}->(0,0);
  $objDumpStatus->{'PRINT'}->();


  return $root;
}

#
# Converts the old country codes to the recent ones
#

sub normalizeCountry {
  my ($country) = @_;
  return 'RU' if $country eq 'SU';
  return 'GB' if $country eq 'UK';
  return $country;
}

#
# sub makeClientNode
#
#  arguments:
#         $clientID - ID of client
#         $full - dump all domains
#         $ptrDomainHash
#         $fakeClient - if all domains are in one fake client
#

sub makeClientNode {
  my($clientID,$full,$ptrDomainsHash,$fakeClient)=@_;
  $full = 1 unless defined($full) ;
  $ptrDomainsHash ||= {};

  my ($item,$root,$sql,$ptrHash,$value,%client,@domains,$ptrRow,$id,%clientParams);

#
# get client info
#
  $sql = "select * from clients where id = $clientID";

  unless($wrapDbh->{'EXECUTE'}->($sql)){
    $wrapDbh->{'FINISH'}->();
    return undef;
  }
  unless($ptrHash = $wrapDbh->{'FETCHHASH'}->()){
    $wrapDbh->{'FINISH'}->();
    return undef;
  }
  %client=%{$ptrHash};

  $wrapDbh->{'FINISH'}->();

  $client{'country'} = normalizeCountry($client{'country'});

  my($attribute, $field, $name);

  if(exists $client{'locale'} && $client{'locale'}) {
    while (($name, $field) = each(%locale_map)) {
      if($client{'locale'} eq $name) {
        $client{'locale'} = $field;
      }
    }
  }

#
# end get client info
#

  $root = &makeXmlNode('client');
  while (($attribute,$field)=each(%clientsAttribute)){
    if(exists($client{$field})){
	  unless (defined($client{$field})) {
		$client{$field} = '';
	  }
      $root->{'ATTRIBUTE'}->($attribute,$client{$field});
    }
  }

  $item = makePasswordNode($client{'passwd'},'plain');
  $root->{'ADDCHILD'}->($item);

  my $status = $Status::ENABLED;
  $status = $Status::ADMIN if $client{'status'};
  $root->{'ADDCHILD'}->(Status::make($status));

  unless (exists $ptrArgs->{'no-content'}) {

	while (($name, $field) = each(%clientsInfo)) {
	  if (exists($client{$field}) && $client{$field}) {
		$item = makeXmlNode('pinfo', $client{$field});
		$item->{'ATTRIBUTE'}->('name', $name);
		$root->{'ADDCHILD'}->($item);
	  }
	}

	$sql = "SELECT param,val FROM cl_param WHERE cl_id = $clientID";
	if($wrapDbh->{'EXECUTE'}->($sql)){
	  while ($ptrRow=$wrapDbh->{'FETCHROW'}->()){
		$clientParams{$ptrRow->[0]}=$ptrRow->[1];
	  }
	}
	$wrapDbh->{'FINISH'}->();

	addPermissionNode($root, 'multiple-sessions', 'true');
  }

	# ip pool
	my @ips;

	foreach my $domain (sort keys %domainsInfo) {
		$ptrRow = $domainsInfo{$domain};
		if ($ptrRow->[3] != $clientID) {
			next;
		}
		push @ips, $ptrRow->[1];
	}

	if (@ips) {
		my $ip_pool = makeXmlNode('ip_pool');
		$root->{'ADDCHILD'}->($ip_pool);

		foreach my $ip (uniq(@ips)) {
			$ip_pool->{'ADDCHILD'}->(makeIpNode($ip));
		}
	}

#
# prepare domain's list
#
  my($domainID,$domainName);
  if($full){
    $sql = "SELECT id,name FROM domains WHERE cl_id=$clientID ORDER BY id";
    if($wrapDbh->{'EXECUTE'}->($sql)){
      while($ptrRow=$wrapDbh->{'FETCHROW'}->()){
		($domainID,$domainName)=@{$ptrRow};
		push @domains,[$domainID,$domainName];
      }
    }
	$wrapDbh->{'FINISH'}->();
  }else{
    foreach $domainID (sort {$a <=> $b} keys %{$ptrDomainsHash}){
      $ptrRow = $ptrDomainsHash->{$domainID};
      next unless $ptrRow->[1]==$clientID;
      push @domains,[$domainID,$ptrRow->[0]];
    }
  }
#
# end prepare domain's list
#

#
# loop by domain of the client
#
  foreach $ptrRow (@domains){
    ($domainID,$domainName)=@{$ptrRow};

    &printToLog("Domain '$domainName' is started");

    $objDumpStatus->{'DOMAIN'}->($domainName);
    $objDumpStatus->{'PRINT'}->();

    $item = &makeDomainNode($domainID);
    if(ref($item)=~/HASH/){
      $root->{'ADDCHILD'}->($item);
      &printToLog("Domain '$domainName' is successfully dumped");
    }else{
      &printToLog("Domain '$domainName' is not dumped");
    }
  }
#
# end loop by domain of the client
#


  return $root;
}

sub addPermissions25 {
  my ($root, $id) = @_;

  # [42098] migration from 7.14 to 7.50 fails
  return unless ($id);

  my ($perm_name, $sql, $ptrRow, $value, $item);
  my @perm_names = ('create_domains', 'manage_dns', 'manage_log');

  foreach $perm_name (@perm_names) {
	$sql = "SELECT DISTINCT val FROM cl_param WHERE cl_id = '$id' AND param = '$perm_name'";
	if ($wrapDbh->{'EXECUTE'}->($sql)) {
		addPermissionNode($root, $perm_name, @{$wrapDbh->{'FETCHROW'}->()}[0]);
	}
	$wrapDbh->{'FINISH'}->();
  }

  $sql = "SELECT DISTINCT val FROM cl_param WHERE cl_id='$id' AND (param='ip_based_allow' OR param='nb_allow')";
  if ($wrapDbh->{'EXECUTE'}->($sql)) {
	addPermissionNode($root, 'manage_phosting', @{$wrapDbh->{'FETCHROW'}->()}[0]);
  }
  $wrapDbh->{'FINISH'}->();
}

sub insertLimitNode {
  my($root,$name,$value) = @_;
  my $item = &makeXmlNode('limit', $value);
  $item->{'ATTRIBUTE'}->('name', $name);
  $root->{'ADDCHILD'}->($item);
}

sub insertLimit {
  my($root,$name,$value) = @_;
  my($item);

  if($value eq '' || $value eq '-1') {
	insertLimitNode($root, $name, '-1');
	return;
  }

  if($name eq 'expiration'){
	my($mday,$mon,$year)=(localtime($value))[3..5];
	$mon++;
	$year+=1900;
	$value = sprintf('%04d-%02d-%02d',$year,$mon,$mday);
  }

  if ($name eq 'mbox_quota' and $value ne '-1') {
	$value = $value * 1024;
  }

  insertLimitNode($root, $name, $value);
}

sub addDomainLimits25 {
  my ($root, $clientId) = @_;
  addLimits25($root, $clientId, \%domainLimits25, 'dom_id', 'dom_param');

  my $ptrRow;
  my $sql = "SELECT size, traffic FROM hosting WHERE dom_id = '$clientId'";
  if ($wrapDbh->{'EXECUTE'}->($sql) && ($ptrRow = $wrapDbh->{'FETCHROW'}->())) {
	insertLimit($root, 'disk_space', $ptrRow->[0]);
	insertLimit($root, 'max_traffic', $ptrRow->[1]);
  }
	$wrapDbh->{'FINISH'}->();
}

sub addLimits25 {
  my($root, $limitId, $limitNames, $idName, $paramName)=@_;

  if(!$limitId) {
	return;
  }

  my($value, $name, $count, $sql, $ptrRow);

  $sql = "SELECT val, param FROM $paramName WHERE $idName='$limitId'";
  if ($wrapDbh->{'EXECUTE'}->($sql)) {
	while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
	  ($name, $value) = @{$ptrRow};
	}
  }

	$wrapDbh->{'FINISH'}->();

  foreach $name (%{$limitNames}) {
	$sql = "SELECT val FROM $paramName WHERE $idName = '$limitId' AND param = '$name'";

	if(($wrapDbh->{'EXECUTE'}->($sql)) && ($ptrRow = $wrapDbh->{'FETCHROW'}->())) {
	  ($value) = @{$ptrRow};
	  insertLimit($root, $name, $value);
	}
	$wrapDbh->{'FINISH'}->();
  }
}


#
# makeDomainMode
#
# arguments:
#           $domainID - domain id
#
# return:
#           $root - pointer to XML node
#
sub makeDomainNode {
  my($domainID)=@_;

  my($sql,$item,$root,%domain,%domParams,$ptrRow,$ptrHash,
     $domainName,$id);

#
# get domain's info
#
  printToLog("Getting domain info");
  $sql = "SELECT * FROM domains WHERE id = $domainID";

  unless($wrapDbh->{'EXECUTE'}->($sql)){
    $wrapDbh->{'FINISH'}->();
    return undef;
  }
  unless($ptrHash = $wrapDbh->{'FETCHHASH'}->()){
    $wrapDbh->{'FINISH'}->();
    return undef;
  }
  %domain=%{$ptrHash};
  $wrapDbh->{'FINISH'}->();

  $sql = "SELECT param,val FROM dom_param WHERE dom_id=$domainID";
  if($wrapDbh->{'EXECUTE'}->($sql)){
    while ($ptrRow = $wrapDbh->{'FETCHROW'}->()){
      $domParams{$ptrRow->[0]}=$ptrRow->[1];
    }
  }
  $wrapDbh->{'FINISH'}->();

#
# end get domain's info
#

  $root=&makeXmlNode('domain');

  $domainName = $domain{'name'};

  addWwwStatus($root, $domainID, $domain{'name'});

  $root->{'ATTRIBUTE'}->('name',$domainName);

  my $domainType = $domain{'htype'};

  printToLog("Getting domain IP");
  my $ip = $domainsInfo{$domainName}->[1];
  if (!$ip) {
	$ip = $defaultIp;
  }
  $root->{'ADDCHILD'}->(makeIpNode($ip));

  printToLog("Dupming domain status");
  my $status = $Status::ENABLED;
  if ($domain{'admin_status'} eq 'true') {
	$status ||= $Status::ADMIN;
  } elsif ($domain{'status'} eq 'true') {
	$status ||= $Status::STATUS;
  }
  $root->{'ADDCHILD'}->(Status::make($status));

  # No further info required if 'no-content' specified
  if (exists $ptrArgs->{'no-content'}) {
	return $root;
  }

  printToLog("Getting domain limits");
#
# limits
#
  addDomainLimits25($root,$domainID);
#
# end limits
#

  printToLog("Dumping domain mailsystem");
#
# mail system
#

  my $rootMail = makeXmlNode('mailsystem');
  $rootMail->{'ADDCHILD'}->(Status::make($Status::ENABLED));

  $sql = "SELECT id, password, 'plain' FROM mail ".
	" WHERE dom_id=$domainID ORDER BY mail_name";


  if ($wrapDbh->{'EXECUTE'}->($sql)){
    my(@mails);
    while($ptrRow = $wrapDbh->{'FETCHROW'}->()){
      push @mails,[@{$ptrRow}];
    }
    $wrapDbh->{'FINISH'}->();

    foreach $ptrRow (@mails){
      $item = &makeMailUserNode(@{$ptrRow},$domainName);
      if(ref($item)=~/HASH/){
		$rootMail->{'ADDCHILD'}->($item);
      }
    }
  }

  getCatchAllAddress($rootMail, $domainID);

  $root->{'ADDCHILD'}->($rootMail);
#
# end mail system
#

  printToLog("Dumping domain DNS");
#
# dns
#
  my $domainIp;

  my $dnsNode = makeXmlNode('dns');

  $sql = "SELECT dns_zone FROM domains WHERE id=$domainID";
  if ($wrapDbh->{'EXECUTE'}->($sql) and my $ptrRow = $wrapDbh->{'FETCHROW'}->()) {
	$dnsNode->{'ATTRIBUTE'}->('enabled',
							  $ptrRow->[0] eq 'false' ? 'false' : 'true');
  }
  $wrapDbh->{'FINISH'}->();

  $sql = "SELECT * FROM dns_recs WHERE dom_id=$domainID";
  if ($wrapDbh->{'EXECUTE'}->($sql)) {
    while ($ptrHash = $wrapDbh->{'FETCHHASH'}->()) {
      next if ($ptrHash->{'type'} =~ /none/);
	  if ($ptrHash->{'type'} eq 'TXT') {
		$ptrHash->{'val'} =~ s/"(.*)"/$1/;
	  }
      $item = &makeXmlNode('dnsrec');
      $item->{'ATTRIBUTE'}->('type', $ptrHash->{'type'});
      $item->{'ATTRIBUTE'}->('src', $ptrHash->{'host'});
      $item->{'ATTRIBUTE'}->('dst', $ptrHash->{'val'});
	  $item->{'ATTRIBUTE'}->('opt', $ptrHash->{'opt'}) if ($ptrHash->{'opt'});
      $dnsNode->{'ADDCHILD'}->($item);

	  if ($ptrHash->{'type'} eq 'A'
		  and $ptrHash->{'host'} eq $domainName . '.') {
		$domainIp = $ptrHash->{'val'};
	  }
    }
  }
  $wrapDbh->{'FINISH'}->();

  $item = makeXmlNode('dnsrec');
  $item->{'ATTRIBUTE'}->('type', 'A');
  $item->{'ATTRIBUTE'}->('src', 'webmail.' . $domainName . '.');
  $item->{'ATTRIBUTE'}->('dst', $domainIp);
  $dnsNode->{'ADDCHILD'}->($item);

  $root->{'ADDCHILD'}->($dnsNode);
#
# end dns
#

  printToLog("Dumping domain databases");
#
# database
#
  $sql = "SELECT id,name,type FROM data_bases WHERE dom_id=$domainID ORDER BY id";
  if($wrapDbh->{'EXECUTE'}->($sql)){
    my (@databases);
    while($ptrRow=$wrapDbh->{'FETCHROW'}->()){
      push @databases,[@{$ptrRow}];
    }
    $wrapDbh->{'FINISH'}->();
    foreach $ptrRow (@databases){
      $item=&makeDatabaseNode(@{$ptrRow});
      if(ref($item)=~/HASH/){
		$root->{'ADDCHILD'}->($item);
      }
    }
  }
  $wrapDbh->{'FINISH'}->();
#
# end database
#

  printToLog("Dumping domain statistics");
#
# traffic
#
  &addCurrentTraffic($root,$domainID);
#
# end traffic
#

  printToLog("Dumping domain certificates");
#
# certificates
#
  $sql = "SELECT id FROM certificates WHERE dom_id=$domainID";

  if($wrapDbh->{'EXECUTE'}->($sql)){
    my (@ids);
    while($ptrRow=$wrapDbh->{'FETCHROW'}->()){
      push @ids,$ptrRow->[0];
    }
    $wrapDbh->{'FINISH'}->();

    foreach $id (@ids){
      $item = &makeCertificateNode($id);
      if(ref($item)=~/HASH/){
		$root->{'ADDCHILD'}->($item);
      }
    }
  }
#
# end certificates
#
  printToLog("Dumping domain user");

  my $domUserNode = makeXmlNode('domainuser');
  $domUserNode->{'ADDCHILD'}->(makePasswordNode('', 'plain'));

  $sql = "SELECT val FROM dom_param WHERE dom_id = $domainID AND param = 'cl_manage_dns'";
  if ($wrapDbh->{'EXECUTE'}->($sql)) {
	my $value = @{$wrapDbh->{'FETCHROW'}->()}[0];
	addPermissionNode($domUserNode, 'manage_dns', ($value eq '1' ? 'true' : 'false'));
  }
  $wrapDbh->{'FINISH'}->();

  $sql = "SELECT val FROM dom_param WHERE dom_id = $domainID AND param = 'cl_manage_log'";
  if ($wrapDbh->{'EXECUTE'}->($sql)) {
	my $value = @{$wrapDbh->{'FETCHROW'}->()}[0];
	addPermissionNode($domUserNode, 'manage_log', ($value eq '1' ? 'true' : 'false'));
  }
  $wrapDbh->{'FINISH'}->();

  printToLog("Getting domain permissions");

  if ($id = $domParams{'perm_id'}) {
    addPermissions($domUserNode,$id);
  }

  $root->{'ADDCHILD'}->($domUserNode);
  
  printToLog("Dumping domain hosting");
#
# hosting
#
  $item=undef;
  if($domainType=~/vrt_hst/){
    $item = &makePhostingNode(\%domain);

	my $domainRoot = $PsaConfig{'HTTPD_VHOSTS_D'}.'/'.$domainName;
	my $fileName;
	#
	# configuration files (conf/vhost.conf)
	#
	my $vhostsConf = $domainRoot .'/conf/vhost.conf';
	if (-f $vhostsConf) {
	  if($fileName = makeDumpFile("$dumpDir/$domainName.conf", "$domainRoot/conf", "vhost.conf")) {
		$root->{'ATTRIBUTE'}->('cid_conf', $fileName);
	  }
	}
	#
	# end configuration files (conf/vhost.conf)
	#


  }elsif($domainType=~/std_fwd/){
    $item = &makeShostingNode(\%domain);
  }elsif($domainType=~/frm_fwd/){
    $item = &makeFhostingNode(\%domain);
  }
  if(ref($item)=~/HASH/){
    $root->{'ADDCHILD'}->($item);
  }

#
# end hosting
#

  return $root;
}

sub addWwwStatus {
  my ($parent, $domainID, $domainName) = @_;

  my $sql = "SELECT * FROM dns_recs WHERE dom_id = $domainID AND type = 'CNAME' " .
	"AND host = 'www.$domainName.'";

  if ($wrapDbh->{'EXECUTE'}->($sql)) {
	$parent->{'ATTRIBUTE'}->('www', 'true');
  } else {
	$parent->{'ATTRIBUTE'}->('www', 'false');
  }
  $wrapDbh->{'FINISH'}->();
}

#
# addCurrentTraffic - add current traffic
#
#     arguments:
#                 $root - XML node to add traffic's nodes
#                 $domainId - ID of domain
#
sub addCurrentTraffic {
  my ($root, $domainId) = @_;
  my ($sql, $ptrRow, $item);

  my $trafficValue = '';

  $sql = "SELECT transfer, date FROM stat WHERE dom_id=$domainId";
  if ($wrapDbh->{'EXECUTE'}->($sql)) {
	while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
	  if ($ptrRow->[0]) {
		$trafficValue .= $ptrRow->[1];
		$trafficValue .= ' http in ';
		$trafficValue .= $ptrRow->[0];
		$trafficValue .= "\n";
	  }
	}
  }
  $wrapDbh->{'FINISH'}->();

  $root->{'ADDCHILD'}->(makeXmlNode('traffic', $trafficValue));
}

sub getCatchAllAddress
{
	my ($parent, $domainId) = @_;

	my ($sql, $catchAllMode);
	$sql = "SELECT dp.val FROM dom_param dp WHERE dp.dom_id = $domainId AND dp.param = 'nonexist_mail'";

	if ($wrapDbh->{'EXECUTE'}->($sql)) {
		$catchAllMode = @{$wrapDbh->{'FETCHROW'}->()}[0];
	}
	$wrapDbh->{'FINISH'}->();

	if ($catchAllMode eq 'bounce') {

		$sql = "SELECT dp.val FROM dom_param dp WHERE dp.dom_id = $domainId AND dp.param = 'bounce_mess'";
		if ($wrapDbh->{'EXECUTE'}->($sql)) {
			my $bounceAddr = @{$wrapDbh->{'FETCHROW'}->()}[0];
			if ($bounceAddr ne '') {
				$parent->{'ADDCHILD'}->(makeXmlNode('catch-all', 'bounce:' . $bounceAddr));
			}
		}
		$wrapDbh->{'FINISH'}->();

	} elsif ($catchAllMode eq 'catch') {

		my $redirAddr;

		$sql = "SELECT m.redir_addr FROM mail m WHERE m.dom_id = $domainId";
		if ($wrapDbh->{'EXECUTE'}->($sql)) {
			$redirAddr = @{$wrapDbh->{'FETCHROW'}->()}[0];
		}
		$wrapDbh->{'FINISH'}->();

		if (!$redirAddr || $redirAddr eq '') {
			$sql = "SELECT dp.val FROM dom_param dp WHERE dp.dom_id = $domainId AND dp.param = 'catch_addr'";
			if ($wrapDbh->{'EXECUTE'}->($sql)) {
				$redirAddr = @{$wrapDbh->{'FETCHROW'}->()}[0];
			}
			$wrapDbh->{'FINISH'}->();
		}

		if ($redirAddr ne '') {
			$parent->{'ADDCHILD'}->(makeXmlNode('catch-all', $redirAddr));
		}
	}
}


sub addUrlDecodedTextNode {
  my ($parent, $name, $value) = @_;

  $parent->{'ADDCHILD'}->(makeXmlNode($name, urlDecode($value))) if ($value);
}

sub makeCertificateNode {
  my($certId)=@_;

  my($sql,$root,%cert,$item);

  $sql = "SELECT * FROM certificates WHERE id=$certId";
  unless($wrapDbh->{'EXECUTE'}->($sql)){
    &printToError("Error: makeCertificateNode: certificate ID '$certId' is not found");
    return undef;
  }

  %cert= %{$wrapDbh->{'FETCHHASH'}->()};
  $wrapDbh->{'FINISH'}->();

  $root = makeXmlNode('certificate');

  addUrlDecodedTextNode($root, 'certificate-data', $cert{'pub_key'});
  addUrlDecodedTextNode($root, 'private-key', $cert{'pvt_key'});
  $root->{'ATTRIBUTE'}->('name', $cert{'name'});

  return $root;
}

my(%PostgreSqlAdmin);

sub makeDatabaseNode {
  my ($dbId, $dbName, $dbTypePlesk) = @_;
  my ($root, $sql, $ptrHash, $ptrRow, $dbType, $item);

  printToLog(">Dumping " . $dbName);

  if ($dbTypePlesk =~ /postgres/) {
	$dbTypePlesk = "postgresql";
  }

  printToLog(">Creating XML node");

  $root = makeXmlNode('database');
  $root->{'ATTRIBUTE'}->('name', $dbName);
  $root->{'ATTRIBUTE'}->('type', $dbTypePlesk);
  if ($dbTypePlesk=~/mysql/){
    $root->{'ATTRIBUTE'}->('version', Db::MysqlUtils::getVersion());
  }

  printToLog("Adding db users");

  $wrapDbh->{'EXECUTE'}->("SELECT login, passwd FROM db_users WHERE db_id = $dbId");

  while ($ptrHash = $wrapDbh->{'FETCHHASH'}->()) {
	my $dbPasswdType;

	$dbPasswdType = 'plain';

	my $dbUserRoot = makeXmlNode('dbuser');
	$dbUserRoot->{'ATTRIBUTE'}->('name', $ptrHash->{'login'});

	$item = makePasswordNode($ptrHash->{'passwd'}, $dbPasswdType);

	$dbUserRoot->{'ADDCHILD'}->($item);
	$root->{'ADDCHILD'}->($dbUserRoot);
  }
  $wrapDbh->{'FINISH'}->();

  printToLog("Determining DB connection settings");
#
# dump
#
  my ($dbPasswd, $dbUser, $dbPasswd, $dbHost);
  if($dbTypePlesk=~/mysql/){

    $dbType = 'mysql';
    $dbUser = $wrapDbh->{'USER'}->();
    $dbPasswd = $wrapDbh->{'PASSWORD'}->();
    $dbHost = $wrapDbh->{'HOST'}->();

  }elsif($dbTypePlesk=~/postgresql/){

    $dbType = 'Pg';
    unless($dbUser && defined($dbPasswd)){ ## if is not any user try to work with admin-account
      unless(keys %PostgreSqlAdmin){
		$sql = "SELECT param, val FROM misc WHERE param like 'postgresql_admin_\%'";
		if($wrapDbh->{'EXECUTE'}->($sql)){
		  while($ptrRow=$wrapDbh->{'FETCHROW'}->()){
			if($ptrRow->[0]=~/login$/){
			  $PostgreSqlAdmin{'user'}=$ptrRow->[1];
			}elsif($ptrRow->[0]=~/passwd$/){
			  $PostgreSqlAdmin{'password'}=$ptrRow->[1];
			}
		  }
		}
		$wrapDbh->{'FINISH'}->();
      }
      $dbUser = $PostgreSqlAdmin{'user'};
      $dbPasswd = $PostgreSqlAdmin{'password'};
    }

  }

  printToLog("Dumping");

  if($dumpFile = &makeDumpDb($dbName,$dbType,$dbUser,$dbPasswd,$dbHost)){
    $root->{'ATTRIBUTE'}->('cid',$dumpFile);
  }
#
# end dump
#


  return $root;
}

#
# makeMailUserNode
#
# argumets:
#           $mailId - mail ID
#           $passwd - password
#           $typePasswd - type of password
#           $domainName - name of domain
#
# return:
#           $root = XML node 'mailuser'
#
sub makeMailUserNode {
  my($mailId,$passwd,$typePasswd,$domainName)=@_;

  my($sql,%mail,$item,$mailName,$ptrRow,$ptrHash,$dir,$id, $mbox_quota);

  $sql = "SELECT * FROM mail WHERE id = $mailId";
  unless($wrapDbh->{'EXECUTE'}->($sql)){
    &printToError("Error: makeMailUserNode: mail ID '$mailId' is not found");
    return undef;
  }
  %mail = %{$wrapDbh->{'FETCHHASH'}->()};
  $wrapDbh->{'FINISH'}->();
  my $root = &makeXmlNode('mailuser');
  $mailName = $mail{'mail_name'};
  $root->{'ATTRIBUTE'}->('name',$mailName);

  if(defined($passwd) && $passwd){
	if($passwd =~ /NULL/) {
	  $item = &makePasswordNode('', 'plain');
	} else {
	  $item = &makePasswordNode($passwd,$typePasswd);
	}

    $root->{'ADDCHILD'}->($item);
  }

  if($mail{'postbox'}=~/true/){
    $item = &makeXmlNode('mailbox');
    $item->{'ATTRIBUTE'}->('type','mdir');
    $root->{'ADDCHILD'}->($item);

    $dir=$PsaConfig{'PLESK_MAILNAMES_D'}."/$domainName/$mailName/Maildir";
    $dumpFile = &makeDumpFile("$dumpDir/$mailName\@$domainName.mdir",
			     $dir);
    if($dumpFile){
      $item->{'ATTRIBUTE'}->('cid',$dumpFile);
    }
  }

  if($mail{'mbox_quota'}){
	$mbox_quota = $mail{'mbox_quota'};

	$mbox_quota = $mbox_quota * 1024;
	$root->{'ATTRIBUTE'}->('mailbox-quota',$mbox_quota);
  }

#
# redirect
#
  if ($mail{'redir_addr'}) {
    $item = makeXmlNode('redirect',$mail{'redir_addr'});
	$item->{'ATTRIBUTE'}->('enabled', $mail{'redirect'});
    $root->{'ADDCHILD'}->($item);
  }
#
# end redirect
#

#
# mail group
#
  $root->{'ATTRIBUTE'}->('mailgroup-enabled', $mail{'mail_group'});

  $sql="SELECT address FROM mail_redir WHERE mn_id=$mailId";
  if($wrapDbh->{'EXECUTE'}->($sql)){
	while ($ptrRow=  $wrapDbh->{'FETCHROW'}->()){
	  $item=&makeXmlNode('mailgroup-member',$ptrRow->[0]);
	  $root->{'ADDCHILD'}->($item);
	}
  }
  $wrapDbh->{'FINISH'}->();

#
# end mail group
#

#
# autoresponders
#
  my $autorespondersNode = makeXmlNode('autoresponders');
  $autorespondersNode->{'ATTRIBUTE'}->('enabled', $mail{'autoresponder'});

  $dir=$PsaConfig{'PLESK_MAILNAMES_D'}."/$domainName/$mailName/\@attachments";
  $dumpFile = &makeDumpFile("$dumpDir/$mailName\@$domainName.attach",
							$dir);
  if($dumpFile){
	$autorespondersNode->{'ATTRIBUTE'}->('cid_attaches',$dumpFile);
  }

  my(@autos);
  $sql = "SELECT id FROM mail_resp WHERE mn_id=$mailId ORDER BY id";
  if($wrapDbh->{'EXECUTE'}->($sql)){
	while ($ptrRow = $wrapDbh->{'FETCHROW'}->()){
	  push @autos,$ptrRow->[0];
	}
  }
  $wrapDbh->{'FINISH'}->();

  my %attaches;
  my @autoresponders;

  foreach $id (@autos) {
	my ($item, $attaches) = &makeAutoresponderNode($id, $mailName . "@" . $domainName);

	push @autoresponders, $item;
	$attaches{$_}=1 foreach @{$attaches};
  }

  foreach my $attach (keys %attaches) {
	my $attachNode = makeXmlNode('attach');
	$attachNode->{'ATTRIBUTE'}->('file', $attach);
	$autorespondersNode->{'ADDCHILD'}->($attachNode);
  }
  foreach my $autoresponder (@autoresponders) {
	$autorespondersNode->{'ADDCHILD'}->($autoresponder);
  }
  $root->{'ADDCHILD'}->($autorespondersNode);
#
# end autoresponders
#

  return $root;
}

#
# makeAutoresponderNode
#
#  arguments:
#             $id - ID of autoreponder
#
#  return:
#             $root - XML node
#
sub makeAutoresponderNode {
  my($autoId, $mailName)=@_;

  my($name,$value,$sql,%auto,$ptrRow,$item);

  $sql = "SELECT * FROM mail_resp WHERE id=$autoId";
  unless($wrapDbh->{'EXECUTE'}->($sql)){
    &printToError("Error: makeAutoresponderNode: autoresponder # $autoId is not found");
    return undef;
  }

  %auto=%{$wrapDbh->{'FETCHHASH'}->()};
  $wrapDbh->{'FINISH'}->();

  my $root = &makeXmlNode('autoresponder');

  $value=$wrapBase64->{'ENCODE'}->($auto{'text'});
  $item=&makeXmlNode('text',$value);
  if (exists $auto{'charset'} && $auto{'charset'} ne '') {
	$item->{'ATTRIBUTE'}->('charset', $auto{'charset'});
  }
  $root->{'ADDCHILD'}->($item);

  if (exists $auto{'content_type'} && $auto{'content_type'}) {
	$root->{'ATTRIBUTE'}->('content-type', $auto{'content_type'});
  }

  $name = $auto{'resp_name'};
  $root->{'ATTRIBUTE'}->('name',$name);

  if ($auto{'resp_on'}=~/false/){
    $root->{'ATTRIBUTE'}->('status','off');
  }

  unless ($auto{'key_where'}=~/no/){
    $value=$auto{'key_where'}.':'.$auto{'keystr'};
    $root->{'ATTRIBUTE'}->('require',$value);
  }

  if($auto{'subject'}){
    $value = $wrapBase64->{'ENCODE'}->($auto{'subject'});
    chomp $value;
    $root->{'ATTRIBUTE'}->('subject',$value);
  }

  if ($auto{'reply_to'} and $auto{'reply_to'} ne $mailName) {
	$root->{'ATTRIBUTE'}->('replyto',$auto{'reply_to'});
  }

#
# forward
#
  $sql="SELECT address FROM resp_forward WHERE rn_id=$autoId";
  if($wrapDbh->{'EXECUTE'}->($sql)){
    my (@list);
    while($ptrRow=$wrapDbh->{'FETCHROW'}->()){
      push @list, $ptrRow->[0] if $ptrRow->[0];
    }
	if (@list) {
	    $root->{'ATTRIBUTE'}->('redirect',join(',',@list));
	}
  }
  $wrapDbh->{'FINISH'}->();
#
# end forward
#
  $item = &makeXmlNode('autoresponder-limit',$auto{'ans_freq'});
  $item->{'ATTRIBUTE'}->('name','ans-freq');
  $root->{'ADDCHILD'}->($item);

  $item = &makeXmlNode('autoresponder-limit',$auto{'mem_limit'});
  $item->{'ATTRIBUTE'}->('name','ans-count');
  $root->{'ADDCHILD'}->($item);


  my @attaches;
#
# attachment
#
  $sql = "SELECT filename FROM resp_attach WHERE rn_id=$autoId";
  if($wrapDbh->{'EXECUTE'}->($sql)){

    while($ptrRow=$wrapDbh->{'FETCHROW'}->()){
      $item=&makeXmlNode('attach');
      $item->{'ATTRIBUTE'}->('file',$ptrRow->[0]);
      $root->{'ADDCHILD'}->($item);

	  push @attaches, $ptrRow->[0];
    }
  }
  $wrapDbh->{'FINISH'}->();
#
# end attachment
#

  return ($root, \@attaches);
}

sub createCgiBinNode {
  my ($root, $vhostRoot, $archiveName) = @_;
  my $fileName;
  `cp -r '$vhostRoot/cgi-bin' '$dumpDir/cgi-bin'`;
  `find $dumpDir/cgi-bin -type f -print0 | xargs -0 -n1 perl -pi -e 's/\\/usr\\/local\\/slash\\/apache\\//\\/home\\/httpd\\//g'`;
  if($fileName = makeDumpFile($dumpDir."/".$archiveName,
							   $dumpDir."/cgi-bin")){
	$root->{'ATTRIBUTE'}->('cid_cgi',$fileName);
  }
  `rm -rf '$dumpDir/cgi-bin'`;
}

sub makePhostingNode {
  my ($ptrDomain)=@_;
  unless(ref($ptrDomain)=~/HASH/){
    &printToError("Error: makePhostNode: bag arguments");
    return undef;
  }
  my $root=&makeXmlNode('phosting');

  my($domainName,$domainRoot,$path,$fileName,$sql,%hosting,$domainId,
    $xmlName,$fieldName,$id,$item,$ptrRow,$ptrHash);
  $domainName = $ptrDomain->{'name'};
  $domainId = $ptrDomain->{'id'};

  $domainRoot=$PsaConfig{'HTTPD_VHOSTS_D'}.'/'.$domainName;
  if (-d $domainRoot) {
    if($fileName = &makeDumpFile("$dumpDir/$domainName.htdocs",
				 "$domainRoot/httpdocs")){
      $root->{'ATTRIBUTE'}->('cid_docroot',$fileName);
    }
    if($fileName = &makeDumpFile("$dumpDir/$domainName.shtdocs",
				 "$domainRoot/httpsdocs")){
      $root->{'ATTRIBUTE'}->('cid_docroot_ssl',$fileName);
    }
	createCgiBinNode($root, $domainRoot, "$domainName.cgi");


	if (-d $domainRoot . "/statistics/webstat") {
	  if ($fileName = makeDumpFile("$dumpDir/$domainName.webstat",
								   "$domainRoot/statistics/webstat")) {
		$root->{'ATTRIBUTE'}->('cid_webstat', $fileName);
	  }
	}

	if (-d $domainRoot . "/statistics/webstat-ssl") {
	  if ($fileName = makeDumpFile("$dumpDir/$domainName.webstat-ssl",
								   "$domainRoot/statistics/webstat-ssl")) {
		$root->{'ATTRIBUTE'}->('cid_webstat_ssl', $fileName);
	  }
	}

	if (-d $domainRoot . "/error_docs") {
	  if ($fileName = makeDumpFile("$dumpDir/$domainName.error_docs",
								   $domainRoot . "/error_docs")) {
		$root->{'ATTRIBUTE'}->('cid_error_docs', $fileName);
	  }
	}
  }

  $sql = "SELECT * FROM hosting WHERE dom_id=$domainId";
  if($wrapDbh->{'EXECUTE'}->($sql)){
    %hosting = %{$wrapDbh->{'FETCHHASH'}->()};
  }
  $wrapDbh->{'FINISH'}->();

  while (($xmlName,$fieldName)=each(%hostingAttribute)){
    if($hosting{$fieldName}=~/true/){
      $root->{'ATTRIBUTE'}->($xmlName,'true');
    }
  }

  $sql = "SELECT val FROM dom_param WHERE dom_id='$domainId' "
	. "AND param='apacheErrorDocs'";
  if ($wrapDbh->{'EXECUTE'}->($sql)) {
	if ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
	  my $errdocs = (@{$ptrRow})[0];
	  if ($errdocs eq 'true') {
		$root->{'ATTRIBUTE'}->('errdocs', 'true');
	  }
	}
  }

#
# sysuser
#
  $item = &makeSyntheticSysUserNode($hosting{'login'}, $hosting{'passwd'}, 'plain');
  if(ref($item) =~ /HASH/) {
	$root->{'ADDCHILD'}->($item);
  }
#
# end sysuser
#

#
# scripting
#
  $item = &makeXmlNode('scripting');
  while(($xmlName,$fieldName)=each(%hostingScripting)){
    if($hosting{$fieldName}=~/true/){
      $item->{'ATTRIBUTE'}->($xmlName,'true');
    }
  }
  $root->{'ADDCHILD'}->($item);
#
# end scripting
#

#
# frontpage user
#
  if($hosting{'fp_adm'}){
    $item = &makeXmlNode('fpuser');
    $item->{'ATTRIBUTE'}->('name',$hosting{'fp_adm'});
    if($hosting{'fp_pass'}){
      my $passNode = &makePasswordNode($hosting{'fp_pass'},'plain');
      $item->{'ADDCHILD'}->($passNode);
    }
    $root->{'ADDCHILD'}->($item);
  }
#
# end frontpage user
#

#
# anonftp
#
  &addAnonFtp($root,$domainId,$domainName);
#
# end anonftp
#

#
# protected dirs
#
  $sql = "SELECT id,path,realm FROM protected_dirs ".
    "WHERE dom_id=$domainId ORDER BY id";
  if($wrapDbh->{'EXECUTE'}->($sql)){
    my (@dirs);
    while ($ptrRow = $wrapDbh->{'FETCHROW'}->()){
      push @dirs,[@{$ptrRow}];
    }
    $wrapDbh->{'FINISH'}->();
    foreach $ptrRow (@dirs){
      $item = &makeProtDirNode(@{$ptrRow});
      if(ref($item)=~/HASH/){
		$root->{'ADDCHILD'}->($item);
      }
    }
  }
  $wrapDbh->{'FINISH'}->();
#
# end protected dirs
#

#
# web users
#
  $sql = "SELECT * FROM web_users WHERE dom_id=$domainId ORDER BY id";
  if($wrapDbh->{'EXECUTE'}->($sql)){
    my (@webs);
    while ($ptrHash = $wrapDbh->{'FETCHHASH'}->()){
      push @webs,{%{$ptrHash}};
    }
    $wrapDbh->{'FINISH'}->();
    foreach $ptrHash (@webs){
      $item = &makeWebUserNode($ptrHash,$domainName);
      if(ref($item)=~/HASH/){
		$root->{'ADDCHILD'}->($item);
      }
    }
  }
#
# end web users
#

  return $root;
}

#
# makeSubDomainNode
#
#  argumenets:
#       $ptrSubDomain - pointer to hash with subdomain's info (row of table 'subdomains')
#       $domainName - name of domain
#
#  return:
#       $root = XMl node
#
sub makeSubDomainNode {
  my($ptrSubDomain,$domainName)=@_;
  unless(ref($ptrSubDomain)=~/HASH/){
    &printToError("Error: makeSubDomainNode: bad argumets");
    return undef;
  }
  my($root,$subDomainRoot,$fileName,$subDomainName,$item,$xmlName,$fieldName);
  $root = &makeXmlNode('subdomain');
  $subDomainName = $ptrSubDomain->{'name'};
  $root->{'ATTRIBUTE'}->('name',$subDomainName);

#
# content
#
  $subDomainRoot=$PsaConfig{'HTTPD_VHOSTS_D'}."/$domainName/subdomains/$subDomainName";
  if(-d $subDomainRoot){
    if($fileName = &makeDumpFile("$dumpDir/$subDomainName.$domainName.htdocs",
			      "$subDomainRoot/httpdocs")){
      $root->{'ATTRIBUTE'}->('cid_docroot',$fileName);
    }
	createCgiBinNode($root, $subDomainRoot, "$subDomainName.$domainName.cgi");
  }
#
# end content
#
  if($ptrSubDomain->{'sys_user_type'}=~/native/){
    $item = &makePleskSysUserNode($ptrSubDomain->{'sys_user_id'});
    if(ref($item)=~/HASH/){
      $root->{'ADDCHILD'}->($item);
    }
  }

#
# scripting
#
  $item = &makeXmlNode('scripting');
  while(($xmlName,$fieldName)=each(%subDomainScripting)){
    if($ptrSubDomain->{$fieldName}=~/true/){
      $item->{'ATTRIBUTE'}->($xmlName,'true');
    }
  }
  $root->{'ADDCHILD'}->($item);
#
# end scripting
#

  return $root;
}


#
# makeWebUserNode
#
#  argumenets:
#       $ptrWebUser - pointer to hash with web user's info (row of table 'web_users')
#       $domainName - name of domain
#
#  return:
#       $root = XML node
#
sub makeWebUserNode {
  my($ptrWebUser,$domainName)=@_;
  unless(ref($ptrWebUser)=~/HASH/){
    &printToError("Error: makeWebUserNode: bad argumets");
    return undef;
  }
  my($root,$home,$fileName,$userName,$item,$xmlName,$fieldName);
  $root = &makeXmlNode('webuser');


	my $username = $ptrWebUser->{'username'};
	my @pwnam = getpwnam($username);
	my $password = $pwnam[1];
	$item = &makeSyntheticSysUserNode($username, $password, 'encrypted');
	$root->{'ADDCHILD'}->($item);
	$root->{'ATTRIBUTE'}->('name',$username);

#
# scripting
#
  $item = &makeXmlNode('scripting');
  while(($xmlName,$fieldName)=each(%webUserScripting)){
    if($ptrWebUser->{$fieldName}=~/true/){
      $item->{'ATTRIBUTE'}->($xmlName,'true');
    }
  }
  $root->{'ADDCHILD'}->($item);
#
# end scripting
#

  return $root;
}

#
# makeProtDirNode - make protected directory node
#
# arguments:
#        $pdirId - ID
#        $pdirPath - path of protected directory
#        $pdirTitke - header text
#
# return:
#        $root - XML node
#
sub makeProtDirNode {
  my($pdirId, $pdirPath, $pdirTitle) = @_;

  my($root, $sql, $userNode, $item, $ptrRow);

  $root = makeXmlNode('pdir');
  # workaround of CLI inabliity to create '' directory.
  $pdirPath = '/' . $pdirPath;

  $root->{'ATTRIBUTE'}->('name', $pdirPath);
  if($pdirTitle) {
    $root->{'ATTRIBUTE'}->('title', $pdirTitle);
  }

    $sql = "SELECT p.login, p.passwd, 'plain' FROM pd_users p ".
      " WHERE p.pd_id=$pdirId ORDER BY p.id";

  if ($wrapDbh->{'EXECUTE'}->($sql)) {
    while ($ptrRow=$wrapDbh->{'FETCHROW'}->()) {
      $userNode = makeXmlNode('pduser');
      $root->{'ADDCHILD'}->($userNode);
      $userNode->{'ATTRIBUTE'}->('name', $ptrRow->[0]);
      $item = makePasswordNode($ptrRow->[1], $ptrRow->[2]);
      $userNode->{'ADDCHILD'}->($item);
    }
  }
  $wrapDbh->{'FINISH'}->();
  return $root;
}

sub makeAnonftpPermissionNode {
  my ($parent, $name) = @_;
  my $node = makeXmlNode('anonftp-permission');
  $node->{'ATTRIBUTE'}->('name', $name);
  $parent->{'ADDCHILD'}->($node);
}

sub makeAnonftpLimitNode {
  my ($parent, $name, $value) = @_;
  if($value != 0) {
	my $node = makeXmlNode('anonftp-limit', $value);
	$node->{'ATTRIBUTE'}->('name', $name);
	$parent->{'ADDCHILD'}->($node);
  }
}

sub addAnonFtp {
  my($root,$domainId,$domainName)=@_;
  unless(ref($root)=~/HASH/){
    &printToError("Error: addAnonFtp: bad argumets");
    return undef;
  }

  my($anonRoot,$ptrHash,$count,$fileName,$domainRoot,$sql);
  $count=0;
  $sql="SELECT * FROM anon_ftp WHERE dom_id=$domainId";
  if($wrapDbh->{'EXECUTE'}->($sql)){
    while($ptrHash=$wrapDbh->{'FETCHHASH'}->()){

      $anonRoot = &makeXmlNode('anonftp');
      $root->{'ADDCHILD'}->($anonRoot);
      if($ptrHash->{'status'}=~/true/){
		$anonRoot->{'ATTRIBUTE'}->('pub','true');
      }
      if($ptrHash->{'incoming'}=~/true/){
		$anonRoot->{'ATTRIBUTE'}->('incoming','true');
      }
	  if (defined($ptrHash->{'max_conn'})) {
		makeAnonftpLimitNode($anonRoot,'max-connections', $ptrHash->{'max_conn'});
	  }
	  if (defined($ptrHash->{'bandwidth'})) {
		makeAnonftpLimitNode($anonRoot, 'bandwidth', $ptrHash->{'bandwidth'});
	  }
	  if (defined($ptrHash->{'quota'})) {
		makeAnonftpLimitNode($anonRoot, 'incoming-disk-quota', $ptrHash->{'quota'});
	  }
	  if (defined($ptrHash->{'display_login'})) {
		$anonRoot->{'ATTRIBUTE'}->('display-login', $ptrHash->{'display_login'});
	  }
	  if ($ptrHash->{'incoming_readable'} =~ /true/) {
		makeAnonftpPermissionNode($anonRoot, 'incoming-download');
	  }
	  if ($ptrHash->{'incoming_subdirs'} =~ /true/) {
		makeAnonftpPermissionNode($anonRoot, 'incoming-mkdir');
	  }
	  if (defined($ptrHash->{'login_text'})) {
		my $loginMessageNode = makeXmlNode('login-message', $ptrHash->{'login_text'});
		$anonRoot->{'ADDCHILD'}->($loginMessageNode);
	  }
      $domainRoot=$PsaConfig{'HTTPD_VHOSTS_D'}.'/'.$domainName;
      if(-d $domainRoot){

		if($fileName = &makeDumpFile("$dumpDir/$domainName.anonftp.pub",
									 "$domainRoot/anon_ftp/pub")){
		  $anonRoot->{'ATTRIBUTE'}->('cid',$fileName);
		}
		if($fileName = &makeDumpFile("$dumpDir/$domainName.anonftp.incoming",
									 "$domainRoot/anon_ftp/incoming")){
		  $anonRoot->{'ATTRIBUTE'}->('cid_incoming',$fileName);
		}
      }
      $count++;
    }
  }
  $wrapDbh->{'FINISH'}->();
  return $count;
}

sub makePleskSysUserNode {
  my ($sysUserId)=@_;

  my ($sql,$root,%sysuser, $quota);
  $sql = "SELECT * FROM sys_users WHERE id=$sysUserId";
  unless($wrapDbh->{'EXECUTE'}->($sql)){
    &printToError("Error: makeSysUser: sys user #$sysUserId  is not found");
    return undef;
  }
  %sysuser=%{$wrapDbh->{'FETCHHASH'}->()};
  $wrapDbh->{'FINISH'}->();

  $root = &makeXmlNode('sysuser');

#
# attributes
#
  $root->{'ATTRIBUTE'}->('name',lc($sysuser{'login'}));
  $root->{'ATTRIBUTE'}->('shell',$sysuser{'shell'}) if $sysuser{'shell'};

  if($sysuser{'quota'}){
	unless($sysuser{'quota'} eq "-1") {
	  $quota = $sysuser{'quota'}*1024*1024;
	}
	$root->{'ATTRIBUTE'}->('quota',$quota) if $quota;
  }

#
# end attributes
#

#
# password node
#
  $root->{'ADDCHILD'}->(makePasswordNode($sysuser{'passwd'}, 'plain'));
#
# end password node
#
  return $root;

}

sub makeSyntheticSysUserNode {
  my ($name, $password, $passtype) = @_;

  my $root = &makeXmlNode('sysuser');
  $root->{'ATTRIBUTE'}->('name', $name);
  my $passwdNode = &makePasswordNode($password, $passtype);
  $root->{'ADDCHILD'}->($passwdNode);
  return $root;
}

sub makeFhostingNode {
  my ($ptrDomain)=@_;
  unless(ref($ptrDomain)=~/HASH/){
    &printToError("Error: makeFhostingNode: bag arguments");
    return undef;
  }
  my($sql,$domainId,$root,$forward);
  $domainId=$ptrDomain->{'id'};
  $sql="SELECT redirect FROM forwarding WHERE dom_id=$domainId";
  if($wrapDbh->{'EXECUTE'}->($sql)){
    ($forward)=@{$wrapDbh->{'FETCHROW'}->()};
    $root=&makeXmlNode('fhosting',$forward);
  }else{
    &printToError("Error: makeFhostingNode: forward for domain '".$ptrDomain->{'name'}."' is not found");
  }
  $wrapDbh->{'FINISH'}->();

  return $root;
}

sub makeShostingNode {
  my ($ptrDomain)=@_;
  unless(ref($ptrDomain)=~/HASH/){
    &printToError("Error: makeShostingNode: bag arguments");
    return undef;
  }

  my($sql,$domainId,$root,$forward);
  $domainId=$ptrDomain->{'id'};
  $sql="SELECT redirect FROM forwarding WHERE dom_id=$domainId";
  if($wrapDbh->{'EXECUTE'}->($sql)){
    ($forward)=@{$wrapDbh->{'FETCHROW'}->()};
    $root=&makeXmlNode('shosting',$forward);
  }else{
    &printToError("Error: makeShostingNode: forward for domain '".$ptrDomain->{'name'}."' is not found");
  }
  $wrapDbh->{'FINISH'}->();

  return $root;
}

##########################################################
#
#	Reads IP information. Alters %exclIp and $defaultIp
#
##########################################################
sub readIpInfo( $ ) 
{
	my ($ptrClients) = @_;

	my ($ptrRow,$ptrHash,$sql,$list,@names,$domain);

	# Get domains
	$sql = "SELECT d.name, d.id, r.val, a.status, d.cl_id FROM dns_recs r, domains d "
		. "LEFT JOIN hosting h ON h.dom_id=d.id "
		. "LEFT JOIN anon_ftp a ON a.dom_id=d.id "
		. "WHERE r.dom_id=d.id AND r.type='A' AND r.host=CONCAT(d.name, '.')";

	if (ref($ptrClients) =~ /HASH/) {
		$list = getSqlList(keys %{$ptrClients});
		$sql .= " AND cl_id IN ($list)"
	}

	if ($wrapDbh->{'EXECUTE'}->($sql)) {
		while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
			$domainsInfo{$ptrRow->[0]} = [@{$ptrRow}[1..4]];
		}
	}
	$wrapDbh->{'FINISH'}->();

	# Get IP from DNS
	while (($domain,$ptrRow) = each (%domainsInfo)) {
		unless ($ptrRow->[1]) {
			push @names, $domain;
		}
	}
	if (@names) {
		$list = getSqlList(@names);
		$sql = "SELECT host, val FROM dns_recs WHERE type='A' AND host in ($list)";
		if ($wrapDbh->{'EXECUTE'}->($sql)) {
			while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
				if ($ptrRow->[1] =~ /^\d+\.\d+\.\d+\.\d+$/) {
					$domainsInfo{$ptrRow->[0]}->[1] = $ptrRow->[1];
				}
			}
		}
		$wrapDbh->{'FINISH'}->();
	}

	$sql = "SELECT val FROM misc WHERE param LIKE 'NameVirtualHost'";
	($wrapDbh->{'EXECUTE'}->($sql) && ($ptrRow = $wrapDbh->{'FETCHROW'}->()))
		or die "Cannot retrieve name-based hosts IP";
	my $defaultIp = (@{$ptrRow})[0];
	$wrapDbh->{'FINISH'}->();

	while (($domain, $ptrRow) = each (%domainsInfo)) {
		unless ($ptrRow->[1] && $ptrRow->[1] !~ /NULL/) {
			$domainsInfo{$domain}->[1] = $defaultIp;
		}
	}

	# Shared or exclusive
	while (($domain, $ptrRow) = each (%domainsInfo)) {
		unless ($ptrRow->[1] && $ptrRow->[1] !~ /NULL/) {
			$domainsInfo{$domain}->[1] = $defaultIp;
		}
		my $ip = $domainsInfo{$domain}->[1];
		$exclIp{$ip} = ($ip ne $defaultIp);
	}
}

sub makeIpNode( $ )
{
	my $ip = shift;

	my $type = 'shared';
	if ($exclIp{$ip}) {
		$type = 'exclusive';
	}

	my $ip_node = makeXmlNode('ip');
	$ip_node->{'ADDCHILD'}->(makeXmlNode('ip-type',    $type));
	$ip_node->{'ADDCHILD'}->(makeXmlNode('ip-address', $ip));
	return $ip_node;
}

#
# Returns password node for given account ID
#
sub makeAccountPasswordNode
{
	my $account_id = shift;
	my $node;

	if ($wrapDbh->{'EXECUTE'}->("SELECT password, type FROM accounts WHERE id='$account_id'")) {
		my ($passwd, $type) = @{$wrapDbh->{'FETCHROW'}->()};
		$node = &makePasswordNode($passwd, $type);
    } else {
		# generates a stub node
		$node = makePasswordNode('');
	}
	$wrapDbh->{'FINISH'}->();

	return $node;
}

sub printAgentStatus {
  my $root = makeXmlNode('agent-status');

  unless (defined AgentConfig::iconvBin()) {
    my $item = makeXmlNode('wrong-platform', 'no iconv found on the source host');
    $root->{'ADDCHILD'}->($item);
  }

  printXml($root, *STDOUT);
}

sub loadMainConfig
{
  my ($ptrHash) = @_;

  my $confFile = '/usr/local/slash/admin/conf/admin.conf';
  unless (-T $confFile) {
	return undef;
  }

  if (open(OUTPUT, "< $confFile")) {
	binmode OUTPUT;
	$ptrHash->{'password'} = <OUTPUT>;
	$ptrHash->{'password'} =~ s/^\s+(.*)\s+$/$1/;
	chomp $ptrHash->{'password'};
	close OUTPUT;
  }

  $ptrHash->{'HTTPD_VHOSTS_D'} = "/usr/local/slash/apache/vhosts";
  $ptrHash->{'PLESK_MAILNAMES_D'} = "/usr/local/slash/qmail/mailnames";

  return 1;
}

sub makeEmptyDump {
  my($rootName)=@_;
  my $root=&makeXmlNode($rootName);

  my $mapRoot = &makeXmlNode('map');
  $root->{'ADDCHILD'}->($mapRoot);

  my $item = &makeXmlNode('shared');
  $mapRoot->{'ADDCHILD'}->($item);

  $item = &makeXmlNode('exclusive');
  $mapRoot->{'ADDCHILD'}->($item);

  return $root;
}

sub printHelp {

  print <<"HELP";

Usage:
  $0 <options>

Options:
  -c |--config=<path>        path to 'psa.conf'

  -s |--get-status           get status of the agent
  -dc|--dump-accounts=<list> a coma separated list of resellers to dump
  -dd|--dump-domains=<list>  a coma separated list of customers to dump
  -da|--dump-all             make a full dump

  -lc|--get-content-list     get list of content files
  -nc|--no-content           do not make content files
  -nz|--no-compress          do not compress content files

  -h |--help                 this help

HELP
}
