package Transformer;

use Logging;
use XmlNode;
use Dumper;

#
# Begin global variables
#

# Hash of client names and client XmlNode elements, Plesk format
my %clientNodes;

# Initialized in initQuickInfo:
# Hash of client names and shallow client XmlNode elements, Plesk format
my %shallowClientNodes;

# Hash of Plesk client and cPanel accounts
my %clients2cpAccount;

# Hash of domains and its owners (clients)
my %domain2client;

# Hash of domains and its hosting type
my %domain2hosting;

my $initialized;
#
# End of global variables
#

#
# Begin constants:
#

# Maximum significant value for the disk space / traffic limit.
#	Values exceeding that would be counted as 'unlimited'
my $max_limit_value = 999999*1024*1024;

my %limit_map = ( 'max_traffic' => 'max_traffic',
                  'disk_space'  => 'disk_space',
                  'max_subdom'  => 'max_subdom',
                  'max_box'     => 'max_box',
                  'max_ftpuser' => 'max_wu');
#
# End of constants
#


#
# Begin service subs
#
sub initialize {
  # Get all accounts from Dumper
  # Get account' basic information, only need for getting the domain list
  # Should initialize %resellerNodes, %clientNodes and %domainNodes with 'undef'
  #
  unless ( $initialized ) {
    Dumper::initialize();

    # Get accounts
    my @cp_accounts = Dumper::getAllAccounts();
    foreach my $account ( sort @cp_accounts ) {
      my $cpAccountNode = Dumper::makeAccountNode( $account, undef, undef, 'shallow_mode');
      if ( ref($cpAccountNode) =~ /XmlNode/ ) {
        # Transformation now is realized in 'one-to-one' mode. This does not preserve cPanel accounts hierarhy.
        my $accountNode = transformAccountNode($cpAccountNode);
        unless ( defined $accountNode ) {
          Logging::error("Account '$accountNode' trandformation failed");
        }
      }
    }
    $initialized = 1;
  }
}

sub getResellers {
  my @resellers;
  # empty
  return @resellers;
}

sub getClients {
  my $owner = shift; # reseller that owns the clients returned. Could be 'undef' for default reseller ('root' or 'admin')

  my @clients;
  # empty
  return @clients;
}

sub getDomains {
  my $owner = shift; # reseller or client that owns the domains returned. Could be 'undef' for default reseller or client ('root' or 'admin')

  # Should return an array of clients identifiers, that could be a number, name or guid. Should be unique through migration dump.

  # The current implementation of cPanel migration deals with domains only.

  initialize();
  my @domains;

  unless ( defined $owner ) {
    foreach my $domain (sort keys %domain2client) {
      push @domains, $domain;
    }
  }

  return @domains;
}

#
# Basic transform routine
#
# Works in 'shallow mode' by default
# Important: dump in 'shallow_mode' is not required to be valid for plesk.xsd.
#
sub transformAccountNode {
  my ($cpAccountNode, $not_shallow_mode) = @_;

  if ( ref($cpAccountNode) =~ /XmlNode/ ) {
    if ($cpAccountNode->getName() eq 'account' ) {

      my $clientNode = XmlNode->new ('client');
      my $cp_account_name = $cpAccountNode->getAttribute( 'name' );
      my $client_name = $cp_account_name;

      $clientNode->setAttribute( 'name', $client_name );
      $clientNode->setAttribute( 'guid', '' );
        
      if ( $not_shallow_mode ) {
        $clientNode->setAttribute( 'contact', 'client ' . $client_name );
        my $crDate = $cpAccountNode->getAttribute( 'date' );
        $clientNode->setAttribute( 'cr-date', $crDate ) if ( defined $crDate );
        $clientNode->addChild( XmlNode->new( 'preferences' ));
        my $clientPropertiesNode = XmlNode->new( 'properties' );
        my $clientPasswordNode = &getClientPassword($cpAccountNode);
        $clientPropertiesNode->addChild( $clientPasswordNode ) if ( defined $clientPasswordNode );
        $clientPropertiesNode->addChild( &makeStatusEnabled() );
        $clientNode->addChild( $clientPropertiesNode );

        addClientLimitsAndPermissions( $clientNode, $cpAccountNode );
      }
      my $clientIpPoolNode = XmlNode->new( 'ip_pool' );
      $clientIpPoolNode->addChild( &getIp($cpAccountNode) );
      $clientNode->addChild( $clientIpPoolNode );

      my @domains = ();

      my $cpDomainNode = $cpAccountNode->getChild( 'domain' );
      if ( defined ( $cpDomainNode ) ) {

        # 1) Transform cPanel account's domain to Plesk domain
        my $domainNode = XmlNode->new ('domain');

        my $domain_name = $cpDomainNode->getAttribute( 'name' );
        $domainNode->setAttribute( 'name', $domain_name );

        $domainNode->setAttribute( 'guid', '' );

        my $crDate = $cpAccountNode->getAttribute( 'date' );
        $domainNode->setAttribute( 'cr-date', $crDate ) if ( defined $crDate );

        if ( $not_shallow_mode ) {
          my $domainPreferencesNode = &getDomainPreferences($cpDomainNode);
          $domainNode->addChild( $domainPreferencesNode );
        }

        my $domainPropertiesNode = &getDomainProperties($cpAccountNode);
        $domainNode->addChild( $domainPropertiesNode );

        if ( $not_shallow_mode ) {
          addDomainLimitsAndPermissions( $domainNode, $cpAccountNode );

          addDomainMailsystem( $domainNode, $cpDomainNode );

          addDomainMaillists( $domainNode, $cpDomainNode );

          addDomainDatabases( $domainNode, $cpDomainNode );

          addDomainuser( $domainNode, $cpAccountNode);

          my $domainPhostingNode = &getPhosting( $domainNode, $cpAccountNode);
          $domainNode->addChild( $domainPhostingNode );
        }

        unless( $not_shallow_mode ) {
          $domain2hosting{$domain_name} = 'phosting';
        }

        # Store 'domain' node. Then we'll create 'domains' node if any 'domain' node is created
        push @domains, $domainNode;

        # 2) Transform cPanel account's subdomains into Plesk domains
        my @cpSubdomainNodes = $cpDomainNode->getChildren( 'subdomain' );
        if ( @cpSubdomainNodes ) {
          foreach my $cpSubdomainNode (sort @cpSubdomainNodes) {
              my $has_ftp_account;
              $has_ftp_account = defined ( $cpSubdomainNode->getChild( 'ftpuser' ) );

              my @cpAddonDomainNodes = $cpSubdomainNode->getChildren( 'addondomain' );

              my $addon_domain_name;
              $addon_domain_name = $cpAddonDomainNodes[0]->getText() if ( @cpAddonDomainNodes );

              if ( $addon_domain_name ) {
                my $domainFromSubdomainNode = XmlNode->new ('domain');
                $domainFromSubdomainNode->setAttribute( 'name', $addon_domain_name );
                $domainFromSubdomainNode->setAttribute( 'guid', '' );

                # Subdomains which have addondomains and        have ftp account are mapped into Plesk domains with phosting
                # Subdomains which have addondomains but do not have ftp account are mapped into Plesk domains with shosting
                my $hosting_type = $has_ftp_account? 'phosting' : 'shosting' ;

                if ( $not_shallow_mode ) {
                  if ( $hosting_type eq 'phosting' ) {
                    # Transform subdomain to domain with hosting
                    &processSubdomain2Phosting($domainFromSubdomainNode, $cpSubdomainNode, $cpDomainNode, $cpAccountNode);
                  }
                  else {
                    # Transform subdomain to domain with shosting
                    &processSubdomain2Shosting($domainFromSubdomainNode, $cpSubdomainNode, $cpDomainNode, $cpAccountNode);
                  }
                }

                unless ( $not_shallow_mode ) {
                  $domain2hosting{$addon_domain_name} = $hosting_type;
                }

                # Store 'domain' node. Then we'll create 'domains' node if any 'domain' node is created
                push @domains, $domainFromSubdomainNode;
              }
          }
        }

        if ( @domains ) {
          my $domainsNode = XmlNode->new( 'domains' );
          foreach $domainNode ( @domains ) {
            $domainsNode->addChild( $domainNode );
            $domain2client{$domainNode->getAttribute( 'name' )} = $client_name;
          }
          $clientNode->addChild( $domainsNode );
        }
      }

      if ( $not_shallow_mode ) {
        $clientNodes{$client_name} = $clientNode;
      }
      else {
        $clients2cpAccount{$client_name} = $cp_account_name;
        $shallowClientNodes{$client_name} = $clientNode;
      }

      return $clientNode;

    }
  }
  return undef;
}


sub getClientNode4Domain {
  my $id = shift; # domain identifier from 'getDomains' result

  initialize();

  if ( exists ( $domain2client{$id} ) ){
    my $client_name = $domain2client{$id};

    return $clientNodes{$client_name} if exists ( $clientNodes{$client_name} );

    my $cpAccountNode = Dumper::makeAccountNode( $client_name, undef, undef);
    if ( Logging::getVerbosity() > 3 ) {
      my $cpAccountNodeDump = $cpAccountNode->serialize();
      Logging::debug("-" x 10 . " cPanel account dump" . "-" x 10);
      Logging::debug($cpAccountNodeDump);
      Logging::debug("-" x 30);
    }

    if ( ref($cpAccountNode) =~ /XmlNode/ ) {

      $clientNode = transformAccountNode($cpAccountNode, 'not_shallow_mode');
      if ( $clientNode ) {
        $clientNodes{$client_name} = $clientNode;
        return $clientNode;
      }
    }
    return undef;
  }
}
#
# End of service subs
#


#
# Begin node transformation subs
#
sub getClientPassword {
  my $cpAccountNode = shift;

  unless ( testXmlNodeParam($cpAccountNode, 'account', 'getClientPassword') ) {return undef;}

  # do nothing
  return undef;
}

sub addClientLimitsAndPermissions {
  my ($clientNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($clientNode, 'client', 'addClientLimitsAndPermissions') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode, 'account', 'addClientLimitsAndPermissions') ) {return undef;}

  # do nothing
}

sub getIp {
  my $cpAccountNode = shift;

  unless ( testXmlNodeParam($cpAccountNode, 'account', 'getIp') ) {return undef;}

  my $cpIpNode = $cpAccountNode->getChild( 'ip' );
  my $ipNode;
  if ( defined $cpIpNode) {
    $ipNode = XmlNode->new('ip');

    my $cpDomainNode = $cpAccountNode->getChild( 'domain' );
    my $allow_anon_ftp = 'false';
    if (defined $cpDomainNode) {
      $allow_anon_ftp = $cpDomainNode->getChildAttribute( 'anonftp', 'pub' );
    }

    # Plesk requires exclusive IP on domain to allow anon tfp access
    my $ipTypeNode = XmlNode->new( 'ip-type' );
    $ipTypeNode->setText( $allow_anon_ftp eq 'true'? 'exclusive' : 'shared' );
    $ipNode->addChild( $ipTypeNode );

    my $ipAddressNode = XmlNode->new( 'ip-address' );
    $ipAddressNode->setText( $cpIpNode->getText() );
    $ipNode->addChild( $ipAddressNode );
  }
  return $ipNode;
}

sub getDomainPreferences {
  my $cpDomainNode = shift;

  unless ( testXmlNodeParam($cpDomainNode, 'domain', 'getDomainPreferences') ) {return undef;}

  my $domainPreferencesNode = XmlNode->new( 'preferences' );

  # Since PPP9.x cPanel account's addondomains are transformed into Plesk domain aliases instead of domain with shosting
  my @cpAddondomainNodes = $cpDomainNode->getChildren( 'addondomain' );
  if (@cpAddondomainNodes) {
    foreach my $cpAddondomainNode (sort @cpAddondomainNodes) {
        my $domainAliasNode = XmlNode->new( 'domain-alias' );
        $domainAliasNode->setAttribute( 'name', $cpAddondomainNode->getText() );
        $domainAliasNode->setAttribute( 'mail', 'true' );
        $domainAliasNode->setAttribute( 'web', 'true' );
        $domainAliasNode->addChild( &makeStatusEnabled() );
        $domainPreferencesNode->addChild( $domainAliasNode );
    }
  }
  return $domainPreferencesNode;
}

sub getDomainProperties {
  my $cpAccountNode = shift;

  unless ( testXmlNodeParam($cpAccountNode, 'account', 'getDomainProperties') ) {return undef;}

  my $domainPropertiesNode = XmlNode->new( 'properties' );

  $domainPropertiesNode->addChild( &getIp($cpAccountNode) );
  $domainPropertiesNode->addChild( &makeStatusEnabled() );

  return $domainPropertiesNode;
}

sub addDomainLimitsAndPermissions {
  my ($domainNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($domainNode, 'domain', 'addDomainLimitsAndPermissions') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode, 'account', 'addDomainLimitsAndPermissions') ) {return undef;}

  my @limitNodes;
  my @cpLimitNodes = $cpAccountNode->getChildren( 'limit' );

  if ( @cpLimitNodes ) {
    foreach my $cpLimitNode ( @cpLimitNodes ) {
      my $limitName = $cpLimitNode->getAttribute( 'name' );
      if ( exists $limit_map{$limitName} ) {
        my $limitValue = $cpLimitNode->getText();
        $limitValue = -1 if ( int($limitValue) > $max_limit_value );
        my $limitNode = XmlNode->new( 'limit' );
        $limitNode->setAttribute( 'name', $limit_map{$limitName} );
        $limitNode->setText( $limitValue );
        push @limitNodes, $limitNode;
      }
    }

    if ( @limitNodes ) {
      my $limitsAndPermissionsNode = XmlNode->new( 'limits-and-permissions' );
      foreach my $limitNode ( @limitNodes ) {
        $limitsAndPermissionsNode->addChild( $limitNode );
      }
      $domainNode->addChild( $limitsAndPermissionsNode );
    }
  }
}

sub addDomainMailsystem {
  my ($domainNode, $cpDomainNode) = @_;

  unless ( testXmlNodeParam($domainNode, 'domain', 'addDomainMailsystem') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode, 'domain', 'addDomainMailsystem') ) {return undef;}

  # do nothing
}

sub addDomainMaillists {
  my ($domainNode, $cpDomainNode) = @_;

  unless ( testXmlNodeParam($domainNode, 'domain', 'addDomainMaillists') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode, 'domain', 'addDomainMaillists') ) {return undef;}

  # do nothing
}

sub addDomainDatabases {
  my ($domainNode, $cpDomainNode) = @_;

  unless ( testXmlNodeParam($domainNode, 'domain', 'addDomainDatabases') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode, 'domain', 'addDomainDatabases') ) {return undef;}

  my @cpDatabaseNodes = $cpDomainNode->getChildren( 'database' );
  if (@cpDatabaseNodes) {
    my @databaseNodes = [];

    foreach my $cpDatabaseNode (@cpDatabaseNodes) {
      my $databaseNode = XmlNode->new( 'database' );
      $databaseNode->setAttribute( 'name', $cpDatabaseNode->getAttribute( 'name' ) );
      $databaseNode->setAttribute( 'type', $cpDatabaseNode->getAttribute( 'type' ) );
      # Plesk dump format requires guid, could be empty however, will be fixed with guid fixer before restore
      $databaseNode->setAttribute( 'guid', '' );

      my $version = $cpDatabaseNode->getAttribute( 'version' );
      $databaseNode->setAttribute( 'version', $version ) if defined $version;

      my @cpDbuserNodes = $cpDatabaseNode->getChildren( 'dbuser' );
      if (@cpDbuserNodes) {
        my @dbuserNodes = [];

        foreach my $cpDbuserNode (@cpDbuserNodes) {
          my $dbuserNode = XmlNode->new( 'dbuser' );
          $dbuserNode->setAttribute( 'name', $cpDbuserNode->getAttribute( 'name' ) );
          my $cpDbuserPassword = $cpDbuserNode->getChild( 'password' );
          my $dbuserPassword = XmlNode->new( 'password' );
          # cPanel password type is 'encrypted' or 'empty'. Both are allowed to be Plesk' password type. No token conversion required.
          $dbuserPassword->setAttribute( 'type', $cpDbuserPassword->getAttribute( 'type') );
          $dbuserPassword->setText( $cpDbuserPassword->getText() );
          $dbuserNode->addChild( $dbuserPassword );

          my @cpAccesshostNodes = $cpDbuserNode->getChildren( 'accesshost' );
          if (@cpAccesshostNodes) {
            foreach my $cpAccesshostNode (@cpAccesshostNodes) {
              my $accesshostNode = XmlNode->new( 'accesshost' );
              $accesshostNode->setText( $cpAccesshostNode->getText() );
              $dbuserNode->addChild( $accesshostNode );
            }
          }

          $databaseNode->addChild( $dbuserNode );
        }
      }

      push @databaseNodes, $databaseNode;
    }
    if (@databaseNodes) {
      my $databasesNode = XmlNode->new( 'databases' );

      my %databasesMetadata;
      $databasesMetadata{'domain'} = $cpDomainNode->getAttribute( 'name' );

      $databasesNode->setMetadata(\%databasesMetadata);

      foreach my $databaseNode (@databaseNodes) {
        $databasesNode->addChild( $databaseNode );
      }
      $domainNode->addChild( $databasesNode );
    }
  }
}

sub addDomainuser {
  my ($domainNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($domainNode, 'domain', 'addDomainuser') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode, 'account', 'addDomainuser') ) {return undef;}

  # cPanel account (not reseller) that belongs to reseller account should be mapped into Plesk doamin user.
  # So far until only cPanel domains are migrated, no domain users mapping will be performed.
}

sub getPhosting {
  my ($domainNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($domainNode, 'domain', 'getPhosting') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode, 'account', 'getPhosting') ) {return undef;}

  my $quota;
  my @cpLimitNodes = $cpAccountNode->getChildren( 'limit' );
  if (@cpLimitNodes) {
    foreach my $cpLimitNode (@cpLimitNodes) {
        if ( $cpLimitNode->getAttribute( 'name' ) eq 'disk_space' ) {
          $quota = $cpLimitNode->getText();
          last;
        }
    }
  }

  my $cpDomainNode = $cpAccountNode->getChild( 'domain' );

  my $phostingNode = &makePhostingAttrs();

  my %phostingMetadata;
  $phostingMetadata{'domain'} = $cpDomainNode->getAttribute( 'name' );
  $phostingMetadata{'account'} = $cpAccountNode->getAttribute( 'name' );

  $phostingNode->setMetadata(\%phostingMetadata);

  my $phostingPreferencesNode = getPhostingPreferences( $phostingNode, $cpDomainNode);
  $phostingNode->addChild( $phostingPreferencesNode );

  &addPhostingLimitsAndPermissions( $phostingNode);

  &addPhostingWebusersAndFtpusers( $phostingNode, $cpDomainNode);

  &addPhostingSubdomains( $phostingNode, $cpDomainNode);

  return $phostingNode;
}

sub getPhostingPreferences {
  my ($phostingNode, $cpDomainNode) = @_;

  unless ( testXmlNodeParam($phostingNode, 'phosting', 'getPhostingPreferences') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode, 'domain', 'getPhostingPreferences') ) {return undef;}

  my $preferencesNode = XmlNode->new( 'preferences' );

  my $sysuserNode;

  my @cpFtpuserNodes = $cpDomainNode->getChildren( 'ftpuser' );
  if (@cpFtpuserNodes) {
    foreach my $cpFtpuserNode (@cpFtpuserNodes) {
      unless ( defined $cpFtpuserNode->getAttribute( 'directory' ) ) {
        $sysuserNode = XmlNode->new( 'sysuser' );
        $sysuserNode->setAttribute( 'name', $cpFtpuserNode->getAttribute( 'name' ) );
        $sysuserNode->setAttribute( 'quota', $cpFtpuserNode->getAttribute( 'quota' ) ) if ( defined ( $cpFtpuserNode->getAttribute( 'quota' ) ) );
        my $passwordNode = XmlNode->new( 'password' );
        $passwordNode->setAttribute( 'type', 'encrypted' );
        $passwordNode->setText( $cpFtpuserNode->getAttribute( 'password' ) );
        $sysuserNode->addChild( $passwordNode );
        last;
      }
    }
  }
  unless ( defined ( $sysuserNode ) ) {
    Logging::error("Unable get ftp user account for domain '" . $cpDomainNode->getAttribute( 'name' ) . "'");
    $sysuserNode = XmlNode->new( 'sysuser' );
    $sysuserNode->setAttribute( 'name', '' );
  }
  $preferencesNode->addChild( $sysuserNode );

  my $cpAnonftpNode = $cpDomainNode->getChild( 'anonftp' );
  if ( defined ( $cpAnonftpNode ) ) {
    my $anonftpNode = XmlNode->new( 'anonftp' );
    if ( defined ($cpAnonftpNode->getAttribute( 'pub' )) and $cpAnonftpNode->getAttribute( 'pub' ) eq 'true' ) {
      $anonftpNode->setAttribute( 'pub', 'true' );
      my $permissionNode = XmlNode->new( 'anonftp-permission' );
      $permissionNode->setAttribute( 'name', 'incoming-download' );
      $anonftpNode->addChild( $permissionNode );
    }
    if ( defined ($cpAnonftpNode->getAttribute( 'incoming' )) and $cpAnonftpNode->getAttribute( 'incoming' ) eq 'true' ) {
      $anonftpNode->setAttribute( 'incoming', 'true' );
      my $permissionNode = XmlNode->new( 'anonftp-permission' );
      $permissionNode->setAttribute( 'name', 'incoming-mkdir' );
      $anonftpNode->addChild( $permissionNode );
    }
    $preferencesNode->addChild( $anonftpNode );
  }

  addPhostingPreferencesPdirs( $preferencesNode, $cpDomainNode);

  return $preferencesNode;
}

sub addPhostingLimitsAndPermissions {
  my ($phostingNode) = @_;

  unless ( testXmlNodeParam($phostingNode, 'phosting', 'addPhostingLimitsAndPermissions') ) {return undef;}

  my $limitsAndPermissionsNode = XmlNode->new( 'limits-and-permissions' );
  my $scriptingNode = &makeScriptingAll();

  $limitsAndPermissionsNode->addChild( $scriptingNode );
  $phostingNode->addChild( $limitsAndPermissionsNode );
}

sub addPhostingWebusersAndFtpusers {
  my ($phostingNode, $cpDomainNode) = @_;

  unless ( testXmlNodeParam($phostingNode, 'phosting', 'addPhostingWebusersAndFtpusers') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode, 'domain', 'addPhostingWebusersAndFtpusers') ) {return undef;}

  my @webuserNodes;

  my @cpFtpuserNodes = $cpDomainNode->getChildren( 'ftpuser' );
  if (@cpFtpuserNodes) {
    foreach my $cpFtpuserNode (@cpFtpuserNodes) {
      if ( defined $cpFtpuserNode->getAttribute( 'directory' ) ) {
        my $webuserNode = XmlNode->new( 'webuser' );
        $webuserNode->setAttribute( 'name', $cpFtpuserNode->getAttribute( 'name' ) );

        my %webuserMetadata;
        $webuserMetadata{'name'} = $cpFtpuserNode->getAttribute( 'name' );
        $webuserNode->setMetadata(\%webuserMetadata);

        my $sysuserNode = XmlNode->new( 'sysuser' );
        $sysuserNode->setAttribute( 'name', $cpFtpuserNode->getAttribute( 'name' ) );
        $sysuserNode->setAttribute( 'quota', $cpFtpuserNode->getAttribute( 'quota' ) ) if ( defined ( $cpFtpuserNode->getAttribute( 'quota' ) ) );
        my $passwordNode = XmlNode->new( 'password' );
        $passwordNode->setAttribute( 'type', 'encrypted' );
        $passwordNode->setText( $cpFtpuserNode->getAttribute( 'password' ) );
        $sysuserNode->addChild( $passwordNode );

        $webuserNode->addChild( $sysuserNode );

        $webuserNode->addChild( &makeScriptingAll() );
        push @webuserNodes, $webuserNode;
      }
    }
    if ( @webuserNodes ) {
      my $webusersNode = XmlNode->new( 'webusers' );
      foreach my $webuserNode ( @webuserNodes ) {
        $webusersNode->addChild( $webuserNode );
      }
      $phostingNode->addChild( $webusersNode );
    }
  }

}

sub addPhostingSubdomains {
  my ($phostingNode, $cpDomainNode) = @_;

  unless ( testXmlNodeParam($phostingNode, 'phosting', 'addPhostingSubdomains') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode, 'domain', 'addPhostingSubdomains') ) {return undef;}

  my @cpSubdomainNodes = $cpDomainNode->getChildren( 'subdomain' );
  if (@cpSubdomainNodes) {
    my @subdomainNodes;

    foreach my $cpSubdomainNode ( @cpSubdomainNodes ) {
      my @cpAddonDomainNodes = $cpSubdomainNode->getChildren( 'addondomain' );
      my $cpFtpuserNode = $cpSubdomainNode->getChild( 'ftpuser' );
      if ((@cpAddonDomainNodes and !defined $cpFtpuserNode) or !@cpAddonDomainNodes) {
        # Convert cPanel subdomains without addon domains to Plesk subdomains
        my $subdomainNode = XmlNode->new( 'subdomain' );
        $subdomainNode->setAttribute( 'name', $cpSubdomainNode->getAttribute( 'name' ) );
        $subdomainNode->setAttribute( 'guid', '' );
        $subdomainNode->setAttribute( 'shared-content', 'true' );
        $subdomainNode->setAttribute( 'https', 'true');

        if ( defined ( $cpFtpuserNode ) ) {
          my $sysuserNode = XmlNode->new( 'sysuser' );
          $sysuserNode->setAttribute( 'name', $cpFtpuserNode->getAttribute( 'name' ) );
          $sysuserNode->setAttribute( 'quota', $cpFtpuserNode->getAttribute( 'quota' ) ) if ( defined ( $cpFtpuserNode->getAttribute( 'quota' ) ) );
          my $passwordNode = XmlNode->new( 'password' );
          $passwordNode->setAttribute( 'type', 'encrypted' );
          $passwordNode->setText( $cpFtpuserNode->getAttribute( 'password' ) );
          $sysuserNode->addChild( $passwordNode );
          $subdomainNode->addChild( $sysuserNode );
        }

        $subdomainNode->addChild( &makeScriptingAll() );
        push @subdomainNodes, $subdomainNode;
      }
    }

    if ( @subdomainNodes ) {
      my $subdomainsNode = XmlNode->new( 'subdomains' );
      foreach my $subdomainNode ( @subdomainNodes ) {
        $subdomainsNode->addChild( $subdomainNode );
      }
      $phostingNode->addChild( $subdomainsNode );
    }

  }
}

sub addPhostingPreferencesPdirs {
  my ($preferencesNode, $cpDomainNode) = @_;

  unless ( testXmlNodeParam($preferencesNode, 'preferences', 'addPhostingPreferencesPdirs') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode, 'domain', 'addPhostingPreferencesPdirs') ) {return undef;}

  my @cpPdirNodes = $cpDomainNode->getChildren( 'pdir' );
  if (@cpPdirNodes) {
    foreach my $cpPdirNode (@cpPdirNodes) {
      my $pdirNode = XmlNode->new( 'pdir' );
      $pdirNode->setAttribute( 'name', $cpPdirNode->getAttribute( 'name' ) );
      my $title = $cpPdirNode->getAttribute( 'title' );
      $pdirNode->setAttribute( 'title', $title ) if ( defined $title );
      $pdirNode->setAttribute( 'nonssl', 'true' );
      
      my @cpPduserNodes = $cpPdirNode->getChildren( 'pduser' );
      if (@cpPduserNodes) {
        foreach my $cpPduserNode (@cpPduserNodes) {
          my $pduserNode = XmlNode->new( 'pduser' );
          $pduserNode->setAttribute( 'name', $cpPduserNode->getAttribute( 'name' ) );

          my $password = $cpPduserNode->getAttribute( 'password' );
          if ( defined ($password) ) {
            my $passwordNode = XmlNode->new( 'password' );
            $passwordNode->setAttribute( 'type', 'encrypted' );

            my $encoding = $cpPduserNode->getAttribute( 'encoding' );
            $passwordNode->setAttribute( 'encoding', $encoding ) if defined ( $encoding );
            $passwordNode->setText( $password );

            $pduserNode->addChild( $passwordNode );
          }

          $pdirNode->addChild( $pduserNode );
        }
      }
      $preferencesNode->addChild( $pdirNode );
    }
  }
}

sub processSubdomain2Phosting {
  my ($domainFromSubdomainNode, $cpSubdomainNode, $cpDomainNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($domainFromSubdomainNode, 'domain', 'processSubdomain2Phosting') ) {return undef;}
  unless ( testXmlNodeParam($cpSubdomainNode, 'subdomain', 'processSubdomain2Phosting') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode, 'domain', 'processSubdomain2Phosting') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode, 'account', 'processSubdomain2Phosting') ) {return undef;}

  my $domainName = $domainFromSubdomainNode->getAttribute( 'name' );

  # Get domain preferences (aliases only)
  my $domainPreferencesNode = XmlNode->new( 'preferences' );

  my @cpSubdomainAddondomainNodes = $cpSubdomainNode->getChildren( 'addondomain' );
  # At least one addondomain should always exist, because addondomain is required for subdomain-to-domain conversion

  foreach my $cpSubdomainAddondomainNode ( @cpSubdomainAddondomainNodes ) {
    next if ( $cpSubdomainAddondomainNode->getText() eq $domainName );

    my $domainAliasNode = XmlNode->new( 'domain-alias' );
    $domainAliasNode->setAttribute( 'name', $cpSubdomainAddondomainNode->getText() );
    $domainAliasNode->setAttribute( 'mail', 'true');
    $domainAliasNode->setAttribute( 'web',  'true');
    $domainPreferencesNode->addChild( $domainAliasNode );
  }

  $domainFromSubdomainNode->addChild( $domainPreferencesNode );

  # Get domain properties (ip and status only)
  my $domainPropertiesNode = &getDomainProperties($cpAccountNode);
  $domainFromSubdomainNode->addChild( $domainPropertiesNode );

  # Add the same limits as for main account's domain
  addDomainLimitsAndPermissions( $domainFromSubdomainNode, $cpAccountNode );

  my $phostingNode = &makePhostingAttrs();

  my %phostingMetadata;
  $phostingMetadata{'domain'} = $cpDomainNode->getAttribute( 'name' );
  $phostingMetadata{'account'} = $cpAccountNode->getAttribute( 'name' );
  $phostingMetadata{'from_subdomain'} = $cpSubdomainNode->getAttribute( 'name' );
  $phostingMetadata{'to_domain'} = $domainName;
  $phostingNode->setMetadata(\%phostingMetadata);

  my $cpFtpuserNode = $cpSubdomainNode->getChild( 'ftpuser' );
  # $cpFtpuserNode should always exist, because ftpuser is required for subdomain-to-domain conversion

  my $sysuserNode = XmlNode->new( 'sysuser' );
  $sysuserNode->setAttribute( 'name', $cpFtpuserNode->getAttribute( 'name' ) );
  $sysuserNode->setAttribute( 'quota', $cpFtpuserNode->getAttribute( 'quota' ) ) if ( defined ( $cpFtpuserNode->getAttribute( 'quota' ) ) );
  my $passwordNode = XmlNode->new( 'password' );
  $passwordNode->setAttribute( 'type', 'encrypted' );
  $passwordNode->setText( $cpFtpuserNode->getAttribute( 'password' ) );
  $sysuserNode->addChild( $passwordNode );
  my $preferencesNode = XmlNode->new( 'preferences' );
  $preferencesNode->addChild( $sysuserNode );

  $phostingNode->addChild( $preferencesNode );

  &addPhostingLimitsAndPermissions($phostingNode);

  $domainFromSubdomainNode->addChild( $phostingNode );
}

sub processSubdomain2Shosting {
  my ($domainFromSubdomainNode, $cpSubdomainNode, $cpDomainNode, $cpAccountNode) = @_;

  unless ( testXmlNodeParam($domainFromSubdomainNode, 'domain', 'processSubdomain2Shosting') ) {return undef;}
  unless ( testXmlNodeParam($cpSubdomainNode, 'subdomain', 'processSubdomain2Shosting') ) {return undef;}
  unless ( testXmlNodeParam($cpDomainNode, 'domain', 'processSubdomain2Shosting') ) {return undef;}
  unless ( testXmlNodeParam($cpAccountNode, 'account', 'processSubdomain2Shosting') ) {return undef;}

  my $domainName = $cpDomainNode->getAttribute( 'name' );

  my $domainPreferencesNode = XmlNode->new( 'preferences' );
  $domainFromSubdomainNode->addChild( $domainPreferencesNode );

  # Get domain properties (ip and status only)
  my $domainPropertiesNode = &getDomainProperties($cpAccountNode);
  $domainFromSubdomainNode->addChild( $domainPropertiesNode );

  $domainFromSubdomainNode->addChild( &makeZeroDomainLimitsAndPermissions() );

  my $shostingNode = XmlNode->new( 'shosting' );
  my $subdomainName = $cpSubdomainNode->getAttribute( 'name' );
  $shostingNode->setText( $subdomainName.".".$domainName);

  $domainFromSubdomainNode->addChild( $shostingNode );
}
#
# End of node transformation subs
#


#
# Begin static content getters
#
sub makeStatusEnabled {
  my $statusEnabledNode = XmlNode->new( 'status' );
  $statusEnabledNode->addChild( XmlNode->new( 'enabled' ) );
	return $statusEnabledNode;
}

sub makeScriptingAll {
  my $scriptingNode = XmlNode->new( 'scripting' );
  $scriptingNode->setAttribute( 'ssi',    'true' );
  $scriptingNode->setAttribute( 'php',    'true' );
  $scriptingNode->setAttribute( 'cgi',    'true' );
  $scriptingNode->setAttribute( 'perl',   'true' );
  $scriptingNode->setAttribute( 'python', 'true' );
  return $scriptingNode;
}

sub makePhostingAttrs {
  my $phostingNode = XmlNode->new( 'phosting' );
  $phostingNode->setAttribute( 'wu_script',      'true' );
  $phostingNode->setAttribute( 'guid',           '' );
  $phostingNode->setAttribute( 'https',          'true' );
  $phostingNode->setAttribute( 'fp',             'true' );
  $phostingNode->setAttribute( 'fpssl',          'true' );
  $phostingNode->setAttribute( 'fpauth',         'true' );
  $phostingNode->setAttribute( 'webstat',        'true' );
  $phostingNode->setAttribute( 'errdocs',        'true' );
  $phostingNode->setAttribute( 'shared-content', 'true' );
  return $phostingNode;
}

sub makeZeroDomainLimitsAndPermissions {
  my $limitsAndPermissionsNode = XmlNode->new( 'limits-and-permissions' );
  $limitsAndPermissionsNode->addChild( 'limit', 'content' => '0',     'attributes' => { 'name' => 'max_traffic' } );
  $limitsAndPermissionsNode->addChild( 'limit', 'content' => '0',     'attributes' => { 'name' => 'max_maillists' } );
  $limitsAndPermissionsNode->addChild( 'limit', 'content' => '0',     'attributes' => { 'name' => 'max_box' } );
  $limitsAndPermissionsNode->addChild( 'limit', 'content' => '0',     'attributes' => { 'name' => 'max_db' } );
  $limitsAndPermissionsNode->addChild( 'limit', 'content' => '0',     'attributes' => { 'name' => 'max_subdom' } );
  $limitsAndPermissionsNode->addChild( 'limit', 'content' => '51200', 'attributes' => { 'name' => 'disk_space' } );
  $limitsAndPermissionsNode->addChild( 'limit', 'content' => '0',     'attributes' => { 'name' => 'max_wu' } );
  return $limitsAndPermissionsNode;
}
#
# End of static content getters
#



#
# Begin miscellanneous subs
#
sub testXmlNodeParam {
  my ($param, $test4name, $message) = @_;
  unless ( ref($param) =~ /XmlNode/ ) {
    Logging::error("XmlNode instance is required : " . $message);
    return undef;
  }

  unless ( $param->getName() eq $test4name ) {
    Logging::error( "'" . $test4name . "' XmlNode is required, got '" . $param->getName() . "' : " . $message);
    return undef;
  }
  return 1;
}
#
# End of miscellanneous subs
#


1;
