<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
  <xsl:output indent="yes" encoding="UTF-8"/>
  <xsl:variable name="agent_name" select='RaQ3dump/@agent-name'/>

  <xsl:template match="RaQ3dump">
    <migration-dump agent-name="{$agent_name}">
	  <xsl:apply-templates select="client"/>
	</migration-dump>
  </xsl:template>

	<xsl:template match="@ip">
		<ip>
			<xsl:choose>
				<xsl:when test="../anonftp">
					<ip-type>exclusive</ip-type>
				</xsl:when>
				<xsl:otherwise>
					<ip-type>shared</ip-type>
				</xsl:otherwise>
			</xsl:choose>
			<ip-address>
				<xsl:value-of select='.'/>
			</ip-address>
		</ip>
	</xsl:template>
	
	<!--
  Key used for removal of duplicate IP addresses in client's pool:
  [IP address value][name of the client][indication of ip based hosting]
  -->
	<xsl:key name="ip" match="domain/@ip" use="concat(., ../../@name, ../anonftp)"/>
	
  <xsl:template match="client">
	<client name="{@name}">
	  <xsl:copy-of select="@contact"/>
	  <xsl:copy-of select="password"/>

	  <xsl:choose>
		<xsl:when test="@enabled and (@enabled != 'false')">
		  <status><enabled/></status>
		</xsl:when>
		<xsl:otherwise>
		  <status><disabled-by name="admin"/></status>
		</xsl:otherwise>
	  </xsl:choose>

	  <xsl:if test="@email">
		<pinfo name="email"><xsl:value-of select="@email"/></pinfo>
	  </xsl:if>

      <ip_pool>
      	<xsl:apply-templates select="domain/@ip[generate-id(.) = generate-id(key('ip',concat(., ../../@name, ../anonftp)))]"/>
      </ip_pool>

	  <xsl:apply-templates select="domain"/>
	</client>
  </xsl:template>

  <xsl:template match="domain">
	<domain name="{@name}" www="{@www}">

	  <!-- IP -->
	  <xsl:apply-templates select="@ip"/>

	  <!-- STATUS -->
	  <xsl:choose>
		<xsl:when test="@enabled and (@enabled != 'false')">
		  <status><enabled/></status>
		</xsl:when>
		<xsl:otherwise>
		  <status><disabled-by name="admin"/></status>
		</xsl:otherwise>
	  </xsl:choose>

	  <!-- LIMITS -->
	  <xsl:for-each select="domain-limit">
		<xsl:if test="@name='quota'">
		  <xsl:call-template name="limit">
			<xsl:with-param name="limit" select="'disk_space'"/>
			<xsl:with-param name="value" select="text()"/>
		  </xsl:call-template>
		</xsl:if>
		<xsl:if test="@name='users'">
		  <xsl:call-template name="limit">
			<xsl:with-param name="limit" select="'max_wu'"/>
			<xsl:with-param name="value" select="text()"/>
		  </xsl:call-template>
		</xsl:if>
		<xsl:if test="@name='traffic'">
		  <xsl:call-template name="limit">
			<xsl:with-param name="limit" select="'max_traffic'"/>
			<xsl:with-param name="value" select="text()"/>
		  </xsl:call-template>
		</xsl:if>
	  </xsl:for-each>

	  <!-- PERMISSION -->

	  <!-- MAILSYSTEM -->
	  <xsl:call-template name="create-mailsystem"/>

	  <!-- MAILLIST -->
	  <xsl:if test="maillist">
		<maillists>
		  <status><enabled/></status>
		  <xsl:apply-templates select="maillist"/>
		</maillists>
	  </xsl:if>

	  <!-- CERTIFICATES -->
	  <xsl:copy-of select="certificate"/>

	  <!-- 'no-content' does not provide admin/sysuser, so there's no sense to create phosting -->
	  <xsl:if test="admin/sysuser">

		<!-- PHOSTING -->
		<phosting fpauth="false">
		  <xsl:if test="@cid">
			<xsl:attribute name="cid_docroot"><xsl:value-of select="@cid"/></xsl:attribute>
		  </xsl:if>

		  <xsl:if test="admin/@cid_private">
			<xsl:attribute name="cid_admin_private"><xsl:value-of select="admin/@cid_private"/></xsl:attribute>
		  </xsl:if>

		  <xsl:if test="domain-permission[@name='ssl']">
			<xsl:attribute name="https">true</xsl:attribute>
			<xsl:if test="domain-permission[@name='frontpage']">
			  <xsl:attribute name="fpssl">true</xsl:attribute>
			</xsl:if>
		  </xsl:if>

		  <xsl:call-template name="convert-boolean-permission">
			<xsl:with-param name="raqname" select="'frontpage'"/>
			<xsl:with-param name="pleskname" select="'fp'"/>
		  </xsl:call-template>

		  <!-- SYSUSER -->
		  <xsl:copy-of select="admin/sysuser"/>

		  <!-- SCRIPTING -->
		  <xsl:call-template name="create-scripting"/>

		  <!-- FPUSER -->
		  <xsl:if test="domain-permission[@name='frontpage']">
			<fpuser>
			  <xsl:attribute name="name">
				<xsl:call-template name="mangle-sysuser-name">
				  <xsl:with-param name="name" select="admin/sysuser/@name"/>
				</xsl:call-template>
			  </xsl:attribute>
			  <xsl:copy-of select="admin/sysuser/password"/>
			</fpuser>
		  </xsl:if>

		  <!-- LOGROTATION -->
		  <!-- ANONFTP -->
		  <anonftp>
			<xsl:attribute name="pub">
			  <xsl:if test="anonftp/@cid">true</xsl:if>
			  <xsl:if test="not(anonftp/@cid)">false</xsl:if>
			</xsl:attribute>
			<xsl:attribute name="incoming">
			  <xsl:if test="anonftp/@cid_incoming">true</xsl:if>
			  <xsl:if test="not(anonftp/@cid_incoming)">false</xsl:if>
			</xsl:attribute>
			<xsl:copy-of select="anonftp/@cid"/>
			<xsl:copy-of select="anonftp/@cid_incoming"/>

			<xsl:if test="anonftp/@cid_incoming and anonftp/anonftp-limit[@name='incoming-quota']">
			  <anonftp-limit name="incoming-disk-quota">
				<xsl:value-of select="anonftp/anonftp-limit[@name='incoming-quota']"/>
			  </anonftp-limit>
			</xsl:if>
			<xsl:if test="anonftp/limit[@name='users']">
			  <anonftp-limit name="max-connections">
				<xsl:value-of select="anonftp/limit[@name='users']"/>
			  </anonftp-limit>
			</xsl:if>
			<anonftp-permission name="incoming-mkdir"/>
			<anonftp-permission name="incoming-download"/>
		  </anonftp>
		  <!-- PDIR -->
		  <!-- WEBUSER -->
		  <xsl:apply-templates select="user" mode="webuser"/>
		</phosting>
	  </xsl:if>
	</domain>
  </xsl:template>

  <xsl:template match="maillist">
       <maillist name="{@name}">
         <status><enabled/></status>
	  <owner><xsl:value-of select="../admin/mail/@mailname"/>@<xsl:value-of select="../@name"/></owner>
	  <xsl:copy-of select="recipient"/>
	</maillist>
  </xsl:template>

  <xsl:template name="create-mailsystem">
	<mailsystem>
	  <xsl:choose>
		<xsl:when test="domain-permission[@name='email']">
		  <status><enabled/></status>
		</xsl:when>
		<xsl:otherwise>
		  <status><disabled-by name="admin"/></status>
		</xsl:otherwise>
	  </xsl:choose>

	  <xsl:for-each select="admin|user">
		<xsl:call-template name="create-mailuser"/>
	  </xsl:for-each>
	  <xsl:if test="@catch-all">
		<catch-all><xsl:value-of select="@catch-all/text()"/></catch-all>
	  </xsl:if>
	</mailsystem>
  </xsl:template>

  <xsl:template match="user" mode="webuser">
	<webuser>
	  <xsl:copy-of select="@cid"/>
	  <xsl:copy-of select="@cid_private"/>
	  <sysuser name="{sysuser/@name}">
		<xsl:if test="sysuser/@quota">
		  <xsl:attribute name="quota">
			<xsl:value-of select="sysuser/@quota"/>
		  </xsl:attribute>
		</xsl:if>
		<xsl:if test="../domain-permission[@name='shell'] and user-permission[@name='shell']">
		  <xsl:attribute name="shell">/bin/bash</xsl:attribute>
		</xsl:if>

		<xsl:copy-of select="sysuser/password"/>
	  </sysuser>
	</webuser>
  </xsl:template>

  <xsl:template name="create-mailuser">
	<mailuser name="{mail/@mailname}" mailgroup-enabled="false">
	  <xsl:copy-of select="sysuser/password"/>
	  <mailbox type="mbox">
        <xsl:if test="mail/@cid">
          <xsl:copy-of select="mail/@cid"/>
        </xsl:if>
        <xsl:if test="mail/@cid_inbox">
          <xsl:copy-of select="mail/@cid_inbox"/>
        </xsl:if>
	  </mailbox>
	  <xsl:if test="user-limit[@name='quota']">
		<xsl:attribute name="mailbox-quota">
		  <xsl:value-of select="user-limit[@name='quota']"/>
		</xsl:attribute>
	  </xsl:if>

	  <xsl:copy-of select="mail/alias"/>

	  <xsl:if test="mail/forward">
		<redirect><xsl:value-of select="mail/forward/text()"/></redirect>
	  </xsl:if>

	  <xsl:if test="mail/autoresponder">
		<autoresponders enabled="true">
		  <autoresponder name="vacation">
			<xsl:copy-of select="mail/autoresponder/@status"/>
			<text><xsl:value-of select="mail/autoresponder/text()"/></text>
		  </autoresponder>
		</autoresponders>
	  </xsl:if>
	</mailuser>
  </xsl:template>

  <xsl:template name="create-scripting">
    <scripting>
      <xsl:call-template name="convert-boolean-permission">
        <xsl:with-param name="raqname" select="'cgi'"/>
        <xsl:with-param name="pleskname" select="'cgi'"/>
      </xsl:call-template>

      <xsl:call-template name="convert-boolean-permission">
        <xsl:with-param name="raqname" select="'ssi'"/>
        <xsl:with-param name="pleskname" select="'ssi'"/>
      </xsl:call-template>
    </scripting>
  </xsl:template>

  <xsl:template name="convert-boolean-permission">
    <xsl:param name="raqname"/>
    <xsl:param name="pleskname"/>
    <xsl:choose>
      <xsl:when test="domain-permission[@name = $raqname]">
        <xsl:attribute name="{$pleskname}">true</xsl:attribute>
      </xsl:when>
      <xsl:otherwise>
        <xsl:attribute name="{$pleskname}">false</xsl:attribute>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template name="mangle-sysuser-name">
    <xsl:param name="name"/>
    <xsl:choose>
      <xsl:when test="$name = 'admin' or $name = 'ftp' or $name = 'anonymous' or $name = 'root' or $name = 'mailman' or $name = 'tomcat4'"><xsl:value-of select="concat($name, '1')"/></xsl:when>
      <xsl:otherwise><xsl:value-of select="$name"/></xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template name="mangle-sysuser">
    <xsl:param name="sysuser"/>
    <sysuser>
      <xsl:attribute name="name">
        <xsl:call-template name="mangle-sysuser-name">
          <xsl:with-param name="name" select="$sysuser/@name"/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:copy-of select="$sysuser/@quota"/>
      <xsl:copy-of select="$sysuser/@shell"/>
      <xsl:copy-of select="$sysuser/password"/>
    </sysuser>
  </xsl:template>

  <xsl:template name="create-sysuser">
    <xsl:param name="user"/>
    <sysuser>
      <xsl:attribute name="name">
        <xsl:call-template name="mangle-sysuser-name">
          <xsl:with-param name="name" select="$user/sysuser/@name"/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:if test="$user/permission[@name = 'shell']">
        <xsl:attribute name="shell">/bin/bash</xsl:attribute>
      </xsl:if>
      <xsl:copy-of select="$user/sysuser/password"/>
    </sysuser>
  </xsl:template>

  <!-- - - - - - - - - - - - - - - - - - - - - - -
       Outputs Plesk's limit (used for domains
       and clients accounts)
       Parameters:
       limit - limit name
       value - limit value
       - - - - - - - - - - - - - - - - - - - - - - - -->
  <xsl:template name="limit">
    <xsl:param name="limit"/>
    <xsl:param name="value"/>
    <xsl:element name="limit">
      <xsl:attribute name="name">
        <xsl:value-of select="$limit"/>
      </xsl:attribute>
      <xsl:value-of select="format-number($value, '#')"/>
    </xsl:element>
  </xsl:template>

</xsl:stylesheet>
