#!/usr/bin/perl

use vars qw($dumpDir $ptrArgs $pwd $workDir $storage);
use strict;
use Socket;

use AgentConfig;

my $VERSION='1.0.1';

my $g_mailertable = "/etc/mail/mailertable";
my $LOCK_SH=1;
my $LOCK_EX=2;
my $LOCK_UN=8;
my $LOCK_NB=4;
my $SYSMAILRC="/usr/local/custom/etc/mail.rc";

#
# parse command line
#
my %arg_opts = ('--help|-h'=>'',
				'--debug|-d'=>'i?',
				'--get-status|-s'=>'',
				'--dump-accounts|-dc'=>'s',
				'--dump-domains|-dd'=>'s',
				'--dump-all|-da'=>'',
				'--get-content-list|-lc'=>'',
				'--no-content|-nc'=>'',
				'--no-compress|-nz'=>'',
				'--output|-o'=>'s',
				'--status-file|-sf'=>'s',
				'--version|-v'=>'',
			   );

my $returnCode = 0;

#
# global variables
#

my $kilobyte = 1024;
my $megabyte = $kilobyte*$kilobyte;

my $ignoredUsers_ = {'root' => '1',
					 'bin' => '1',
					 'daemon' => '1',
					 'adm' => '1',
					 'lp' => '1',
					 'sync' => '1',
					 'shutdown' => '1',
					 'halt' => '1',
					 'mail' => '1',
					 'uucp' => '1',
					 'operator' => '1',
					 'games' => '1',
					 'gopher' => '1',
					 'ftp' => '1',
					 'news' => '1',
					 'mysql' => '1',
					 'nobody' => '1',
					 'zope' => '1',
					 'majordomo' => '1',
					 'tomcat4' => '1',
					 'apache' => '1',
					 'sshd' => '1',
                                         'users' => '1',
                                         'opt' => '1',
                                         'www' => '1'
					};

my %ignoredUsers = %{$ignoredUsers_};

#
# end global variables
#

require 'agent.include.pl';

$ptrArgs = &getArguments (\@ARGV,\%arg_opts);

my ($outPath,$outFh,$outHas,$dumpFile,$statusFile,$app);

$app = $0;
if ($app=~/(.+)\/([^\/]+)$/) {
  $workDir = $1;
  $app = $2;
}
$app=~s/\.pl$//;

$pwd = AgentConfig::cwd();

if ($workDir eq '.') {
  $workDir = $pwd;
} elsif ($workDir !~ /^\//) {
  $workDir = $pwd.'/'.$workDir;
}

$dumpFile = $workDir.'/dump.xml';
$statusFile = $workDir.'/dumping-status.xml';
$dumpDir = $workDir.'/pma';

my $objDumpStatus = makeDumpStatus($ptrArgs->{'status-file'} || $statusFile);

if (exists $ptrArgs->{'get-content-list'}) {
  makeContentList();
  exit 0;
}

if (exists $ptrArgs->{'version'}) {
  print $outFh $VERSION,"\n";
  exit 0;
}

my $rootName = 'migration-dump';
my $dtdName = 'plesk.dtd';

my $wrapDbh = getDbConnect('mysql', 'root', getMySqlPassword(), 'mysql', 'localhost');
unless(ref($wrapDbh)=~/HASH/ && ref($wrapDbh->{'EXECUTE'})=~/CODE/){
  die "Error: can not connect to database\n";
}

if (exists $ptrArgs->{'get-status'}) {
  my $outFh = &openOutput($dumpFile);
  &printAgentStatus($outPath=$ptrArgs->{'output'}||$dumpFile,
					$objDumpStatus->{'FILE'}->(), $outFh);
  &closeOutput();
  exit 0;
}

#
# get MIME Base64 encoder
#
my $wrapBase64 = makeMIMEBase64();


if (exists ($ptrArgs->{'dump-all'}) ||
	exists ($ptrArgs->{'dump-accounts'}) ||
	exists ($ptrArgs->{'dump-domains'})) {

  my ($root,@accounts,@domains,$ptrAccounts,$ptrDomains, $value);

  initStorage();
  getDumpDir($dumpDir);

  if (exists $ptrArgs->{'no-compress'}) {
	setCompress(0);
  }

  &printToLog("Work dir: $workDir");
  &printToLog("Dump file: $dumpFile");
  &printToLog("Status file: ".$objDumpStatus->{'FILE'}->());


  if ($value = $ptrArgs->{'dump-accounts'}) {
    if ($value eq "-") {
      $value = <STDIN>;
      chomp $value;
    }
    @accounts = split(/\s*,\s*/,$value);
    $ptrAccounts = \@accounts;
  }

  if ($value = $ptrArgs->{'dump-domains'}) {
    if ($value eq "-") {
      $value = <STDIN>;
      chomp $value;
    }
    @domains = split(/\s*,\s*/,$value);
    $ptrDomains = \@domains;
  }
  #
  # generate a xml dump
  #

  $root = &getAlabanzaXDump($dumpDir, $ptrAccounts, $ptrDomains, $rootName);

  $storage->finish($root);
} else {
  &printHelp();
}

exit $returnCode;

#
# end main
#
#==============================================================
#
# subroutines
#

sub printAgentStatus {
  my $dumpFile = shift;
  my $statusFile = shift;
  my $outFh = shift || *STDOUT;
  my $rootName = shift || 'agentstatus';

  my ($root,$util,$out,$item,$engineRoot,$rpm,$sql,$ptrRow);

  $root = &makeXmlNode($rootName);

  &printXml($root, $outFh, $rootName . '.dtd');
  return;

  $item = &makeXmlNode('dumpfile', $dumpFile);
  $root->{'ADDCHILD'}->($item);
  $item = &makeXmlNode('statusfile', $statusFile);
  $root->{'ADDCHILD'}->($item);

  &printXml($root, $outFh, $rootName . '.dtd');
}

sub getAlabanzaXDump {
  my ($dumpDir, $ptrAccounts, $ptrDomains, $rootName) = @_;
  my (%ipMap,$item);
  $rootName = 'PleskXDump' unless $rootName;

  my $root = &makeXmlNode($rootName);
  $root->{'ATTRIBUTE'}->('agent-name','AlabanzaX');

  if (ref($ptrDomains) =~ /ARRAY/ && (@{$ptrDomains} > 0)) {

    createSelectedDomains($root,$ptrDomains);

  } elsif (ref($ptrAccounts) =~ /ARRAY/ && (@{$ptrAccounts} > 0)) {

    createSelectedAccounts($root,$ptrAccounts);

  } else {
    createAllAccounts($root);
  }

  return $root;
}

sub createAllAccounts {
  my ($parent) =@_;

  my $usersFile = "/etc/passwd";
  my @users;

  open (USERSFILE, "< $usersFile");

  while(<USERSFILE>) {
    my ($user,$dummy) = split(/:/,$_);
    next if defined($ignoredUsers{$user});
    push @users,$user;
  }

  close(USERFILE);

  createSelectedAccounts($parent,\@users);
}

sub createSelectedAccounts {

  my ($parent,$accounts) = @_;

  my $account;

  $objDumpStatus->{'COUNTS'}->( scalar(@{$accounts}), 0 );

  foreach $account (@{$accounts}) {
    &createAccountNode($parent, $account);
  }

}

sub createSelectedDomains {
  my ($parent,$ptrDomains)=@_;
  my $domain;

  $objDumpStatus->{'COUNTS'}->( 1, scalar(@{$ptrDomains}) );
  $objDumpStatus->{'ACCOUNT'}->( 'client' );

  my $account = makeXmlNode('client');
  $account->{'ATTRIBUTE'}->('name','client');
  createPasswordNode($account, '', 'plain');

  my $statusNode = makeXmlNode('status');
  $statusNode->{'ADDCHILD'}->(makeXmlNode('enabled'));
  $account->{'ADDCHILD'}->($statusNode);

  foreach $domain (@{$ptrDomains}) {
    &createDomain($account, $domain);
  }

  $parent->{'ADDCHILD'}->($account);
}

sub createAccountNode {
  my ($parent, $account) = @_;

  my ($item,$password);
  &printToLog("Creating account $account");
  $objDumpStatus->{'ACCOUNT'}->( $account );

  $item = makeXmlNode('client');

  $item->{'ATTRIBUTE'}->('name',$account);

  my $shadowFile = "/etc/shadow";

  open(SHADOW, "<$shadowFile");

  while (<SHADOW>) {
    my ($u,$passwd,@restp) = split(/:/, $_);
    if ($u eq $account) {
      $password = $passwd;
    }
  }
  close(SHADOW);

  createPasswordNode($item,$password,'encrypted');

  my $statusNode = makeXmlNode('status');
  $statusNode->{'ADDCHILD'}->(makeXmlNode('enabled'));
  $item->{'ADDCHILD'}->($statusNode);

  createAllDomainsForAccount($item, $account);

  $parent->{'ADDCHILD'}->($item);
}

sub createAllDomainsForAccount {
  my ($parent,$account) = @_;
#  my $homePath = get_home($account);
#  my @homeDirContent = `ls $homePath`;
#  foreach my $homeDirElement (@homeDirContent){
#    if ($homeDirElement=~/(\w+)-www/){
#      createDomain($parent, $1);
#    }
#  }
  my ($s,@domains,$domain);
  open(MAILERTABLE,$g_mailertable);
  flock(MAILERTABLE, $LOCK_SH);
  while($s=<MAILERTABLE>){
    chop($s);
    if($s=~/^(\S*)\s*procmail:$account.*$/){
      push @domains, $1;
    }
  }

  my $ipPoolNode = makeXmlNode('ip_pool');
  foreach $domain (@domains){
    makeDomainIp($ipPoolNode,$domain);
  }
  $parent->{'ADDCHILD'}->($ipPoolNode);

  foreach $domain (@domains){
    createDomain($parent, $domain);
  }
  flock(MAILERTABLE, $LOCK_UN);
  close(MAILERTABLE);
}

sub createDomain {
  my ($parent,$domain_name) = @_;
  my $item = &makeXmlNode('domain');
  $item->{'ATTRIBUTE'}->('name',$domain_name);

  $objDumpStatus->{'DOMAIN'}->($domain_name);
  #Create domain ip

  &makeDomainIp($item,$domain_name);

  my $statusNode = makeXmlNode('status');
  $statusNode->{'ADDCHILD'}->(makeXmlNode('enabled'));
  $item->{'ADDCHILD'}->($statusNode);

  if (exists($ptrArgs->{'no-content'})) {
    $parent->{'ADDCHILD'}->($item);
    return;
  }

  # Limits
  my $cmd = "grep \"^$domain_name:\" $SYSMAILRC";
  my $s = `$cmd`;
  chop ($s);
  if ($s) {
    my($dummy,$emailLimit,$listLimit)=split(/:/,$s);
    createLimitNode($item,'max_box',$emailLimit);
    createLimitNode($item,'max_maillists',$listLimit);
  }
  # Permissions

  # Mailsystem

  createMailNode($item,$domain_name);

  # DataBases

  createAllDatabases($item,$domain_name);

  # Mailing lists

  createMaillistNode($item,$domain_name);

  # Phys Hosting

  createPhysHostingNode($item,$domain_name);

  $parent->{'ADDCHILD'}->($item);
}

sub createMaillistNode {
  my ($parent,$domain)=@_;
}

sub createPhysHostingNode {
  my ($parent,$domain) = @_;

  my ($dom,$dummy) = split ('\.',$domain);
  my $owner = get_owner_of_domain($domain);
  my $home = get_home($owner);

  my $content = "$home/$dom-www";
  my $contentCgi = "$home/$dom-www/cgi-bin";

  my $userType;

  my $item = makeXmlNode ('phosting');

 # SysUser

  $userType = 'sys';
  createUserNode($item,$owner,$userType);

  #Scripting

  createScriptingNode($item,$domain);

 # FrontPage

  my $fpFile = "/usr/local/frontpage/$domain:80.cnf";

  if (-e $fpFile) {
    $item->{'ATTRIBUTE'}->('fp','true');
    $userType = 'fp';
    createUserNode($item,$owner,$userType);
  }

  #Anonymous FTP

  my $anonFtpDir = "$home/$dom-anonftp";
  if (-d $anonFtpDir) {

    createAnonFtpNode($item,$domain,$anonFtpDir);

  }

  #Content

  if (-d $content) {
    my $docroot = makeDumpFile("$dumpDir/$domain\_docroot",$content,'',$contentCgi);
    $item->{'ATTRIBUTE'}->('cid_docroot',$docroot);
  }

  if (-d $contentCgi) {
    my $docrootCgi = makeDumpFile("$dumpDir/$domain\_cgi",$contentCgi);
    $item->{'ATTRIBUTE'}->('cid_cgi',$docrootCgi);
  }

  $parent->{'ADDCHILD'}->($item);
}

sub createScriptingNode {
  my ($parent,$domain)=@_;
  my $item = makeXmlNode('scripting');
  $parent->{'ADDCHILD'}->($item);

}

sub createAnonFtpNode {
  my ($parent,$domain,$anonFtpDir)=@_;
  my $item = makeXmlNode('anonftp');

  if (-d "$anonFtpDir/pub") {
    $item->{'ATTRIBUTE'}->('pub','true');
    my $docrootPub = makeDumpFile("$dumpDir/anonftp-$domain\_pub","$anonFtpDir/pub");
    $item->{'ATTRIBUTE'}->('cid',$docrootPub);

  } elsif (-d "$anonFtpDir/incoming") {
    $item->{'ATTRIBUTE'}->('incoming','true');

    my $docrootInc = makeDumpFile("$dumpDir/anonftp-$domain\_incoming","$anonFtpDir/incoming");

    $item->{'ATTRIBUTE'}->('cid_incoming',$docrootInc);
  }

  if (-e "$anonFtpDir/welcome.msg") {
    $item->{'ATTRIBUTE'}->('display-login','true');
    open (WELCOME,"$anonFtpDir/welcome.msg");
    my $message = <WELCOME>;
    close(WELCOME);

    $item->{'ADDCHILD'}->(makeXmlNode('login-message',$message));
  }

  # Limits && Permissions.

  $parent->{'ADDCHILD'}->($item);
}

sub createUserNode {
  my ($parent,$user,$type) = @_;

  my ($password,$item,$quota);

  my $shadowFile = "/etc/shadow";
  open(SHADOW, "<$shadowFile");

  while (<SHADOW>) {
    my ($u,$passwd,@restp) = split(/:/, $_);
    if ($u eq $user) {
      $password = $passwd;
    }
  }
  close(SHADOW);

  if ($type eq 'sys'){

    $item = makeXmlNode('sysuser');

    $quota = `/usr/bin/quota -u $user | /usr/bin/tail -1`;
    chop($quota);
    if($quota=~/^ *[^ ]+ +([0-9]+) +([0-9]+)/) {
      $item->{'ATTRIBUTE'}->('quota',$2);
    }

  } elsif ($type eq 'fp') {

    $item = makeXmlNode('fpuser');

  }

  $item->{'ATTRIBUTE'}->('name',$user);
  createPasswordNode($item,$password,'encrypted');

  $parent->{'ADDCHILD'}->($item);
}

sub createMailNode {
  my ($parent,$domain) =@_;
  my (@mailUsers,$mailUser,$catchAll,@temp,$s);
  my $item = &makeXmlNode('mailsystem');

  my $statusNode = makeXmlNode('status');
  $statusNode->{'ADDCHILD'}->(makeXmlNode('enabled'));
  $item->{'ADDCHILD'}->($statusNode);

  my ($dom,$dummy) = split ('\.',$domain);
  my $owner = get_owner_of_domain($domain);
  my $home = get_home($owner);
  my $mailFile = "$home/.mm/$domain.mailaddr";
  my $pwdFile = "$home/.mm/$domain.passwd";
  my $mailDir = "$home/$dom-mail";

  if (-e $mailFile) {
    open(MAILFILE,$mailFile);
    while (<MAILFILE>) {
      chop $_;
      if ($_ = ~/^(.*):(.*)$/) {
        @temp = split(/,/,$2);
	if((scalar(@temp) == 1) && ($1 eq $2)) {
	  createMailUserNode($item,$1,$pwdFile,$mailDir,$domain,0,undef);
        } elsif (scalar(@temp) > 1){
	  my @groupMembers;
          foreach my $temp (@temp) {
	    next if ($temp eq $1);
	    push @groupMembers,$temp;
	  }
	  createMailUserNode($item,$1,$pwdFile,$mailDir,$domain,1,@groupMembers);
	} elsif ($1 eq 'default') {
	  $catchAll = @temp[0];
	}
      }
    }
    close(MAILFILE);
  } else {
    printToLog("File: $mailFile does not exist.\n");
    return;
  }

  if ($catchAll) {
    my $catchNode = makeXmlNode('catch-all',$catchAll.'@'.$domain);
    $item->{'ADDCHILD'}->($catchNode);
  }

  $parent->{'ADDCHILD'}->($item);
}

sub createMailUserNode {
  my ($parent,$mailUser,$pwdFile,$mailDir,$domain,$isMailgroup,@groupMembers)=@_;
  my $item = makeXmlNode('mailuser');
  $item->{'ATTRIBUTE'}->('name',$mailUser);

  my (@strings,$password,$dummy);

  @strings = `cat $pwdFile`;
  @strings = grep /^$mailUser:/, @strings;
  ($dummy,$password) = split (/:/,$strings[0]);

  createPasswordNode($item,$password,'encrypted');

  if (-e $mailDir.'/'.$mailUser) {
    createMailboxNode($item,$mailDir,$mailUser,$domain);
  }

  if ($isMailgroup) {
    $item->{'ATTRIBUTE'}->('mailgroup-enabled','true');
    foreach my $member (@groupMembers) {
      addGroupMember($item,$member,$domain);
    }
  } else {
    $item->{'ATTRIBUTE'}->('mailgroup-enabled','false');
  }

  $parent->{'ADDCHILD'}->($item);
}

sub addGroupMember {
  my ($parent,$member,$domain)=@_;
  my $item;

  if ($member = ~/^([^@]+)@([^@]+)$/) {
    $item = makeXmlNode('mailgroup-member',$member);
  } else {
    $item = makeXmlNode('mailgroup-member',$member.'@'.$domain);
  }

  $parent->{'ADDCHILD'}->($item);
}

sub createMailboxNode {
  my ($parent,$mailDir,$mailUser,$domain) = @_;

  my $item = makeXmlNode('mailbox');
  $item->{'ATTRIBUTE'}->('enabled','true');
  $item->{'ATTRIBUTE'}->('type','mbox');

  my $filename = makeDumpFile ("$dumpDir/$mailUser\@$domain\_mbox",$mailDir,$mailUser);

  $item->{'ATTRIBUTE'}->('cid',$filename) if $filename;

  $parent->{'ADDCHILD'}->($item);
}

sub getipbyname {
  my ($domain)=@_;
  my ($name,$aliases,$addrtype,$length,@addrs)=gethostbyname($domain);
  my ($a0,$a1,$a2,$a3)=unpack('C4',$addrs[0]);
  if($a0 eq "" || $a1 eq "" || $a2 eq "" || $a3 eq "") {
    return "";
  } else {
    return "$a0.$a1.$a2.$a3";
  }
}

sub makeDomainIp {
  my ($root,$domain)=@_;

  my $ip = getipbyname($domain);
  my $item = makeXmlNode('ip');
  $item->{'ADDCHILD'}->(makeXmlNode('ip-type', getIpType($domain)));
  $item->{'ADDCHILD'}->(makeXmlNode('ip-address', $ip));

  $root->{'ADDCHILD'}->($item);
}

sub get_owner_of_domain {
  my($domain)=@_;
  my $s;
  if(!$domain) { die("Parameter missing!"); }
  my($owner);
  open(MAILERTABLE,$g_mailertable) || die("failed to open $g_mailertable");
  flock(MAILERTABLE, $LOCK_SH);
  while($s=<MAILERTABLE>) {
    chop($s);
    if($s=~/^$domain.*procmail:(\w+).*$/){
      $owner=$1;
      last;
    }
  }
  flock(MAILERTABLE, $LOCK_UN);
  close(MAILERTABLE);
  return $owner;
}

sub get_home {
  my($user)=@_;
  if($user eq ""){
    printToLog("Get home for unspecified user failed.");
  } elsif(-d "/home5/$user") {
    return "/home5/$user";
  } elsif(-d "/home4/$user") {
    return "/home4/$user";
  } elsif(-d "/home3/$user") {
    return "/home3/$user";
  } elsif(-d "/home2/$user") {
    return "/home2/$user";
  }
  return "/home/$user";
}

sub createPasswordNode {
  my ($parent, $password, $type) = @_;

  my $item = &makeXmlNode('password', $password);
  $item->{'ATTRIBUTE'}->('type', $type);
  $parent->{'ADDCHILD'}->($item);
}

sub createPinfoNode {
  my ($parent, $username, $fullname, $email) = @_;

  my $item = &makeXmlNode('pinfo');
  $item->{'ATTRIBUTE'}->('username', $username);
  $item->{'ATTRIBUTE'}->('fullname', $fullname) if defined($fullname);
  $item->{'ATTRIBUTE'}->('e-mail', $email) if defined($email);
  $parent->{'ADDCHILD'}->($item);
}

sub createLimitNode {
  my ($parent, $prefix, $name, $value, $unlimited) = @_;

  if ($unlimited) {
    return;
  }

  my $item = &makeXmlNode($prefix . "-limit", $value);
  $item->{'ATTRIBUTE'}->('name', $name);
  $parent->{'ADDCHILD'}->($item);
}

sub createIpNode {
  my ($ptrHash,$ptrAccounts,$ptrDomains) = @_;

  my ($root,$shared,$exclusive,$list,@domains,$ipType,$ipNode,$ip,$id,$item);
  $root = makeXmlNode('map');
  $shared = &makeXmlNode('shared');
  $exclusive = &makeXmlNode('exclusive');
  $root->{'ADDCHILD'}->($shared);
  $root->{'ADDCHILD'}->($exclusive);

  if(ref($ptrAccounts)=~/ARRAY/) {
    $list = join(',',@{$ptrAccounts});
    # need to create @domains - list of domains
  } elsif (ref($ptrDomains)=~/ARRAY/){
    foreach my $domain (@{$ptrDomains}) {
      $ipType = getIpType($domain);
      $ipNode = makeXmlNode('ip');
      $ip = getipbyname($domain);
      $id = 'ip_'.join('',unpack('n2',pack('C4', split(/\./,$ip))));
      $ipNode->{'ATTRIBUTE'}->('id',$id);
      $item = makeXmlNode('old',$ip);
      $ipNode->{'ADDCHILD'}->($item);
      $item = &makeXmlNode('new');
      $ipNode->{'ADDCHILD'}->($item);

      if ($ipType = ~/shared/) {
        $shared->{'ADDCHILD'}->($ipNode);
      } else {
        $exclusive->{'ADDCHILD'}->($ipNode);
      }

    }

    return $root;
  }

}

sub getIpType {
  my ($domain) =@_;
  my $type;
  my ($dom,$dummy) = split ('\.',$domain);
  my $owner = get_owner_of_domain($domain);
  my $home = get_home($owner);

  if (-d "$home/$dom-anonftp") {

    $type = 'exclusive';

  } else {

    $type = 'shared';
  }

  return $type;
}

sub getMySqlPassword {
  my $pwdFile = '/etc/.mysqlpw';
  my $password = `cat $pwdFile`;
  chop $password;
  return $password;
}

sub createAllDatabases {
  my ($parent,$domain)=@_;
  my ($ptrRow,$database,@databases);
  my $sql = "SELECT db from db WHERE Host='$domain' GROUP BY db";

  if ($wrapDbh->{'EXECUTE'}->($sql)) {

    while($ptrRow = $wrapDbh->{'FETCHROW'}->()) {

      push @databases,$ptrRow->[0];

    }

  }
  $wrapDbh->{'FINISH'}->();

  foreach $database (@databases) {
    createDatabaseNode($parent,$database,$domain);
  }

}

sub createDatabaseNode {
  my ($parent,$dbName,$domain)=@_;

  my $dumpFile;

  my $item = makeXmlNode('database');
  $item->{'ATTRIBUTE'}->('name',$dbName);
  $item->{'ATTRIBUTE'}->('type','mysql');
  $item->{'ATTRIBUTE'}->('version', Db::MysqlUtils::getVersion());

  addDatabaseUsers($item,$dbName);

  if($dumpFile = makeDumpDb($dbName,'mysql','root',getMySqlPassword(),'localhost')) {
    $item->{'ATTRIBUTE'}->('cid',$dumpFile);
  }

  $parent->{'ADDCHILD'}->($item);
}

sub addDatabaseUsers {
  my ($dbNode, $dbName) = @_;

  printToLog("Dupming $dbName database users");

  my (@users, $dbUser, $ptrRow);
  my $sql = "SELECT DISTINCT User FROM db WHERE Db = '$dbName'";
  $wrapDbh->{'EXECUTE'}->($sql);
  while ($ptrRow = $wrapDbh->{'FETCHROW'}->()) {
    push @users, (@{$ptrRow})[0];
  }
  $wrapDbh->{'FINISH'}->();

  foreach $dbUser (@users) {
    addDatabaseUser($dbNode, $dbUser);
  }
}

my %dbUsers;

sub addDatabaseUser {
  my ($dbNode, $user) = @_;

  &printToLog("Dumping mysql user '$user'");

  my ( $item, $password );

  if ( ref( $wrapDbh ) =~ /HASH/ &&
			 ref( $wrapDbh->{'EXECUTE'} ) =~ /CODE/ ) {

  my ($sql,$ptrRow,@accesshosts,$accesshost);

  $sql = "SELECT password,host FROM user WHERE user='$user'";

  if ( $wrapDbh->{'EXECUTE'}->( $sql ) ) {
    $item = makeXmlNode( 'dbuser' );
    if ($user) {
      if ( exists( $dbUsers{$user} ) ) {
        my $clon = 1;
        while(exists($dbUsers{$user})){
          if($user =~ /(.*)c(\d+)$/){
            $user = $1;
            $clon = $2;
            $clon++;
          }
          $user .= "c$clon";
        }
      }
      $dbUsers{$user} = 1;

    }
    $item->{'ATTRIBUTE'}->('name', $user);

    while ( $ptrRow = $wrapDbh->{'FETCHROW'}->() ) {

      $password = $ptrRow->[0];

      push @accesshosts, $ptrRow->[1];
    }
    createPasswordNode($item, $password , 'encrypted');
    foreach $accesshost ( @accesshosts ) {
      $item->{'ADDCHILD'}->( makeXmlNode( 'accesshost', $accesshost ) );
    }
    $dbNode->{'ADDCHILD'}->( $item );
  }
  $wrapDbh->{'FINISH'}->();
  }
}

sub createLimitNode {
  my($root,$name,$value) = @_;
  my $item = &makeXmlNode('limit', $value);
  $item->{'ATTRIBUTE'}->('name', $name);
  $root->{'ADDCHILD'}->($item);
}

sub printHelp {

  print <<"HELP";

Usage:
$0 <options>

Options:
-s |--get-status           get status of the agent
-dc|--dump-accounts=<list> a comma separated list of resellers to dump
-dd|--dump-domains=<list>  a comma separated list of domains to dump
-da|--dump-all             make a full dump

-lc|--get-content-list     get list of content files
-nc|--no-content           do not make content files
-nz|--no-compress          do not compress content files

-o |--output=<path>        redirect output to file

-h |--help                 this help
-d |--debug=<level>        level of debug messages
-v |--version

HELP
}