#!/usr/bin/perl

use vars qw($dumpDir $ptrArgs $workDir $storage);
use strict;

use AgentConfig;
use XmlNode;

use lib "/usr/sausalito/perl";

#
# parse command line
#
my %arg_opts = ('--help|-h'=>'',
				'--get-status|-s'=>'',
				'--dump-accounts|-dc'=>'s',
				'--dump-domains|-dd'=>'s',
				'--dump-all|-da'=>'',
				'--get-content-list|-lc'=>'',
				'--no-content|-nc'=>'',
				'--no-compress|-nz'=>'',
				'--output|-o'=>'s',
				'--status-file|-sf'=>'s',
				'--raw|-r'=>'',
	       );

my $megabyte = 1024*1024;

my %domain_permissions = ('shell' => 'Shell', 'cgi' => 'CGI', 'ssi' => 'SSI',
						  'frontpage' => 'FP', 'apop' => 'APOP', 'php' => 'PHP',
						  'jsp' => 'Java', 'ssl' => 'SSL');

my %user_permissions = ('shell' => 'Shell',
						'frontpage' => 'FP',
						'apop' => 'APOP' );

#@@INCLUDE FILE="agent.include.pl"@@
if(-f 'agent.include.pl'){
  require 'agent.include.pl';
}else{
  require '../agent.include.pl';
}
#@@/INCLUDE@@

$ptrArgs = &getArguments (\@ARGV,\%arg_opts);

my ($outPath,$dumpFile,$statusFile);

$workDir = AgentConfig::cwd();

$dumpFile = $workDir.'/dump.xml';
$statusFile = $workDir.'/dumping-status.xml';
$dumpDir = $workDir.'/pma';

my $objDumpStatus = &makeDumpStatus($ptrArgs->{'status-file'}||$statusFile);

#
# get MIME Base64 encoder
#
my $wrapBase64 = makeMIMEBase64();

my $rootName = 'RaQ550dump';
my $dtdName = 'RaQ550.dtd';

#
# Open connection to the Sausalito CCE (Cobalt's server engine interface)
#

my $cce;
unless((eval "require CCE") and $cce = new CCE) {
  $cce = undef;
} else {
  $cce->connectuds();
}

#
# RAW DUMP (DEBUG)
#

if(exists $ptrArgs->{'raw'}) {
  my (undef, @classes) = $cce->classes();

  my ($class, $oid, $namespace, $prop, $value);
  foreach $class (@classes) {
	print "Class " . $class . "\n";

	my @oids = $cce->find($class);
	foreach $oid (@oids) {
	  print $oid . ":\n";
	  my (undef, @namespaces) = $cce->names($oid);
	  push @namespaces, '';

	  foreach $namespace (@namespaces) {
		print $namespace . ": ";

		my (undef, $object) = $cce->get($oid, $namespace);

		while(($prop, $value) = each %{$object}) {
		  print $prop.'->'.$value.'; ';
		}
		print "\n";
	  }
	}
  }
  exit(0);
}

my %users;
my %domainToUsers;
my %domainToMaillists;
my %domainToAdmins;
my %adminToDomains;

my @allMaillists;

#---------------------------------------
# Cache required server data structures
#---------------------------------------

my @allDomains = $cce->find('Vsite');

# Create domain->users, domain->domain-admins relations

my @userlist = $cce->find('User');
my $user;
foreach $user (@userlist) {
  my (undef, $object) = $cce->get($user, '');
  $users{$object->{'name'}} = $user;

  unless (defined $domainToUsers{$object->{'site'}}) {
    $domainToUsers{$object->{'site'}} = [];
  }
  push @{$domainToUsers{$object->{'site'}}}, $user;

  if ($object->{'capLevels'} =~ /siteAdmin/) {
	unless ($domainToAdmins{$object->{'site'}}) {
	  $domainToAdmins{$object->{'site'}} = [];
	}
	push @{$domainToAdmins{$object->{'site'}}}, $object->{'name'};
  }
}

# Create lexically-first-domain-admin -> domains relation
my $domain;
foreach $domain (keys %domainToAdmins) {
  my @admins = sort @{$domainToAdmins{$domain}};
  if (@admins > 0) {
	$domainToAdmins{$domain} = $admins[0];
  } else {
	$domainToAdmins{$domain} = undef;
  }
}

# Create domain -> maillists relation

@allMaillists = $cce->find('MailList');
my $list;
foreach $list (@allMaillists) {
  my (undef, $object) = $cce->get($list, '');
  unless (defined $domainToMaillists{$object->{'site'}}) {
    $domainToMaillists{$object->{'site'}} = [];
  }
  push @{$domainToMaillists{$object->{'site'}}}, $list;
}

###########

my $defaultCobaltDomains = "Default Cobalt Domains";
my $defaultCobaltDomainsSysuser = "defaultcobaltdomains";

if(exists $ptrArgs->{'get-status'}){
  printAgentStatus();
  exit 0;
}

die "Cannot connect to the RaQ Sausalito CCE" unless $cce;

if (exists ($ptrArgs->{'dump-all'}) ||
   exists ($ptrArgs->{'dump-accounts'}) ||
   exists ($ptrArgs->{'dump-domains'})) {

  my ($root,@accounts,@domains,$ptrAccounts,$ptrDomains, $value);
  initStorage();
  getDumpDir($dumpDir);

  if (exists $ptrArgs->{'no-compress'}) {
	setCompress(0);
  }

  &printToLog("Work dir: $workDir");
  &printToLog("Dump file: $dumpFile");
  &printToLog("Status file: ".$objDumpStatus->{'FILE'}->());


  if ($value = $ptrArgs->{'dump-accounts'}){
  	if ($value eq "-") {
		$value = <STDIN>;
		chomp $value;
	}
    @accounts = split(/\s*,\s*/,$value);
    $ptrAccounts = \@accounts;
  }

  if ($value = $ptrArgs->{'dump-domains'}){
  	if ($value eq "-") {
		$value = <STDIN>;
		chomp $value;
	}
    @domains = split(/\s*,\s*/,$value);
    $ptrDomains = \@domains;
  }
#
# generate a xml dump
#

  $root = &getRaQ550Dump($dumpDir, $ptrAccounts, $ptrDomains, $rootName);

#
# print dump to output
#
  $storage->finish($root);
}elsif(exists $ptrArgs->{'get-content-list'}){
  makeContentList();
}else{
  &printHelp();
}

exit 0;

#######################################################

sub printAgentStatus {
  my $root = makeXmlNode('agent-status');

  unless (defined($cce) && defined AgentConfig::iconvBin()) {
    my $item;
	if (defined AgentConfig::iconvBin()) {
	  $item = makeXmlNode('wrong-platform', '');
	} else {
	  $item = makeXmlNode('wrong-platform', 'no iconv found on the source host');
	}

    $root->{'ADDCHILD'}->($item);
  }

  printXml($root, *STDOUT);
}

sub getRaQ550Dump() {
  my ($dumpDir, $ptrAccounts, $ptrDomains, $rootName) = @_;

  my $root = makeXmlNode($rootName);
  $root->{'ATTRIBUTE'}->('agent-name','RaQ550');

  if (ref($ptrDomains) =~ /ARRAY/) {
    createDomains($root, $ptrDomains);
  } elsif (ref($ptrAccounts) =~ /ARRAY/) {
	createAccounts($root, $ptrAccounts);
  } else {
    createAllAccounts($root);
  }

  return $root;
}

sub createAccounts {
  my ($parent, $ptrAccounts) = @_;

  $objDumpStatus->{'COUNTS'}->(1, scalar(@allDomains));
  $objDumpStatus->{'ACCOUNT'}->($defaultCobaltDomains);
  $objDumpStatus->{'PRINT'}->();

  my $account;
  foreach $account (@{$ptrAccounts}) {
	if ($account eq $defaultCobaltDomainsSysuser) {
	  createDomainsByOid($parent, \@allDomains);
	}
  }
}

sub createAllAccounts {
  my ($parent) = @_;

  $objDumpStatus->{'COUNTS'}->(1, scalar(@allDomains));
  $objDumpStatus->{'ACCOUNT'}->($defaultCobaltDomains);
  $objDumpStatus->{'PRINT'}->();

  createDomainsByOid($parent, \@allDomains)
}

sub createDummyClient {
  my $dummyclient = makeXmlNode('client');
  $dummyclient->{'ATTRIBUTE'}->('disk-quota', '-1');
  $dummyclient->{'ATTRIBUTE'}->('enabled', 'true');
  $dummyclient->{'ATTRIBUTE'}->('fullname', $defaultCobaltDomains);
  createSysuserNode($dummyclient, $defaultCobaltDomainsSysuser, '', 'plain');

  return $dummyclient;
}

sub createDomains {
  my ($parent, $ptrDomains) = @_;

  my $dummyclient = createDummyClient();

  $objDumpStatus->{'COUNTS'}->(0, scalar(@{$ptrDomains}));
  $objDumpStatus->{'PRINT'}->();

  my ($domainName);
  foreach $domainName (@{$ptrDomains}) {
    createDomain($dummyclient, $domainName);
  }

  $parent->{'ADDCHILD'}->($dummyclient);
}

sub createDomainsByOid {
  my ($parent, $ptrOids) = @_;

  my $dummyclient = createDummyClient();

  my $oid;
  foreach $oid (@{$ptrOids}) {
    createDomainByOid($dummyclient, $oid);
  }

  $parent->{'ADDCHILD'}->($dummyclient);
}

sub getDomainName {
  my ($props) = @_;

  if($props->{'hostname'} eq 'www') {
	return $props->{'domain'};
  }
  return $props->{'fqdn'};
}

sub createDomain {
  my ($parent, $domainName) = @_;
  my @il = $cce->find('Vsite', {'fqdn' => $domainName});
  if(@il == 0) {
	@il = $cce->find('Vsite', {'domain' => $domainName, 'hostname' => 'www'});
	if(@il == 0) {
	  printToLog("No $domainName found");
	  return;
	}
  }
  createDomainByOid($parent, $il[0]);
}

sub createDomainByOid {
  my ($parent, $oid) = @_;

  my ($ok, $props) = $cce->get($oid, '');
  unless($ok) {
	printToLog("Unable to retrieve $oid domain data");
	return;
  }

  $objDumpStatus->{'DOMAIN'}->(getDomainName($props));
  $objDumpStatus->{'PRINT'}->();

  my $item = makeXmlNode('domain');
  $parent->{'ADDCHILD'}->($item);

  $item->{'ATTRIBUTE'}->('name', getDomainName($props));
  $item->{'ATTRIBUTE'}->('ip', $props->{'ipaddr'});

  # No more info required in 'no-content' mode
  if (exists $ptrArgs->{'no-content'}) {
	return;
  }

  $item->{'ATTRIBUTE'}->('cid',
			 makeDumpFile("$dumpDir/$oid-www", $props->{'basedir'}."/web"), '.', [], '-h');
  if($props->{'suspend'} eq '1') {
    $item->{'ATTRIBUTE'}->('state', 'false');
  } else {
    $item->{'ATTRIBUTE'}->('state', 'true');
  }

  if ($domainToAdmins{$props->{'name'}}) {
	$item->{'ATTRIBUTE'}->('admin-user-name',
						   $domainToAdmins{$props->{'name'}});
  };

  createDomainAliases($item, $props);

  my (undef, $diskprops) = $cce->get($oid, 'Disk');

  my $disk_quota = adaptLimit($diskprops, 'quota', sub { $_[0] * $megabyte });

  my $disklimitnode = makeXmlNode('limit', $disk_quota);
  $disklimitnode->{'ATTRIBUTE'}->('name', 'disk-quota');
  $item->{'ADDCHILD'}->($disklimitnode);

  createPermissionNodes($item, \%domain_permissions, $oid);

  createAnonFtp($item, $oid, $props->{'basedir'});

  createDomainUsers($item, $props->{'name'}, $props->{'basedir'}, getDomainName($props));

  createMaillists($item, $props->{'name'}, getDomainName($props));
}

sub createDomainAliases {
  my ($parent, $props) = @_;

  my @mx_aliases = $cce->scalar_to_array($props->{'mailAliases'});
  my @ptr_aliases = $cce->scalar_to_array($props->{'webAliases'});

  my (%mailaliases, %webaliases);
  my $alias;
  foreach $alias (@mx_aliases) {
     $mailaliases{$alias} = 1;
  }
  foreach $alias (@ptr_aliases) {
     $webaliases{$alias} =1;
  }

  my ($pointer, $k, $v, %aliases);

  foreach $pointer (\%mailaliases, \%webaliases) {
    while (($k, $v) = each %{$pointer}) {
	next if exists $aliases{$k};
	$aliases{$k} = 1;
    }
  }

  foreach $alias (keys %aliases) {

    my $danode = XmlNode->new("domain-alias", 'attributes' => {'name' => $alias});
    $danode->addChild(XmlNode->new('status', 'children' => [XmlNode->new('enabled')]));
    $danode->setAttribute('mail', (exists $mailaliases{$alias}) ? 'true' : 'false');
    $danode->setAttribute('web', (exists $webaliases{$alias}) ? 'true' : 'false');
    $parent->{'ADDCHILD'}->($danode);
  }
}

sub createAnonFtp {
  my ($parent, $oid, $basedir) = @_;

  printToLog(".anonftp");

  my (undef, $anonftpprops) = $cce->get($oid, 'AnonFtp');

  if($anonftpprops->{'enabled'} eq '0') {
    return;
  }

  my $item = makeXmlNode('anonftp');

  my $disklimit = makeXmlNode('limit', adaptLimit($anonftpprops, 'quota', sub {$_[0]*$megabyte}));
  $disklimit->{'ATTRIBUTE'}->('name', 'disk-quota');
  $item->{'ADDCHILD'}->($disklimit);

  my $accountlimit = makeXmlNode('limit', adaptLimit($anonftpprops, 'maxConnections'));
  $accountlimit->{'ATTRIBUTE'}->('name', 'account');
  $item->{'ADDCHILD'}->($accountlimit);

  my $anonftpdump = makeDumpFile("$dumpDir/$oid-anonftp", $basedir . "/ftp", ".", "incoming");

  if(defined $anonftpdump) {
    $item->{'ATTRIBUTE'}->('cid', $anonftpdump);

    my $anonftpincomingdump = makeDumpFile("$dumpDir/$oid-anonftpincoming", $basedir . "/ftp/incoming");
    if(defined $anonftpincomingdump) {
      $item->{'ATTRIBUTE'}->('cid_incoming', $anonftpincomingdump);
    }
  }

  $parent->{'ADDCHILD'}->($item);
}

sub createDomainUsers {
  my ($parent, $site, $basedir, $domainName) = @_;

  printToLog(".users");

  my $userid;
  foreach $userid (@{$domainToUsers{$site}}) {
    createDomainUser($parent, $userid, $basedir, $domainName);
  }
}

sub createDomainUser {
  my ($parent, $userid, $basedir, $domainName) = @_;

  my (undef, $props) = $cce->get($userid, '');
  my (undef, $diskprops) = $cce->get($userid, 'Disk');

  printToLog("..user " . $props->{'name'});

  my $item = makeXmlNode('user');
  $item->{'ATTRIBUTE'}->('fullname', $props->{'fullName'});
  $item->{'ATTRIBUTE'}->('disk-quota', adaptLimit($diskprops, 'quota', sub {$_[0]*$megabyte}));

  if($props->{'enabled'} eq '1') {
    $item->{'ATTRIBUTE'}->('state', 'true');
  } else {
    $item->{'ATTRIBUTE'}->('state', 'false');
  }

  createSysuser($item, $props);
  createPermissionNodes($item, \%user_permissions, $userid);
  createMail($item, $userid, $domainName, $basedir);

  $parent->{'ADDCHILD'}->($item);
}

sub createMail {
  my ($parent, $userid, $domainName, $basedir) = @_;
  my (undef, $props) = $cce->get($userid, '');
  my (undef, $mailprops) = $cce->get($userid, 'Email');

  my $item = makeXmlNode('mail');
  $item->{'ATTRIBUTE'}->('mailname', $props->{'name'});

  createForwardNode($item, $mailprops, $domainName);
  createAliasesNode($item, $mailprops);
  createAutoresponderNode($item, $mailprops);
  createMailboxNode($item, $props->{'name'}, $userid, $basedir);

  $parent->{'ADDCHILD'}->($item);
}

sub createForwardNode {
  my ($parent, $mailprops, $domainName) = @_;

  if($mailprops->{'forwardEnable'} eq '1') {
	my @forwards = $cce->scalar_to_array($mailprops->{'forwardEmail'});

	return unless @forwards;

	@forwards = map { $_ !~ /@/ ? $_ . "@" . $domainName : $_ } @forwards;

	if(@forwards == 1) {
	  $parent->{'ADDCHILD'}->(makeXmlNode('forward', $forwards[0]));
	} else {
	  my $item = makeXmlNode('mailgroup');
	  my $forward;
	  foreach $forward (@forwards) {
		$item->{'ADDCHILD'}->(makeXmlNode('forward', $forward));
	  }
	  $parent->{'ADDCHILD'}->($item);
	}
  }
}

sub createAliasesNode {
  my ($parent, $mailprops) = @_;

  my @aliases = $cce->scalar_to_array($mailprops->{'aliases'});
  my $alias;
  foreach $alias (@aliases) {
	$parent->{'ADDCHILD'}->(makeXmlNode('alias', $alias));
  }
}

sub createAutoresponderNode {
  my ($parent, $mailprops) = @_;

  my $text = $wrapBase64->{'ENCODE'}->($mailprops->{'vacationMsg'});
  my $item = makeXmlNode('autoresponder', $text);

  $item->{'ATTRIBUTE'}->('state',
						 $mailprops->{'vacationOn'} eq '1' ? 'true' : 'false');

  $parent->{'ADDCHILD'}->($item);
}

sub createMailboxNode {
  my ($parent, $name, $userid, $basedir) = @_;

  my $mboxdir = $basedir."/users/".$name;

  my $item = makeXmlNode('mailbox');
  my @extraParams = ('-h');
  $item->{'ATTRIBUTE'}->('type', 'mbox');
  if(-r $mboxdir."/mbox" or -l $mboxdir."/mbox") {
	my $cid = makeDumpFile("$dumpDir/${userid}-mbox", $mboxdir, 'mbox', [], \@extraParams);
	$item->{'ATTRIBUTE'}->('cid_inbox', $cid);
  }
  # no 'additional' mailboxes found ('cid')
  $parent->{'ADDCHILD'}->($item);
}

sub createMaillists {
  my ($parent, $site, $domain_name) = @_;

  printToLog(".maillists");

  my $listid;
  foreach $listid (@{$domainToMaillists{$site}}) {
    createMaillistByOid($parent, $listid, $domain_name);
  }
}

sub createMaillistByOid {
  my ($parent, $oid, $domain_name) = @_;

  my (undef, $props) = $cce->get($oid, '');

  printToLog("..maillist " . $props->{'name'});

  my $mlnode = makeXmlNode('maillist');
  $mlnode->{'ATTRIBUTE'}->('owner', $props->{'moderator'}.'@'.$domain_name);
  if ($props->{'apassword'} eq '') {
	$mlnode->{'ADDCHILD'}->(makePasswordNode('', 'plain'));
  } else {
	$mlnode->{'ADDCHILD'}->(makePasswordNode($props->{'apassword'}, 'plain'));
  }
  $mlnode->{'ATTRIBUTE'}->('name', $props->{'name'});
  $mlnode->{'ATTRIBUTE'}->('post-policy', $props->{'postPolicy'});
  $mlnode->{'ATTRIBUTE'}->('subscribe-policy', $props->{'subPolicy'});

  my @local_recipients = $cce->scalar_to_array($props->{'local_recips'});
  my @remote_recipients = $cce->scalar_to_array($props->{'remote_recips'});

  my $recip;
  foreach $recip (@local_recipients) {
    my $recipient = makeXmlNode('recipient', $recip . '@' . $domain_name);
    $mlnode->{'ADDCHILD'}->($recipient);
  }
  foreach $recip (@remote_recipients) {
    my $recipient = makeXmlNode('recipient', $recip);
    $mlnode->{'ADDCHILD'}->($recipient);
  }
  $parent->{'ADDCHILD'}->($mlnode);
}

sub createPermissionNodes {
  my ($parent, $permissionsPtr, $oid) = @_;

  my ($limit, $namespace);
  while (($limit, $namespace) = each(%{$permissionsPtr})) {
    my (undef, $params) = $cce->get($oid, $namespace);
    if($params->{'enabled'} eq '1') {
      my $permissionNode = makeXmlNode('permission');
      $permissionNode->{'ATTRIBUTE'}->('name', $limit);;
      $parent->{'ADDCHILD'}->($permissionNode);
    }
  }
}

sub createSysuser {
  my ($parent, $props) = @_;

  my $name = $props->{'name'};
  my $passwdtype = "" ne $props->{'md5_password'} ? 'encrypted' : 'plain';
  my $passwd = $props->{'md5_password'} || $props->{'password'};
  createSysuserNode($parent, $name, $passwd, $passwdtype);
}

sub adaptLimit {
  my ($props, $name, $adapter) = @_;

  my $value = $props->{$name};
  if("" eq $value) {
    $value = -1;
  } elsif (defined $adapter) {
    $value = &$adapter($value);
  }
  return $value;
}

sub printHelp {

  print <<"HELP";

Usage:
  $0 <options>

Options:

  -s |--get-status           get status of the agent
  -dc|--dump-accounts=<list> a coma separated list of resellers to dump
  -dd|--dump-domains=<list>  a coma separated list of customers to dump
  -da|--dump-all             make a full dump

  -lc|--get-content-list     get list of content files
  -nc|--no-content           do not make content files
  -nz|--no-compress          do not compress content files

  -h |--help                 this help

HELP
}

