import posixpath
from parallels.common.actions.base.subscription_action import SubscriptionAction
from parallels.common.utils.file_permissions import \
		bulk_change_permissions, PermissionRule, PERMISSION_CHANGE_MODE

class FixPermissions(SubscriptionAction):
	def get_description(self):
		return "Fix permissions for web files"

	def get_failure_message(self, global_context, subscription):
		return "Failed to fix permissions for web files of subscription '%s'" % (subscription.name)

	def filter_subscription(self, global_context, subscription):
		return True

	def run(self, global_context, subscription):
		"""Set file owner and permissions for transferred subscription content.
		
		Plesk vhost permissions are described in 'Parallels Plesk Panel 12.0
		Advanced Administration Guide':
		http://download1.parallels.com/Plesk/PP12/12.0/Doc/en-US/online/plesk-linux-advanced-administration-guide/68806.htm
		"""
		target_vhost_dir = posixpath.join(
			subscription.web_target_server.vhosts_dir, 
			subscription.name.encode('idna')
		)
		sysuser_name = subscription.converted_backup.get_phosting_sysuser_name()

		def httpdocs_permissions_list(path):
			return [
				PermissionRule(
					mode=PERMISSION_CHANGE_MODE.SELF, path=path, 
					user=sysuser_name, group='psaserv', 
					permissions='u=rwx,g=rx,o='
				),
				PermissionRule(
					mode=PERMISSION_CHANGE_MODE.SUBFILES_AND_SUBDIRS, 
					path=path, 
					user=sysuser_name, group='psacln', 
				),
				# cgi-bin is the last so group is not 
				# overwritten by changes before
				PermissionRule( 
					mode=PERMISSION_CHANGE_MODE.SELF, path='%s/cgi-bin' % path,
					user=sysuser_name, group='psaserv', 
					permissions='u=rwx,g=rx,o='
				),
			]

		permissions_list = []

		permissions_list.extend(httpdocs_permissions_list('httpdocs'))
		for subdomain in subscription.converted_backup.iter_subdomains():
			permissions_list.extend(
				httpdocs_permissions_list(
					'subdomains_wwwroot/%s' % subdomain.short_name
				)
			)

		for webuser_name in subscription.converted_backup.get_phosting_webuser_names():
			permissions_list.extend([
				PermissionRule(
					mode=PERMISSION_CHANGE_MODE.SELF, 
					path='web_users/%s' % webuser_name,
					user=sysuser_name, group='psaserv',
					permissions='u=rwx,g=rx,o='
				),
				PermissionRule(
					mode=PERMISSION_CHANGE_MODE.SUBFILES_AND_SUBDIRS, 
					path='web_users/%s' % webuser_name,
					user=sysuser_name, group='psacln',
				),
			])

		permissions_list.extend([
			PermissionRule(
				mode=PERMISSION_CHANGE_MODE.SELF, path='anon_ftp/pub',
				user=sysuser_name, group='psaserv',
				permissions='u=rwx,g=rx,o=rx'
			),
			PermissionRule(
				mode=PERMISSION_CHANGE_MODE.SUBFILES_AND_SUBDIRS, 
				path='anonftp/pub',
				user=sysuser_name, group='psaserv',
			),
			PermissionRule(
				mode=PERMISSION_CHANGE_MODE.SELF, path='anon_ftp/incoming',
				user=sysuser_name, group='psaserv',
				permissions='u=rwx,g=rwx,o=rwx'
			),
			PermissionRule(
				mode=PERMISSION_CHANGE_MODE.SUBFILES_AND_SUBDIRS, 
				path='anonftp/incoming',
				user=sysuser_name, group='psaserv',
			),
		])

		with subscription.web_target_server.runner() as runner_target:
			bulk_change_permissions(runner_target, target_vhost_dir, permissions_list)

