from parallels.core.actions.base.base_action \
    import BaseAction
from parallels.core.actions.utils.multithreading_properties import MultithreadingProperties


class SubscriptionAction(BaseAction):
    """Base class for actions performed on single subscription

    Action is a single operation in migration process. Subscription action
    could be performed on each subscription independently.

    Examples of actions: copy web content of a subscription, copy mail messages
    of a subscription, restore hosting settings of a subscription.
    """

    def is_critical(self):
        """If action is critical or not

        If action is critical and it failed for a subscription, migration tool
        won't run the next operations for the subscription.

        :rtype: bool
        """
        return True

    def is_execute_for_failed_subscriptions(self):
        """Whether this action should be executed for completely failed subscription or not

        For example, subscription is completely failed if we failed to create it on the target server.
        Default is False, most of the actions should not be executed for such subscriptions.
        But some, for example print subscription report, should be executed anyway.

        :rtype: bool
        """
        return False

    def filter_subscription(self, global_context, subscription):
        """Check if we should run this action on given subscription or not

        The function returns true if it should be executed for the given
        subscription, and false if we should not run the action. For example,
        we should not work with .Security files for Linux subscriptions -
        filter them out in that function.

        Arguments:
        - global_context - registry with different objects that reused among different actions
        - subscription - object with information about single subscription

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :rtype: bool
        """
        return True

    def get_failure_message(self, global_context, subscription):
        """Get message for situation when action failed

        Arguments:
        - global_context - registry with different objects that reused among different actions
        - subscription - object with information about single subscription

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :rtype: basestring
        """
        raise NotImplementedError()

    def get_multithreading_properties(self):
        """Get how multithreading should be applied for that action

        :rtype: parallels.core.actions.utils.multithreading_properties.MultithreadingProperties
        """
        return MultithreadingProperties()

    def get_repeat_count(self):
        """Gets a number of attempts to perform the action

        :rtype: int
        """
        return 3

    def run(self, global_context, subscription):
        """Run action on given subscription

        Arguments:
        - global_context - registry with different objects that reused among different actions
        - subscription - object with information about single subscription

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :rtype: None
        """
        raise NotImplementedError()
