from parallels.core import messages
import logging

from parallels.core.actions.base.common_action import CommonAction
from parallels.core.actions.hosting_settings.import_backups import ImportedBackups
from parallels.core.actions.utils.logging_properties import LoggingProperties
from parallels.core.utils.restore_hosting_utils import get_restore_hosting_utils

logger = logging.getLogger(__name__)


class RemoveImportedBackups(CommonAction):
    """Remove imported Plesk backup files from target panel's PMM repository

    Customers could see parts of the backup in their control panels if we don't remove it, so
    we should remove backups not bother customers. Also, there will be a useless
    garbage which wastes disk space if we won't remove them.
    """

    def get_description(self):
        return messages.REMOVE_IMPORTED_BACKUP_DUMPS_FROM_TARGET

    def get_failure_message(self, global_context):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        return messages.FAILED_REMOVE_IMPORTED_BACKUP_DUMPS_TARGET

    def is_critical(self):
        """If action is critical or not

        If action is critical and it failed for a subscription, migration tool
        won't run the next operations for the subscription.

        :rtype: bool
        """
        return False

    def get_logging_properties(self):
        """Get how action should be logged to migration tools end-user

        :rtype: parallels.core.actions.utils.logging_properties.LoggingProperties
        """
        return LoggingProperties(compound=False)

    def run(self, global_context):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        restore_hosting_utils = get_restore_hosting_utils(global_context.conn.target.plesk_server)
        for backup in ImportedBackups.get_instance().get_all():
            restore_hosting_utils.remove_backup(backup.backup_info_file)
