import ntpath
import os

from parallels.core import messages
from parallels.core.content.mail.base import CopyMailBase
from parallels.core.reports.model.issue import Issue
from parallels.core.runners.utils import pipe_commands, transfer_file
from parallels.core.utils import get_thirdparties_base_path
from parallels.core.utils.common import cached, is_empty, safe_format
from parallels.core.utils.common.logging import create_safe_logger
from parallels.core.utils.common.threading_utils import synchronized, synchronized_by_args
from parallels.core.utils.download_utils import download_zip
from parallels.core.utils.windows_utils import format_command

logger = create_safe_logger(__name__)


class CopyMailMailMigrator(CopyMailBase):
    """Copy mail content with MailMigrator.exe utility

    The utility could dump mail messages for specified mailbox and restore them on another server.
    It dumps messages into universal format, that does not depend on mail server software. For example,
    it allows to transfer messages from MailEnable to SmarterMail and vice verso. The format is binary,
    for better performance.

    Here we copy mail messages by streaming data from the utility on source server
    to the utility on target server, with no intermediate files with messages. That provides:
    - less disk space requirements: we don't create archive with messages
    - (in theory) better performance, as there is no disk I/O
    - potential ability to track progress: how many messages were completely transferred.
    """
    def copy_mail(self, global_context, subscription, issues):
        for domain in subscription.mail_converted_dump.iter_domains():
            if domain.mailsystem is None:
                logger.debug(messages.SKIP_COPYING_MAIL_CONTENT_FOR_DOMAIN, domain.name)
                continue

            self._copy_domain_mail(subscription, domain, issues)

    def _copy_domain_mail(self, subscription, domain, issues):
        for mailbox in domain.iter_mailboxes():
            logger.finfo(messages.COPY_MESSAGES_OF_MAILBOX, mailbox=mailbox.full_name)
            self._copy_mailbox(subscription, domain, mailbox, issues)

    def _copy_mailbox(self, subscription, domain, mailbox, issues):
        source_server = subscription.mail_source_server
        target_server = subscription.mail_target_server

        target_mailbox_temp_dir = target_server.get_session_file_path(
            ntpath.join('mail', domain.name.encode('idna'), mailbox.name.encode('idna'))
        )
        source_mailbox_temp_dir = source_server.get_session_file_path(
            ntpath.join('mail', domain.name.encode('idna'), mailbox.name.encode('idna'))
        )
        target_mailbox_description_file = ntpath.join(target_mailbox_temp_dir, 'mailbox-description')
        source_mailbox_description_file = ntpath.join(source_mailbox_temp_dir, 'mailbox-description')
        target_exclude_message_ids_file = ntpath.join(target_mailbox_temp_dir, 'exclude-message-ids')
        source_exclude_message_ids_file = ntpath.join(source_mailbox_temp_dir, 'exclude-message-ids')

        mail_migrator_deployer = MailMigratorDeployer.get_instance()
        target_mail_migrator_exe = mail_migrator_deployer.deploy(target_server)
        source_mail_migrator_exe = mail_migrator_deployer.deploy(source_server)

        with source_server.runner() as runner_source, target_server.runner() as runner_target:
            mailbox_description_contents = "\n".join([
                domain.name.encode('idna'), mailbox.name, mailbox.password
            ])
            runner_target.mkdir(target_mailbox_temp_dir)
            runner_source.mkdir(source_mailbox_temp_dir)

            # Create files describing mailbox to transfer
            runner_target.upload_file_content(target_mailbox_description_file, mailbox_description_contents)
            runner_source.upload_file_content(source_mailbox_description_file, mailbox_description_contents)

            list_message_ids_command_str, list_message_ids_command_options = self._create_list_message_ids_command(
                source_server, target_exclude_message_ids_file, target_mail_migrator_exe,
                target_mailbox_description_file, target_mailbox_temp_dir
            )

            runner_target.sh(list_message_ids_command_str, list_message_ids_command_options)

            transfer_file(
                runner_target, target_exclude_message_ids_file, runner_source, source_exclude_message_ids_file
            )

            backup_command_str, backup_command_options = self._create_backup_command(
                source_server, source_exclude_message_ids_file,
                source_mail_migrator_exe, source_mailbox_description_file, source_mailbox_temp_dir,
            )
            restore_command_str, restore_command_options = self._create_restore_command(
                source_server, target_mail_migrator_exe, target_mailbox_description_file, target_mailbox_temp_dir
            )

            # Start utility on the source server, in backup mode. Start utility on the target server in restore mode.
            # Attach stdout of utility on the source server (which writes message dumps to stdout) to stdin
            # of utility on the target server (which accepts message dumps to stdin)
            execution_results = pipe_commands(
                runner_source, backup_command_str, backup_command_options,
                runner_target, restore_command_str, restore_command_options
            )

            # remove files with mailbox description, as they could contain password of mailbox
            runner_target.remove_file(target_mailbox_description_file)
            runner_source.remove_file(source_mailbox_description_file)

            self._check_results(
                execution_results, issues,
                mailbox,
                backup_command_options, backup_command_str,
                restore_command_options, restore_command_str
            )

    @staticmethod
    def _create_restore_command(
        source_server, target_mail_migrator_exe, target_mailbox_description_file, target_mailbox_temp_dir
    ):
        restore_command_str = (
            "{mail_migrator_exe} restore --account-description-file={description}"
        )
        restore_command_options = dict(
            description=target_mailbox_description_file,
            mail_migrator_exe=target_mail_migrator_exe,
        )
        if source_server.mail_settings.target_log_enabled:
            restore_command_str += ' --log-file={log_file}'
            restore_command_options['log_file'] = ntpath.join(target_mailbox_temp_dir, 'mail-restore.log')
        if not is_empty(source_server.mail_settings.target_provider):
            restore_command_str += ' --provider={provider}'
            restore_command_options['provider'] = source_server.mail_settings.target_provider
        if not is_empty(source_server.mail_settings.target_additional_options):
            restore_command_str += ' ' + source_server.mail_settings.target_additional_options

        return restore_command_str, restore_command_options

    @staticmethod
    def _create_backup_command(
        source_server, source_exclude_message_ids_file, source_mail_migrator_exe,
        source_mailbox_description_file, source_mailbox_temp_dir
    ):
        backup_command_str = (
            "{mail_migrator_exe} backup --account-description-file={description} "
            "--exclude-message-ids-file={exclude_ids_file}"
        )
        backup_command_options = dict(
            description=source_mailbox_description_file,
            mail_migrator_exe=source_mail_migrator_exe,
            exclude_ids_file=source_exclude_message_ids_file,
        )
        if source_server.mail_settings.source_log_enabled:
            backup_command_str += ' --log-file={log_file}'
            backup_command_options['log_file'] = ntpath.join(source_mailbox_temp_dir, 'mail-backup.log')
        if not is_empty(source_server.mail_settings.source_provider):
            backup_command_str += ' --provider={provider}'
            backup_command_options['provider'] = source_server.mail_settings.source_provider
        if not is_empty(source_server.mail_settings.source_additional_options):
            backup_command_str += ' ' + source_server.mail_settings.source_additional_options
        return backup_command_str, backup_command_options

    @staticmethod
    def _create_list_message_ids_command(
        source_server, target_exclude_message_ids_file, target_mail_migrator_exe,
        target_mailbox_description_file, target_mailbox_temp_dir
    ):
        list_message_ids_command_str = (
            "{mail_migrator_exe} list-message-ids "
            "--account-description-file={description} "
            "--output-file={exclude_ids_file}"
        )
        list_message_ids_command_options = dict(
            mail_migrator_exe=target_mail_migrator_exe,
            description=target_mailbox_description_file,
            exclude_ids_file=target_exclude_message_ids_file,
        )
        if source_server.mail_settings.target_log_enabled:
            list_message_ids_command_str += ' --log-file={log_file}'
            list_message_ids_command_options['log_file'] = ntpath.join(
                target_mailbox_temp_dir, 'mail-list-messages.log'
            )
        if not is_empty(source_server.mail_settings.target_provider):
            list_message_ids_command_str += ' --provider={provider}'
            list_message_ids_command_options['provider'] = source_server.mail_settings.target_provider
        if not is_empty(source_server.mail_settings.target_additional_options):
            list_message_ids_command_str += ' ' + source_server.mail_settings.target_additional_options
        return list_message_ids_command_str, list_message_ids_command_options

    @staticmethod
    def _check_results(
            execution_results, issues,
            mailbox,
            backup_command_options, backup_command_str,
            restore_command_options, restore_command_str
    ):
        if execution_results.source_exit_code != 0 or execution_results.source_stderr != '':
            issues.append(
                Issue(
                    'mail-migrator-backup-failed', Issue.SEVERITY_ERROR,
                    safe_format(
                        messages.WINDOWS_MAIL_BACKUP_FAILED,
                        mailbox=mailbox.full_name,
                        command=format_command(backup_command_str, **backup_command_options),
                        exit_code=execution_results.source_exit_code,
                        stderr=execution_results.source_stderr,
                    )
                )
            )
        if execution_results.target_exit_code != 0 or execution_results.target_stderr != '':
            issues.append(
                Issue(
                    'mail-migrator-restore-failed', Issue.SEVERITY_ERROR,
                    safe_format(
                        messages.WINDOWS_MAIL_RESTORE_FAILED,
                        mailbox=mailbox.full_name,
                        command=format_command(restore_command_str, **restore_command_options),
                        exit_code=execution_results.target_exit_code,
                        stderr=execution_results.target_stderr
                    )
                )
            )


class MailMigratorDeployer(object):
    """Deploy mail migrator to source and target servers"""

    MAIL_MIGRATOR_VERSION = '1.4'
    MAIL_MIGRATOR_PACKAGE = 'MailMigrator'

    def __init__(self):
        self._servers = dict()

    @staticmethod
    @synchronized
    @cached
    def get_instance():
        return MailMigratorDeployer()

    @synchronized_by_args
    def deploy(self, server):
        if server not in self._servers:
            store_path = self._download_package()
            with server.runner() as runner:
                for file_name in os.listdir(store_path):
                    local_path = os.path.join(store_path, file_name)
                    remote_path = server.get_session_file_path(file_name)
                    runner.upload_file(local_path, remote_path)
                    self._servers[server] = server.get_session_file_path('MailMigrator.exe')

        return self._servers[server]

    @synchronized
    @cached
    def _download_package(self):
        store_path = os.path.join(
            get_thirdparties_base_path(),
            '{package}_{version}'.format(
                package=self.MAIL_MIGRATOR_PACKAGE, version=self.MAIL_MIGRATOR_VERSION
            )
        )
        if not os.path.exists(store_path):
            store_url = 'http://autoinstall.plesk.com/panel-migrator/thirdparties/{package}_{version}.zip'.format(
                package=self.MAIL_MIGRATOR_PACKAGE, version=self.MAIL_MIGRATOR_VERSION
            )
            download_zip(store_url, store_path)

        return store_path
