"""Model representing objects that already exist in target panel"""

from parallels.core.utils.entity import Entity


class ExistingObjectsModel(Entity):
    """Model representing objects that already exist in target panel"""

    yaml_prefix = 'target'

    def __init__(
        self, resellers=None, customers=None, plans=None,
        addon_plans=None, subscriptions=None,
        domains=None, domain_aliases=None, subdomains=None,
        ip_addresses=None
    ):
        """Class constructor

        :type resellers: dict[str | unicode, parallels.core.hosting_repository.reseller.ResellerEntity]
        :type customers:  dict[str | unicode, parallels.core.hosting_repository.customer.CustomerEntity]
        :type plans: list[parallels.core.hosting_repository.service_plan.ServicePlanEntity]
        :type addon_plans: list[parallels.core.hosting_repository.service_plan_addon.ServicePlanAddonEntity]
        :type subscriptions: list[parallels.core.hosting_repository.subscription.SubscriptionEntity]
        :type domains: list[parallels.core.hosting_repository.domain.DomainEntity]
        :type domain_aliases: list[parallels.core.hosting_repository.domain_alias.DomainAliasEntity]
        :type subdomains: list[parallels.core.hosting_repository.subdomain.SubdomainEntity]
        :type ip_addresses: list[parallels.core.utils.plesk_utils.IPAddressInfo]
        """
        self._resellers = resellers
        self._customers = customers
        self._plans = plans
        self._addon_plans = addon_plans
        self._subscriptions = subscriptions
        self._domains = domains
        self._domain_aliases = domain_aliases
        self._subdomains = subdomains
        self._ip_addresses = ip_addresses

    @property
    def resellers(self):
        """Dictionary reseller login to reseller info for resellers on target server

        Resellers that are migrated are listed here only.

        :rtype: dict[str | unicode, parallels.core.hosting_repository.reseller.ResellerEntity]
        """
        return self._resellers

    @property
    def customers(self):
        """Dictionary customer login to customer info for customers on target server

        Customers that are migrated are listed here only.

        :rtype: dict[str | unicode, parallels.core.hosting_repository.customer.CustomerEntity]
        """
        return self._customers

    @property
    def plans(self):
        """List of plans on target server

        Plans for resellers that are migrated are listed here only, plus admin's plans.

        :rtype: list[parallels.core.hosting_repository.service_plan.ServicePlanEntity]
        """
        return self._plans

    @property
    def addon_plans(self):
        """List of addon plans on target server

        Plans for resellers that are migrated are listed here only, plus admin's addon plans.

        :rtype: list[parallels.core.hosting_repository.service_plan_addon.ServicePlanAddonEntity]
        """
        return self._addon_plans

    @property
    def subscriptions(self):
        """List of subscriptions on target server

        Subscriptions that are migrated are listed here only.

        :rtype: list[parallels.core.hosting_repository.subscription.SubscriptionEntity]
        """
        return self._subscriptions

    @property
    def domains(self):
        """List of all addon domains on target server

        :rtype: list[parallels.core.hosting_repository.domain.DomainEntity]
        """
        return self._domains

    @property
    def domain_aliases(self):
        """List of all domain aliases on target server

        :rtype: list[parallels.core.hosting_repository.domain_alias.DomainAliasEntity]
        """
        return self._domain_aliases

    @property
    def subdomains(self):
        """List of all subdomains on target server

        :rtype: list[parallels.core.hosting_repository.subdomain.SubdomainEntity]
        """
        return self._subdomains

    @property
    def ip_addresses(self):
        """List of IP addresses on target server

        :rtype: list[parallels.core.utils.plesk_utils.IPAddressInfo]
        """
        return self._ip_addresses
