import os

from parallels.core import messages
from parallels.core.runners.exceptions.non_zero_exit_code import NonZeroExitCodeException
from parallels.core.utils.clean_logging import clean_dict_args_for_log, clean_list_args_for_log
from parallels.core.utils.common import safe_string_repr
from parallels.core.utils.common.logging import hide_text, create_safe_logger
from parallels.core.utils.steps_profiler import get_default_steps_profiler

logger = create_safe_logger(__name__)
profiler = get_default_steps_profiler()


class BaseRunner(object):
    """Base class for runner abstraction, which provides ability to run commands and work with files on a server"""

    def __init__(self, host_description, hostname):
        self._host_description = host_description
        self._hostname = hostname
        if self._host_description is None:
            self._host_description = "server '%s'" % self._hostname

    @property
    def host_description(self):
        """Human-readable description of the host this runner is working with.

        Examples: "server '192.168.1.12'", "database server 'my-db-server.example.com'", "local server", etc
        Host description is used in error messages, when runner failed to execute some command.

        :rtype: str | unicode
        """
        return self._host_description

    @property
    def is_windows(self):
        raise NotImplementedError()

    def run_unchecked(
        self, cmd, args=None, stdin_content=None, output_codepage=None, error_policy='strict',
        env=None, log_output=True
    ):
        """Run command with specified args. Arguments are properly escaped.

        :type cmd: str | unicode
        :type args: list[str | unicode] | None
        :type stdin_content: str | unicode | None
        :type output_codepage: str | unicode | None
        :type error_policy: str | unicode
        :type env: dict[str | unicode]
        :type log_output: bool
        :rtype: tuple
        """
        command_str = self._format_run_command_for_log(cmd, args)
        logger.debug(messages.DEBUG_EXECUTE_COMMAND.format(
            server=self.host_description, command=command_str
        ))

        with profiler.measure_command_call(command_str, self.host_description):
            exit_code, stdout, stderr = self._run_unchecked_no_logging(
                cmd, args, stdin_content, output_codepage, error_policy, env
            )

        logger.debug(
            messages.DEBUG_COMMAND_RESULTS.format(
                stdout=hide_text(stdout, not log_output),
                stderr=hide_text(stderr, not log_output),
                exit_code=exit_code
            )
        )

        return exit_code, stdout, stderr

    def _run_unchecked_no_logging(
        self, cmd, args=None, stdin_content=None, output_codepage=None, error_policy='strict', env=None
    ):
        """Run command with specified args. Arguments are properly escaped. No logging is performed.

        :type cmd: str | unicode
        :type args: list[str | unicode] | None
        :type stdin_content: str | unicode | None
        :type output_codepage: str | unicode | None
        :type error_policy: str | unicode
        :type env: dict[str | unicode]
        :rtype: tuple
        """
        raise NotImplementedError()

    def sh_unchecked(
        self, cmd_str, args=None, stdin_content=None, output_codepage=None,
        error_policy='strict', env=None, log_output=True, working_dir=None,
        redirect_output_file=None
    ):
        """Substitute format fields in 'cmd_str', then execute the command string.

        For Unix: whole command is executed, then if you run

            "runner.sh('ls *')"

        , '*' will be substituted by shell, while if you run

            "runner.sh('ls {file}', dict(file='*'))"

        , '*' will be escaped, much like "run('ls', ['*'])".

        For Windows: there are no general command arguments parsing rules, and
        '*' is substituted by each particular program, this function differs
        from run_unchecked only by a way to substitute arguments.

        If None is passed as args, no command formatting is performed, command
        run as is.

        Returns a tuple (exit_code, stdout, stderr)

        :type cmd_str: str | unicode
        :type args: dict[str | unicode, str | unicode] | None
        :type stdin_content: str | unicode | None
        :type output_codepage: str | unicode | None
        :type error_policy: str | unicode
        :type env: dict[str | unicode]
        :type log_output: bool
        :type working_dir: str | unicode
        :type redirect_output_file: str | unicode | None
        :rtype: tuple
        """
        command_str = self._format_sh_command_for_log(cmd_str, args)
        logger.debug(messages.DEBUG_EXECUTE_COMMAND.format(
            server=self.host_description, command=command_str
        ))

        with profiler.measure_command_call(command_str, self.host_description):
            exit_code, stdout, stderr = self._sh_unchecked_no_logging(
                cmd_str, args, stdin_content, output_codepage, error_policy,
                env, log_output, working_dir, redirect_output_file=redirect_output_file
            )

        logger.debug(
            messages.DEBUG_COMMAND_RESULTS.format(
                stdout=hide_text(stdout, not log_output),
                stderr=hide_text(stderr, not log_output),
                exit_code=exit_code
            )
        )

        return exit_code, stdout, stderr

    def _sh_unchecked_no_logging(
        self, cmd_str, args=None, stdin_content=None, output_codepage=None,
        error_policy='strict', env=None, log_output=True, working_dir=None,
        redirect_output_file=None
    ):
        """Substitute format fields in 'cmd_str', then execute the command string. No logging is performed.

        :type cmd_str: str | unicode
        :type args: dict[str | unicode, str | unicode] | None
        :type stdin_content: str | unicode | None
        :type output_codepage: str | unicode | None
        :type error_policy: str | unicode
        :type env: dict[str | unicode]
        :type log_output: bool
        :type working_dir: str | unicode
        :type redirect_output_file: str | unicode | None
        :rtype: tuple
        """
        raise NotImplementedError()

    def run(
        self, cmd, args=None, stdin_content=None, output_codepage=None, error_policy='strict', env=None,
        log_output=True
    ):
        """The same as run_unchecked(), but checks exit code and returns only stdout

        :type cmd: str | unicode
        :type args: list[str | unicode] | None
        :type stdin_content: str | unicode | None
        :type output_codepage: str | unicode | None
        :type error_policy: str | unicode
        :type env: dict[str | unicode]
        :type log_output: bool
        :rtype: str | unicode
        """
        exit_code, stdout, stderr = self.run_unchecked(
            cmd, args, stdin_content, output_codepage, error_policy, env, log_output
        )
        return self._check_exit_code(
            self._format_run_command_for_log(cmd, args), exit_code, stdout, stderr, log_output
        )

    def sh(
        self, cmd, args=None, stdin_content=None, output_codepage=None, error_policy='strict',
        env=None, log_output=True, working_dir=None, redirect_output_file=None
    ):
        """The same as sh_unchecked(), but checks exit code and returns only stdout

        :type cmd: str | unicode
        :type args: dict[str | unicode, str | unicode] | None
        :type stdin_content: str | unicode | None
        :type output_codepage: str | unicode | None
        :type error_policy: str | unicode
        :type env: dict[str | unicode]
        :type log_output: bool
        :type working_dir: str | unicode
        :type redirect_output_file: str | unicode | None
        :rtype: str | unicode
        """
        exit_code, stdout, stderr = self.sh_unchecked(
            cmd, args, stdin_content, output_codepage, error_policy,
            env, log_output=log_output, working_dir=working_dir, redirect_output_file=redirect_output_file
        )
        return self._check_exit_code(self._format_sh_command_for_log(cmd, args), exit_code, stdout, stderr, log_output)

    def _check_exit_code(self, command_str, exit_code, stdout, stderr, log_output):
        """Check exit code of a command, raise exception if it is not equal to 0, return stdout if it is equal to 0

        :type command_str: str | unicode
        :type exit_code: int
        :type stdout: str | unicode
        :type stderr: str | unicode
        :raises: NonZeroExitCodeException
        :rtype: str | unicode
        """
        if exit_code != 0:
            raise NonZeroExitCodeException(
                messages.RUNNER_COMMAND_STR_FAILED_WITH_EXIT_CODE.format(
                    server=self.host_description, command=command_str,
                    exit_code=exit_code,
                    stdout=hide_text(safe_string_repr(stdout), not log_output),
                    stderr=hide_text(safe_string_repr(stderr), not log_output)
                ),
                stdout=stdout,
                stderr=stderr,
                exit_code=exit_code
            )
        else:
            return stdout

    def append_file_content(self, filename, content):
        raise NotImplementedError()

    def create_file(self, filename):
        raise NotImplementedError()

    def upload_file_content(self, filename, content):
        raise NotImplementedError()

    def upload_file(self, local_filename, remote_filename):
        raise NotImplementedError()

    def get_file(self, remote_filename, local_filename):
        raise NotImplementedError()

    def get_file_contents(self, remote_filename):
        raise NotImplementedError()

    def get_file_size(self, remote_filename):
        raise NotImplementedError()

    def move(self, src_path, dst_path):
        raise NotImplementedError()

    def remove_file(self, filename):
        """Remove file. If file does not exist - silently do nothing.

        :type filename: basestring
        """
        raise NotImplementedError()

    def remove_directory(self, directory, is_remove_root=True):
        raise NotImplementedError()

    def file_exists(self, filename):
        raise NotImplementedError()

    def is_dir(self, path):
        """Returns True, False or None if file does not exist
        :type path: str | unicode
        :rtype: bool | None
        """
        raise NotImplementedError()

    def mkdir(self, dirname):
        raise NotImplementedError()

    def download_directory(self, remote_directory, local_directory):
        raise NotImplementedError()

    def download_directory_as_zip(self, remote_directory, local_zip_filename):
        raise NotImplementedError()

    def check(self, server_description):
        """Check connection with that runner

        :type server_description: basestring
        :rtype None
        """
        raise NotImplementedError()

    def get_files_list(self, path):
        raise NotImplementedError()

    def iter_files_list_nested(self, path):
        """Iterate over all (including files in nested directories) file names inside a path

        :type path: str | unicode
        """
        raise NotImplementedError()

    def execute_sql(self, query, query_args=None, connection_settings=None, log_output=True):
        """Execute SQL query
        Returns list of rows, each row is a dictionary.
        Query and query arguments:
        * execute_sql("SELECT * FROM clients WHERE type=%s AND country=%s", ['client', 'US'])
        Connection settings: host, port, user, password, database

        :type query: str|unicode
        :type query_args: list|dict
        :type connection_settings: dict
        :type log_output: bool
        :rtype: list[dict]
        :raises: parallels.core.MigrationError
        """
        raise NotImplementedError()

    def resolve(self, address):
        """Resolve hostname into IP address.

        Return the first IP address, or None if we were not able to resolve.
        If address is already a valid IP address - return it as is.

        :type address: str | unicode
        :rtype: None | str | unicode
        """
        raise NotImplementedError()

    def resolve_all(self, address):
        """Resolve hostname into list of IP addresses.

        If address is already a valid IP address - return
        list that contains only that address.
        Otherwise resolve and return list of IP addresses
        (IPv6 and IPv4 all in one list).
        If we were not able to resolve, empty list is returned.

        :type address: str | unicode
        :rtype: list[str | unicode]
        """
        raise NotImplementedError()

    def add_allowed_program_firewall_rule(self, name, program):
        """
        :param str|unicode name: Firewall rule name
        :param str|unicode program: Full path to program executable
        """
        raise NotImplementedError()

    def delete_allowed_program_firewall_rule(self, name, program):
        """Delete firewall rule. Call is completed successfully for existent and non-existent rules.
        :param str|unicode name: Firewall rule name
        :param str|unicode program: Full path to program executable
        """
        raise NotImplementedError()

    def is_built_in_administrator(self, username):
        """Check that system account with given username is built-in administrator
        :type username: str
        :rtype: bool
        """
        raise NotImplementedError()

    def is_run_by_built_in_administrator(self):
        """Check that current process executed by built-in administrator
        :rtype: bool
        """
        raise NotImplementedError()

    def get_env(self, name):
        """Get value of specified environment variable

        :param str | unicode name: Name of variable, case-insensitive
        :rtype: str | unicode | None
        """
        for key, value in os.environ.iteritems():
            if key.upper() == name.upper():
                return value
        return None

    def mssql_execute(self, connection_settings, query, args=None):
        """Execute MSSQL query, do not return any results

        :type connection_settings: parallels.core.runners.entities.MSSQLConnectionSettings
        :type query: str | unicode
        :type args: dict | None
        :raises parallels.core.runners.exceptions.mssql.MSSQLException:
        :rtype: None
        """
        raise NotImplementedError()

    def mssql_query(self, connection_settings, query, args=None):
        """Execute MSSQL query, return results as a list of dictionaries

        :type connection_settings: parallels.core.runners.entities.MSSQLConnectionSettings
        :type query: str | unicode
        :type args: dict | None
        :raises parallels.core.runners.exceptions.mssql.MSSQLException:
        :rtype: list[dict]
        """
        raise NotImplementedError()

    def _log_mssql_query(self, connection_settings, query, args=None):
        if args is None:
            logger.fdebug(
                messages.EXECUTE_MSSQL_QUERY,
                executor_server=self._host_description, server=connection_settings.host,
                query=query
            )
        else:
            logger.fdebug(
                messages.EXECUTE_MSSQL_QUERY_ARGS,
                executor_server=self._host_description, server=connection_settings.host,
                query=query, arguments=repr(args)
            )

    @staticmethod
    def _log_mssql_query_result(result):
        logger.fdebug(messages.EXECUTE_MSSQL_QUERY_RESULT, result=repr(result))

    def _format_run_command_for_log(self, cmd, args=None):
        """Same as _format_run_command, but hides all data that looks like a password, for correct logging

        :type cmd: str | unicode
        :type args: list[str | unicode] | None
        :rtype: str | unicode
        """

        return self._format_run_command(cmd, clean_list_args_for_log(args))

    def _format_run_command(self, cmd, args=None):
        """Format command arguments passed for 'run' and 'run_unchecked' methods into command line string

        This command line will be added to log and error messages, so it is expected that you could
        simply copy-paste it from logs or migrator output, and execute.

        :type cmd: str | unicode
        :type args: list[str | unicode] | None
        :rtype: str | unicode
        """
        raise NotImplementedError()

    def _format_sh_command_for_log(self, cmd, args=None):
        """Same as _format_sh_command, but hides all data that looks like a password, for correct logging

        :type cmd: str | unicode
        :type args: dict[str | unicode, str | unicode] | None
        :rtype: str | unicode
        """
        return self._format_sh_command(cmd, clean_dict_args_for_log(args))

    def _format_sh_command(self, cmd, args=None):
        """Format command arguments passed for 'sh' and 'sh_unchecked' methods into command line string

        This command line will be added to log and error messages, so it is expected that you could
        simply copy-paste it from logs or migrator output, and execute.

        :type cmd: str | unicode
        :type args: dict[str | unicode, str | unicode] | None
        :rtype: str | unicode
        """
        raise NotImplementedError()
