import logging
import os
import re
from collections import namedtuple
from contextlib import contextmanager
from functools import wraps

from parallels.core.actions.migration_list.utils import get_migration_list_file_location
from parallels.core import MigrationError, MigrationNoContextError
from parallels.core import messages
from parallels.core.logging_context import log_context
from parallels.core.registry import Registry
from parallels.core.utils import unix_utils, plesk_utils, get_thirdparties_base_path
from parallels.core.utils.common import cached, is_ascii_string, is_run_on_windows, safe_format, open_no_inherit
from parallels.core.utils.common.logging import BleepingLogger
from parallels.core.utils.common_constants import SSH_DEFAULT_PORT
from parallels.core.utils.config_utils import get_option
from parallels.core.utils.download_utils import download_zip
from parallels.core.utils.file_utils import files_have_equal_contents
from parallels.core.utils.locks.file_lock_exception import FileLockException
from parallels.core.utils.steps_profiler import get_default_steps_profiler
from parallels.core.utils.unix_utils import format_command

logger = logging.getLogger(__name__)


def copy_vhost_content_unix(
    source_ip, source_user, runner_source, runner_target,
    subscription_name, key_pathname, skip_if_source_dir_not_exists=False, exclude=None
):
    if exclude is None:
        exclude = []

    source_vhost_dir = plesk_utils.get_unix_vhost_dir(runner_source, subscription_name)
    if unix_utils.file_exists(runner_source, source_vhost_dir):
        target_vhost_dir = plesk_utils.get_unix_vhost_dir(runner_target, subscription_name)
        copy_directory_content_unix(
            source_ip, source_user, runner_source, runner_target,
            source_vhost_dir, target_vhost_dir, key_pathname, exclude
        )
    else:
        if skip_if_source_dir_not_exists:
            # source virtual host directory does not exists, so there is nothing to copy 
            return
        else:
            raise MigrationError(
                messages.ERROR_WHILE_COPYING_VIRTUAL_HOSTS_CONTENT % (
                    source_vhost_dir, source_ip
                )
            )


def copy_vhost_system_content_unix(
    source_ip, source_user, runner_source, runner_target, subscription_name, key_pathname
):
    source_vhost_dir = plesk_utils.get_unix_vhost_system_dir(runner_source, subscription_name)
    target_vhost_dir = plesk_utils.get_unix_vhost_system_dir(runner_target, subscription_name)

    copy_directory_content_unix(
        source_ip, source_user, runner_source, runner_target, source_vhost_dir, target_vhost_dir, key_pathname
    )


def copy_directory_content_unix(
    source_ip, source_user, runner_source, runner_target, source_path, target_path, 
    key_pathname, exclude=None, skip_if_source_dir_not_exists=False,
    rsync_additional_args=None,
    user_mapping=None,
    group_mapping=None,
    source_rsync_bin=None, source_port=SSH_DEFAULT_PORT
):
    """
    :type user_mapping: dict[basestring, basestring]
    :type group_mapping: dict[basestring, basestring]
    """
    if exclude is None:
        exclude = []
    if user_mapping is None:
        user_mapping = {}
    if group_mapping is None:
        group_mapping = {}

    if unix_utils.file_exists(runner_source, source_path):
        _copy_content_unix(
            source_ip, source_user, runner_source, runner_target, source_path, target_path,
            key_pathname, content_type='d', exclude=exclude, 
            rsync_additional_args=rsync_additional_args,
            user_mapping=user_mapping,
            group_mapping=group_mapping,
            source_rsync_bin=source_rsync_bin,
            source_port=source_port
        )
    else:
        if skip_if_source_dir_not_exists:
            # source directory does not exists, so there is nothing to copy 
            return
        else:
            raise SourceDirectoryDoesNotExistError(
                safe_format(messages.SOURCE_DIRECTORY_DOES_NOT_EXIST, path=source_path, server_ip=source_ip)
            )


class SourceDirectoryDoesNotExistError(MigrationError):
    pass


DEFAULT_RSYNC_PATH = '/usr/bin/rsync'


def detect_rsync_path(runner, node_info_string):
    if unix_utils.file_exists(runner, DEFAULT_RSYNC_PATH):
        return DEFAULT_RSYNC_PATH
    else:
        raise MigrationError(
            messages.UNABLE_TO_FIND_RSYNC_BINARY % (
                node_info_string, DEFAULT_RSYNC_PATH
            )
        )


def create_rsync_command(
    key_pathname, source_runner, source_user, source_ip, 
    source_filepath, target_filepath, exclude=None,
    rsync_additional_args=None,
    source_rsync_bin=None,
    source_port=SSH_DEFAULT_PORT
):
    if exclude is None:
        exclude = []

    cmd = '/usr/bin/rsync'

    if rsync_additional_args is None:
        rsync_additional_args = []

    args = [ 
        "-e",
        format_command(
            "ssh -i {key} -p {port} "
            "-o PasswordAuthentication=no -o StrictHostKeyChecking=no -o GSSAPIAuthentication=no",
            key=key_pathname,
            port=source_port
        ),
        "--archive"
    ] + rsync_additional_args + [
        u"{source_user}@{source_ip}:{source_filepath}".format(
            source_user=source_user,
            source_ip=source_ip,
            source_filepath=source_filepath
        ),
        u"%s" % (target_filepath,),
    ] + [u"--exclude=%s" % ex for ex in exclude]

    if source_rsync_bin is None:
        # detect rsync binary location on the source node
        source_rsync_bin = detect_rsync_path(
            source_runner, source_ip
        )
        if source_rsync_bin != DEFAULT_RSYNC_PATH: # non-default path
            args = ["--rsync-path=%s" % (source_rsync_bin,)] + args

    return cmd, args


def _copy_content_unix(
    source_ip, source_user, runner_source, runner_target, 
    source_path, target_path, key_pathname, content_type='d',
    exclude=None, rsync_additional_args=None,
    user_mapping=None, group_mapping=None,
    source_rsync_bin=None, source_port=SSH_DEFAULT_PORT
):
    """
    :type user_mapping: dict[basestring, basestring]
    :type group_mapping: dict[basestring, basestring]
    """
    if exclude is None:
        exclude = []
    if user_mapping is None:
        user_mapping = {}
    if group_mapping is None:
        group_mapping = {}

    if content_type == 'd':
        source_path += '/'
        target_path += '/'

    cmd, args = create_rsync_command(
        key_pathname, runner_source, source_user, 
        source_ip, source_path, target_path, exclude,
        rsync_additional_args,
        source_rsync_bin, source_port=source_port
    )
    runner_target.run(cmd, args)

    for source_user, target_user in user_mapping.iteritems():
        logger.debug(messages.CHANGE_FILE_PERMISSIONS_ACCORDING_USER_MAP)
        unix_utils.map_copy_owner_user(
            runner_source, runner_target, source_path, target_path, source_user, target_user
        )
    for source_group, target_group in group_mapping.iteritems():
        logger.debug(messages.CHANGE_FILE_PERMISSIONS_ACCORDING_GROUP_MAP)
        unix_utils.map_copy_owner_group(
            runner_source, runner_target, source_path, target_path, source_group, target_group
        )


def vhost_dir_exists_unix(runner, vhost_name):
    vhost_dir = plesk_utils.get_unix_vhost_dir(runner, vhost_name)
    exit_code, _, _ = runner.run_unchecked("/usr/bin/test",  ["-d", vhost_dir])
    return exit_code == 0


def vhost_system_dir_exists_unix(runner, vhost_name):
    vhost_system_dir = plesk_utils.get_unix_vhost_system_dir(runner, vhost_name)
    exit_code, _, _ = runner.run_unchecked("/usr/bin/test",  ["-d", vhost_system_dir])
    return exit_code == 0


def trace(op_name, message):
    def decorator(method):
        @wraps(method)
        def wrapper(self, *args, **kw):
            with trace_step(op_name, message, profile=True):
                return method(self, *args, **kw)
        return wrapper
    return decorator


@contextmanager
def trace_step(op_name, message, profile=False, log_level='info', compound=True):
    if log_level == 'info':
        log_function = logger.info
    elif log_level == 'debug':
        log_function = logger.debug
    else:
        raise Exception("Invalid log level '%s'" % log_level)

    if compound:
        # put an empty line before the block
        log_function(u"")
        log_function(messages.TRACE_START, message)
    else:
        log_function(message)
    try:
        with log_context(op_name):
            if profile:
                with get_default_steps_profiler().measure_time(
                    op_name, message
                ):
                    yield
            else:
                yield
        if compound:
            log_function(messages.TRACE_FINISH, message)
    except MigrationNoContextError:
        if compound:
            log_function(messages.TRACE_FINISH, message)
        raise
    except:
        logger.error(messages.TRACE_ABORT, message)
        raise


class DbServerInfo(namedtuple('DbServerInfo', ('dbtype', 'host', 'port', 'login', 'password'))):
    @staticmethod
    def from_plesk_api(db_server_info):
        """Convert from plesk_api.DbServerInfo to DbServerInfo"""
        return DbServerInfo(
            dbtype=db_server_info.dbtype,
            host=db_server_info.host,
            port=db_server_info.port,
            login=db_server_info.admin,
            password=db_server_info.password
        )

    @staticmethod
    def from_plesk_backup(db_server_info):
        """Convert plesks_migrator.data_model.DatabaseServer to DbServerInfo"""
        return DbServerInfo(
            dbtype=db_server_info.dbtype,
            host=db_server_info.host,
            port=db_server_info.port,
            login=db_server_info.login,
            password=db_server_info.password
        )


def get_package_root_path(package):
    """Get path to Python package root directory"""
    dirs = [p for p in package.__path__]
    assert all(d == dirs[0] for d in dirs)
    return dirs[0]


def get_package_extras_file_path(package, filename):
    """Get path to extras file of Python package

    :type filename: str | unicode
    :rtype: str | unicode
    """
    return os.path.join(
        get_package_root_path(package), 'extras', filename
    )


def get_package_scripts_file_path(package, filename):
    """Get path to scripts file of Python package

    :type filename: str | unicode
    :rtype: str | unicode
    """
    return os.path.join(
        get_package_root_path(package), 'scripts', filename
    )


def version_to_tuple(version):
    return tuple(map(int, (version.split("."))))


def split_unix_path(path):
    """Split path by directory name and file name

    Returns tuple (directory name, file name)

    :rtype tuple
    """
    last_slash_pos = path.rfind('/')
    directory = path[:last_slash_pos]
    filename = path[last_slash_pos+1:]
    return directory, filename


def normalize_domain_name(domain_name):
    """Normalize domain name - convert to lowercase punycode

    :rtype basestring
    """
    return domain_name.encode('idna').lower()


@contextmanager
def upload_temporary_file_content(runner, filename, content):
    """Upload contents to temporary file on remote server"""
    runner.upload_file_content(filename, content)
    yield
    runner.remove_file(filename)


def get_customer_name_in_report(customer):
    name = customer.login
    if customer.personal_info is not None:
        details = ', '.join(
            x for x in [
                customer.personal_info.first_name,
                customer.personal_info.last_name,
                customer.personal_info.email
            ] if x is not None and x != ''
        )
        if details != "":
            name += " (%s)" % details

    return name


def get_bleeping_logger(logger):
    """Create and return enhanced logger object."""
    default_context_length = 0
    log_context_length = get_option('log-message-context-length', default_context_length)
    return BleepingLogger(logger, context_length=log_context_length)


@cached
def get_version():
    version_file_name = os.path.join(Registry.get_instance().get_base_dir(), 'version')
    if not os.path.exists(version_file_name):
        return 'DEVELOPMENT'
    else:
        return read_string_from_file(version_file_name)


@cached
def get_installation_type():
    installation_type_file_name = os.path.join(Registry.get_instance().get_base_dir(), '.installation-type')
    if not os.path.exists(installation_type_file_name):
        return 'UNKNOWN'
    else:
        return read_string_from_file(installation_type_file_name)


def read_string_from_file(file_name):
    with open_no_inherit(file_name, 'r') as fp:
        return fp.read().strip()


def format_client_login(client_login):
    """Format client login variable into a string to be used in messages

    Considers administrator (for which login is None, according to different models used in migrator)

    :type client_login: str | unicode
    :rtype : unicode
    """
    if client_login is None:
        return u'administrator'
    else:
        return u"customer '%s'" % client_login


def format_owner_login(owner_login):
    """Format login variable of admin/reseller/client into a string to be used in messages

    Considers administrator (for which login is None, according to different models used in migrator)

    :type owner_login: str | unicode
    :rtype: unicode
    """
    if owner_login is None:
        return u'administrator'
    else:
        return u"'%s'" % owner_login


def get_option_value_safe(options, option_name, default_value):
    """Read command line option in a safe way

    If there is no command line option for a command, warning will be emitted and default value returned.
    If you will try to use option directly, migrator will completely fail.
    """
    if hasattr(options, option_name):
        return getattr(options, option_name)
    else:
        logger.warning(messages.NO_COMMAND_LINE_OPTION_WARNING.format(option_name=option_name))
        return default_value


def get_optional_option_value(options, option_name, default_value):
    """Read command line option, which could exist or not exist

    If there is no command line option for a command, default value returned.
    """
    if hasattr(options, option_name):
        return getattr(options, option_name)
    else:
        return default_value


def is_locked():
    result = True
    migration_lock = Registry.get_instance().get_migration_lock()
    try:
        migration_lock.acquire()
        migration_lock.release()
        result = False
    except FileLockException:
        # assume that command is still in progress
        pass
    return result


def safe_idn_decode(domain_name):
    """Safely decode domain name from punycode to unicode

    If domain name is already in unicode - return it as is. If any error occured during decoding - return domain
    name as is.

    :type domain_name: str | unicode
    :rtype: unicode
    """
    if not is_ascii_string(domain_name):
        return domain_name

    try:
        return domain_name.decode('idna')
    except:
        logger.debug(messages.LOG_EXCEPTION, exc_info=True)
        logger.warning(messages.UNABLE_TO_CONVERT_IDN_NAME.format(domain_name=domain_name))
        return domain_name


def is_unicode_domain(domain_name):
    """Whether specified domain is in unicode or in punycode (IDNA). For unicode form - return true.

    :rtype: bool
    """
    return domain_name.encode('idna') != domain_name


def safe_mail_idn_decode(email_address):
    """
    :type email_address: str | unicode
    :rtype: str | unicode
    """
    email_address_parts = email_address.split('@')
    if len(email_address_parts) != 2:
        return email_address
    mail_name, domain_name = email_address_parts
    return '@'.join([mail_name, safe_idn_decode(domain_name)])


def path_startswith(path1, path2):
    path1_items = [i for i in re.split('[/\\\\]', path1) if i != '']
    path2_items = [i for i in re.split('[/\\\\]', path2) if i != '']
    return path1_items == path2_items[:len(path1_items)]


def secure_write_open(filename, is_replace_existing=True):
    """Open file for writing. On Linux, if file does not exist - create it with secure permissions (600)

    :type filename: str | unicode
    """
    mode = 'wb' if is_replace_existing else 'ab'
    if is_run_on_windows():
        return open_no_inherit(filename, mode)
    else:
        if not os.path.exists(filename):
            return os.fdopen(os.open(filename, os.O_WRONLY | os.O_CREAT, 0o600), 'w')
        else:
            # if file already exists - we consider that it already has correct permissions
            return open(filename, mode)


def is_running_from_cli():
    """Check if migrator is running from command-line interface (comparing to GUI)

    :rtype: bool
    """
    messages_ui = os.getenv('MIGRATOR_MESSAGES_UI')
    return messages_ui != 'gui'


def ip_file_path(ip_address):
    """Get file path path which corresponds to specified IP address,

    Function replaces symbols which are valid in IP address, but not valid in filesystem path

    :type ip_address: str | unicode
    :rtype: str | unicode
    """
    return ip_address.replace(':', '_').replace('.', '_')


def download_thirdparty_zip(remote_name, local_name, log_message):
    """Download thirdparty ZIP archive and unpack to a directory

    If it was already downloaded before - skip.
    Returns path to a local directory with thirdparty.

    :type remote_name: str | unicode
    :type local_name: str | unicode
    :type log_message: str | unicode
    :rtype: str | unicode
    """
    dir_path = os.path.join(get_thirdparties_base_path(), local_name)

    if not os.path.exists(dir_path):
        url = 'http://autoinstall.plesk.com/panel-migrator/thirdparties/%s' % remote_name
        logger.info(log_message.format(url=url))
        download_zip(url, dir_path)

    return dir_path


def is_cache_enabled_and_valid(global_context, cache_migration_list_file):
    """Whether caching is enabled and cache is valid for different files and models like converted dump, etc

    By default caching is disabled for 'transfer-accounts' and 'check' commands, but enabled for post-migration
    commands like 'copy-content', 'copy-web-content', 'test-sites', etc. It allows to run post-migration
    operations fast once migration was performed, but keep data up to date when running migration itself.
    Caching is performed by migration list: if migration list file changed (even by single whitespace),
    then cache is considered invalid.

    Cache migration list file is a file which contains contents of migration list file by the moment when cache
    was updated last time.

    :type global_context: parallels.core.global_context.GlobalMigrationContext
    :type cache_migration_list_file: str | unicode
    :rtype: bool
    """
    if not _is_caching_enabled(global_context):
        return False

    migration_list_file = get_migration_list_file_location(global_context)
    if migration_list_file is None:
        return False

    if not os.path.exists(cache_migration_list_file):
        return False

    if not os.path.exists(migration_list_file):
        return False

    if not files_have_equal_contents(migration_list_file, cache_migration_list_file):
        return False

    return True


def _is_caching_enabled(global_context):
    """Whether caching is enabled for different files and models like converted dump, existing objects model, etc

    By default caching is disabled for 'transfer-accounts' and 'check' commands, but enabled for post-migration
    commands like 'copy-content', 'copy-web-content', 'test-sites', etc.

    :type global_context: parallels.core.global_context.GlobalMigrationContext
    :rtype: bool
    """
    if hasattr(global_context.options, 'use_cache'):
        return global_context.options.use_cache
    else:
        return False
