from parallels.core import messages
from parallels.core.utils import config_utils
from parallels.core.utils.common.logging import create_safe_logger
from parallels.core.utils.migrator_utils import get_optional_option_value

logger = create_safe_logger(__name__)


class BaseWebFiles(object):
    """Base interface to list web files and directories to be copied from source panel"""

    def need_to_copy_files(self, global_context, subscription):
        """Check whether we need to copy files of that subscription

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :rtype: bool
        """
        source_server = subscription.web_source_server
        target_server = subscription.web_target_server

        # Condition #1: source and target servers are defined
        if source_server is None or target_server is None:
            return False

        # Condition #2: we have copy web content option enabled in configuration file, or disabled by CLI option
        webcontent_transfer_option = config_utils.get_option('copy-web-content', 'full', source_server.node_id)
        config_skip_copy_web_content = webcontent_transfer_option != 'full'
        cli_skip_copy_web_content = get_optional_option_value(
            global_context.options, 'skip_copy_web_content', False
        )
        if config_skip_copy_web_content or cli_skip_copy_web_content:
            logger.info(messages.LOG_NO_NEED_TO_COPY_WEBCONTENT_BECAUSE_DISABLED_BY_USER, subscription.name)
            return False

        # Condition #3: subscription has virtual hosting
        if subscription.raw_dump.hosting_type != 'phosting':
            return False

        return True

    def list_files_to_copy(self, global_context, subscription):
        """Make a list of source server directories and files to be transferred.

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :rtype: list[parallels.core.utils.paths.copy_web_content.CopyWebContentItem]
        """
        raise NotImplementedError()

    def get_source_vhosts_dir(self, global_context, source_server):
        """Get directory where virtual hosts data is located.

        This function could return None when we don't know where virtual hosts directory is located,
        for example if we don't have direct access to the source server, but have only manually configured
        rsync server.

        :type source_server: parallels.core.connections.source_server.SourceServer
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :rtype: str | unicode | None
        """
        return source_server.vhosts_dir
