import ntpath
from xml.etree import ElementTree
from parallels.core.utils.common import cached
from parallels.core.utils.plesk_utils import query_plesk_db


def get_installed_components(plesk_server):
    """Get list of components installation status in simple format

    For now only Perl and Python are supported. Extend if necessary.
    Example result: {'perl': True, 'python': False}

    :rtype: dict[str, bool]
    """
    components = {}

    if plesk_server.is_windows():
        with plesk_server.runner() as runner:
            packages_xml_str = runner.sh(
                ur'{defpackagemng_bin} --get',
                dict(defpackagemng_bin=ntpath.join(plesk_server.plesk_dir, ur'admin\bin\defpackagemng'))
            )
            packages_xml = ElementTree.fromstring(packages_xml_str.encode('utf-8'))
            for elem in packages_xml.findall('type'):
                if elem.attrib.get('name') == 'script.python':
                    name = 'python'
                elif elem.attrib.get('name') == 'script.perl':
                    name = 'perl'
                else:
                    name = None

                if name is not None:
                    installed = False
                    package_node = elem.find('package')
                    if package_node is not None:
                        installed = package_node.attrib.get('state') != 'not_installed'

                    components[name] = installed
    else:
        query_results = query_plesk_db(
            plesk_server,
            "SELECT name FROM ServiceNodeEnvironment "
            "WHERE section = 'componentsPackages' AND name in ('mod_python', 'mod_perl') "
            "AND value != ''"
        )
        raw_components = {row[0] for row in query_results if len(row) > 0}
        components['python'] = 'mod_python' in raw_components
        components['perl'] = 'mod_perl' in raw_components

    return components


@cached
def get_available_components(plesk_server):
    """Get list of components that are installed or can be installed

    Now works for Unix only.

    :rtype: set[str | unicode]
    """

    query_results = query_plesk_db(
        plesk_server,
        "SELECT name FROM ServiceNodeEnvironment WHERE section = 'componentsPackages'"
    )
    return {row[0] for row in query_results if len(row) > 0}
