from xml.etree import ElementTree
from collections import namedtuple
from contextlib import closing
import os
import logging

from parallels.core import messages
from parallels.core.dump.dump_archive_writer import DumpArchiveWriter
from parallels.core.utils.common.xml import xml_to_string_pretty
from parallels.core.utils.yaml_utils import pretty_yaml
from parallels.core.utils.common import cached, open_no_inherit

logger = logging.getLogger(__name__)


class DumpAll(namedtuple('DumpAll', [])):
    # noinspection PyUnusedLocal
    @staticmethod
    def covers(selection):
        """Check if this selection covers another selection

        If true, that means that backup dump created by current
        selection contains all objects from another selection, and so
        could be used.
        :type selection: parallels.core.utils.pmm.agent.DumpAll|parallels.core.utils.pmm.agent.DumpSelected
        :rtype: bool
        """
        return True

    @staticmethod
    def get_filter_xml():
        return None

    # noinspection PyUnusedLocal
    @staticmethod
    def is_reseller_selected(name):
        """
        :type name: str|unicode
        :rtype: bool
        """
        return True

    # noinspection PyUnusedLocal
    @staticmethod
    def is_client_selected(name):
        """
        :type name: str|unicode
        :rtype: bool
        """
        return True

    # noinspection PyUnusedLocal
    @staticmethod
    def is_domain_selected(name):
        """
        :type name: str|unicode
        :rtype: bool
        """
        return True

    def __repr__(self):
        return 'DumpAll()'


DumpAll = pretty_yaml(DumpAll)


class DumpSelected(namedtuple('DumpSelected', ('resellers', 'clients', 'domains'))):
    def covers(self, selection):
        """Check if this selection covers another selection

        If true, that means that backup dump created by current
        selection contains all objects from another selection, and so
        could be used.
        :type selection: parallels.core.utils.pmm.agent.DumpAll|parallels.core.utils.pmm.agent.DumpSelected
        :rtype: bool
        """
        if isinstance(selection, DumpSelected):
            return (
                set(self.resellers) >= set(selection.resellers) and
                set(self.clients) >= set(selection.clients) and
                set(self.domains) >= set(selection.domains)
            )
        else:
            return False

    def get_filter_xml(self):
        migration_objects_list = ElementTree.Element("migration-objects-list")

        resellers = ElementTree.SubElement(migration_objects_list, "resellers")
        for reseller in self.resellers:
            ElementTree.SubElement(resellers, "reseller").text = reseller
        clients = ElementTree.SubElement(migration_objects_list, "clients")
        for client in self.clients:
            ElementTree.SubElement(clients, "client").text = client
        domains = ElementTree.SubElement(migration_objects_list, "domains")
        for domain in self.domains:
            ElementTree.SubElement(domains, "domain").text = domain

        return xml_to_string_pretty(ElementTree.ElementTree(migration_objects_list))

    def is_reseller_selected(self, name):
        """
        :type name: str|unicode
        :rtype: bool
        """
        return name in self.resellers

    def is_client_selected(self, name):
        """
        :type name: str|unicode
        :rtype: bool
        """
        return name in self.clients

    def is_domain_selected(self, name):
        """
        :type name: str|unicode
        :rtype: bool
        """
        return name in self.domains

    def __repr__(self):
        return "DumpSelected(%r, %r, %r)" % (
            self.resellers, self.clients, self.domains
        )

DumpSelected = pretty_yaml(DumpSelected)


class PmmMigrationAgentBase(object):

    def __init__(self, global_context, source_server):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type source_server: parallels.plesk.source.plesk.server.PleskSourceServer
        """
        self.global_context = global_context
        self._source_server = source_server
        self.agent_dir = self._deploy()

    def source_server(self):
        """
        :rtype: parallels.plesk.source.plesk.server.PleskSourceServer
        """
        return self._source_server

    def create_dump(self, filename, is_archive_dump=False, selection=DumpAll()):
        dump_file = filename if not is_archive_dump else self.dump_file

        logger.debug(messages.RUN_PMM_AGENT_CREATE_DUMP_XML)
        self._run(dump_file, self.configuration_dump_log, selection)

        if is_archive_dump:
            logger.debug(messages.PACK_DUMP_XML_INTO_DUMP_ARCHIVE)
            with open_no_inherit(dump_file) as dump_xml_fp, closing(DumpArchiveWriter(filename)) as archive_writer:
                archive_writer.add_dump(dump_xml_fp.read())

    def create_shallow_dump(self, filename):
        logger.debug(messages.RUN_PMM_AGENT_CREATE_SHALLOW_DUMP)
        self._run_shallow(filename)

    def create_capability_dump(self, filename, selection=DumpAll()):
        logger.debug(messages.RUN_PMM_AGENT_CREATE_CAPABILITY_DUMP)
        self._run_capability(filename, self.capability_dump_log, selection)

    def check(self):
        raise NotImplementedError()

    def execute_sql(self, query, query_args=None, log_output=True):
        """Execute SQL query on panel database
        Returns list of rows, each row is a dictionary.
        Query and query arguments:
        * execute_sql("SELECT * FROM clients WHERE type=%s AND country=%s", ['client', 'US'])

        :type query: str|unicode
        :type query_args: list|dict
        :type log_output: bool
        :rtype: list[dict]
        """
        raise NotImplementedError()

    def _run(self, dump_xml_filename, dump_log_filename, selection=DumpAll()):
        raise NotImplementedError()

    def _run_shallow(self, dump_xml_filename):
        raise NotImplementedError()

    def _run_capability(self, dump_xml_filename, dump_log_filename, selection=DumpAll()):
        raise NotImplementedError()

    def _deploy(self):
        raise NotImplementedError()

    def uninstall(self):
        raise NotImplementedError()

    @cached
    def _get_pmm_session_dir(self):
        directory = self.global_context.session_files.get_path_to_pmm_agent_dir()
        with self.global_context.migrator_server.runner() as runner:
            runner.mkdir(directory)
        return directory

    @property
    def dump_file(self):
        return os.path.join(self._get_pmm_session_dir(), 'dump.xml')

    @property
    def configuration_dump_log(self):
        return os.path.join(self._get_pmm_session_dir(), 'configuration-dump.log')

    @property
    def capability_dump_log(self):
        return os.path.join(self._get_pmm_session_dir(), 'capability-dump.log')

    @property
    def check_report_file(self):
        return os.path.join(self._get_pmm_session_dir(), 'pre-migration.xml')

    @property
    def check_log_file(self):
        return os.path.join(self._get_pmm_session_dir(), 'pre-migration.log')
