"""Module with filters to filter subscriptions we are migrating"""

from parallels.core.registry import Registry
from parallels.core.migrator_config import MailContent
from parallels.core.utils.common import exists


def copy_mail(subscription):
    """Filter subscriptions for which we are going to copy mail

    Arguments:
    - subscription - instance of MigrationSubscription
    """
    source_server = subscription.mail_source_server
    target_server = subscription.mail_target_server

    if source_server is None:
        return False
    if target_server is None:
        return False
    if source_server.mail_settings.mode == MailContent.NONE:
        return False

    return True


def copy_mail_full(subscription):
    """Filter subscriptions for which we are going to copy mail in full mode

    Full mode usually means usage of imapsync or something else
    which copies all customer's mail messages 

    Arguments:
    - subscription - instance of MigrationSubscription
    """
    source_server = subscription.mail_source_server
    return (
        copy_mail(subscription) and
        source_server.mail_settings.mode == MailContent.FULL
    )


def copy_mail_messages(subscription):
    """Filter subscriptions for which we are going to copy mail in messages mode

    Full mode usually means usage of imapsync or something else
    which copies all customer's mail messages 

    Arguments:
    - subscription - instance of MigrationSubscription
    """
    source_server = subscription.mail_source_server
    return (
        copy_mail(subscription) and
        source_server.mail_settings.mode == MailContent.MESSAGES
    )


def windows_with_virtual_hosting(subscription):
    """Filter Windows subscriptions which have virtual hosting

    :type subscription: parallels.core.migrated_subscription.MigratedSubscription
    :rtype: bool
    """
    if subscription.web_target_server is None:
        # subscription w/o web cannot be a virtual hosting subscription
        return False

    if not subscription.web_target_server.is_windows():
        # target web server is unix, so all migrated subscriptions become unix subscriptions
        return False

    return subscription.raw_dump.hosting_type == 'phosting'


def unix_with_virtual_hosting(subscription):
    """Filter Unix subscriptions which have virtual hosting

    :type subscription: parallels.core.migrated_subscription.MigratedSubscription
    :rtype: bool
    """
    if subscription.web_target_server is None:
        # subscription w/o web cannot be a virtual hosting subscription
        return False

    if subscription.web_target_server.is_windows():
        # target web server is windows, so all migrated subscriptions become windows subscriptions
        return False

    return subscription.raw_dump.hosting_type == 'phosting'


def windows_with_iis_dedicated_app_pool_enabled(subscription):
    """Filter Windows subscriptions with dedicated IIS application pool

    Arguments:
    - subscription - instance of MigrationSubscription
    """
    return (
        windows_with_virtual_hosting(subscription) and
        subscription.raw_dump.dedicated_iis_application_pool_enabled
    )


def windows_with_iis_dedicated_app_pool_disabled(subscription):
    """Filter Windows subscriptions with shared IIS application pool

    Arguments:
    - subscription - instance of MigrationSubscription
    """
    return (
        windows_with_virtual_hosting(subscription) and
        not subscription.raw_dump.dedicated_iis_application_pool_enabled
    )


def reset_permissions(subscription, global_context):
    """
    :type subscription: parallels.core.migrated_subscription.MigratedSubscription
    :type global_context: parallels.core.global_context.GlobalMigrationContext
    :rtype: bool
    """
    if not exists(
        global_context.target_existing_objects.subscriptions,
        lambda s: s.name_canonical == subscription.name_canonical
    ):
        return True
    state_vars = Registry.get_instance().get_migration_state().get_vars(
        action='copy-security/reset-permissions',
        subscription=subscription.name
    )
    return not state_vars.get('executed', False)
