from parallels.core import messages
from parallels.core.actions.base.subscription_backup_action import SubscriptionBackupAction
from parallels.core.actions.utils.logging_properties import LoggingProperties
from parallels.core.utils.plesk_utils import get_server_info_with_key


class RemoveLimitsAndPermissions(SubscriptionBackupAction):
    """Remove all limits and permissions of subscription when migrating to Plesk Web Admin and Web App editions

    We need to remove limits and permissions because in these editions you can't even manage it.
    If they are migrated from the source, customer can not see them on target,
    and cannot change them on target from GUI. So, for example if expiration date was set on the source server,
    subscription will get inevitably suspended. Or, if disk domains limit was set to 3 domains, customer
    will never be able to create more than 3 domains from GUI.
    """
    def get_description(self):
        """Get short description of action as string

        :rtype: str
        """
        return messages.ACTION_REMOVE_LIMITS_AND_PERMISSIONS_DESCRIPTION

    def get_failure_message(self, global_context, subscription):
        """Get message for situation when action failed

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :rtype: basestring
        """
        return messages.ACTION_REMOVE_LIMITS_AND_PERMISSIONS_FAILURE

    def is_critical(self):
        """If action is critical or not

        If action is critical and it failed for a subscription, migration tool
        won't run the next operations for the subscription. This one is not critical,
        we could safely continue migration
        """
        return False

    def get_logging_properties(self):
        """Get how action should be logged to migration tools end-user

        We don't want this action to be logged to info log.

        :rtype: parallels.core.actions.utils.logging_properties.LoggingProperties
        """
        return LoggingProperties(info_log=False)

    def _filter_subscription_backup(self, global_context, subscription, subscription_backup):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        # Here we detect Web Admin and Web App editions of Plesk - if target server has such license,
        # then this action should run
        server_info = get_server_info_with_key(global_context.conn.target)
        can_manage_accounts = server_info.key.properties.get('can-manage-accounts', '1')
        return can_manage_accounts == '0'

    def _run_subscription_backup(self, global_context, subscription, subscription_backup):
        """Run action on single subscription on provided backup

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        subscription_backup.remove_limit_and_permissions()