from parallels.core.utils.common import is_empty, cached
from parallels.plesk.hosting_repository.utils.cli.base import BaseCli


class ResellerPrefUpdateCli(BaseCli):
    def __init__(self, runner, reseller, permissions_descriptor, limits_descriptor):
        """Construct command to update reseller preferences in Plesk via CLI

        :type runner: parallels.core.utils.plesk_cli_runner.PleskCLIRunnerBase
        :type reseller: parallels.core.target_data_model.Reseller
        :type permissions_descriptor: parallels.plesk.utils.xml_rpc.plesk.core.Descriptor
        :type limits_descriptor: parallels.plesk.utils.xml_rpc.plesk.core.Descriptor
        """
        super(ResellerPrefUpdateCli, self).__init__(runner)
        self._reseller = reseller
        self._permissions_descriptor = permissions_descriptor
        self._limits_descriptor = limits_descriptor
        self._pref_args = None

    @property
    def name(self):
        return 'reseller_pref'

    def _get_args(self):
        return ['--update', self._reseller.login] + self._get_pref_args()

    @cached
    def _get_pref_args(self):
        pref_args = []
        pref_args_map = {'max_odbc': 'max_odbc_dsn'}

        def add_pref_arg(_name, _value):
            if not is_empty(_value):
                pref_args.extend([_name, _value])

        if self._reseller.settings is not None:
            add_pref_arg('-overuse', self._reseller.settings.overuse)
            add_pref_arg('-oversell', self._reseller.settings.oversell)

        if self._reseller.settings is not None:
            for name, value in self._reseller.settings.permissions.iteritems():
                if self._permissions_descriptor.is_writable(name):
                    add_pref_arg('-%s' % pref_args_map.get(name, name), value)
            for name, value in self._reseller.settings.limits.iteritems():
                if self._limits_descriptor.is_writable(name):
                    if (
                        not is_empty(value) and
                        value != '-1' and
                        self._limits_descriptor.properties[name].value_type == 'bytes'
                    ):
                        value = '%sB' % value
                    add_pref_arg('-%s' % pref_args_map.get(name, name), value)

        return pref_args

    def _is_useless(self):
        return len(self._get_pref_args()) == 0


class ResellerPrefUpdateLocaleCli(BaseCli):
    def __init__(self, runner, username, locale):
        """Construct command to update locale of reseller with given username in Plesk via CLI

        :type runner: parallels.core.utils.plesk_cli_runner.PleskCLIRunnerBase
        :type username: str
        :type locale: str
        """
        super(ResellerPrefUpdateLocaleCli, self).__init__(runner)
        self._username = username
        self._locale = locale

    @property
    def name(self):
        return 'reseller_pref'

    def _get_args(self):
        return [
            '--update', self._username,
            '-locale', self._locale
        ]
