import logging

from parallels.plesk.source.confixx import messages
from parallels.core.dump.data_model import DatabaseServer
from parallels.plesk.source.confixx.content.mail import ConfixxCopyMailContent
from parallels.plesk.source.confixx.web_files import ConfixxWebFiles
from parallels.plesk.source.plesk.infrastructure_checks import checks as infrastructure_checks
from parallels.plesk.source.confixx import connections
from parallels.plesk.source.legacy.migrator import PmmUnixMigrator
from parallels.plesk.source.confixx.global_context import ConfixxGlobalMigrationContext


class Migrator(PmmUnixMigrator):
    logger = logging.getLogger(__name__)

    def _load_connections_configuration(self, global_context, target_panel_type):
        return connections.MigratorConnections(global_context, self._get_target_panel_by_name(target_panel_type))

    def _create_global_context(self):
        return ConfixxGlobalMigrationContext()

    @property
    def web_files(self):
        """Object to list files to be transferred from source to target

        :rtype: parallels.core.utils.paths.web_files.BaseWebFiles
        """
        return ConfixxWebFiles()

    # ======================== copy mail content ==============================

    def copy_mail_content_single_subscription(self, subscription, issues):
        copy_mail = ConfixxCopyMailContent(
            self.global_context.confixx_configuration,
            agent_dir=self.global_context.dump_agent.agent_dir,
            mailmapping_tool='mailmapping.pl'
        )
        return copy_mail.copy_mail(self.global_context, subscription, issues)

    # ======================== databases =====================================

    def _get_src_db_server(self, database_server_host, database_server_port, database_server_type, server_dump):
        configuration = self.global_context.confixx_configuration
        return DatabaseServer(
            host='localhost',
            port='3306',
            login=configuration.get_mysql_user(),
            password=configuration.get_mysql_password(),
            dbtype='mysql',
            is_default=True,
        )

    # ======================== infrastructure checks ==========================

    def _get_check_service_connection_commands(self):
        return {
            'web': self._check_unix_copy_web_content_rsync_connections,
            'mail': self._check_unix_copy_mail_content_rsync_connections,
            'db': self._check_unix_copy_db_content_scp_connections
        }

    def _check_infrastructure_connections(self, report, safe):
        self.logger.info(messages.LOG_CHECK_CONNECTION_REQUIREMENTS)

        for service in ('web', 'mail', 'db'):
            self._check_connection(service, report, safe)

    def _check_connection(self, service, report, safe):
        service_checkers = self._get_check_service_connection_commands()
        report_messages = {
            'title':
                messages.REPORT_TITLE_INFRASTRUCTURE_CHECKS_CONNECTIONS,
            'error':
                messages.FAILED_TO_CHECK_CONNECTIONS}
        report = report.subtarget(report_messages['title'] % service, None)
        checks = infrastructure_checks.InfrastructureChecks()
        with safe(report, report_messages['error'] % service):
            service_checkers[service](checks, report)

    def _check_disk_space(self, report, safe):
        self.logger.info(messages.LOG_CHECK_DISK_SPACE_REQUIREMENTS)
        disk_space_report = report.subtarget(messages.REPORT_TARGET_DISK_SPACE_REQUIREMENTS, None)
        super(Migrator, self)._check_disk_space_unix(disk_space_report)
