import logging

from parallels.core.actions.base.subscription_action import SubscriptionAction
from parallels.core.actions.content.web.copy_unix_content_base import UnixTargetWebPathConverter
from parallels.core.utils.paths import web_paths
from parallels.plesk.source.custom import messages

logger = logging.getLogger(__name__)


class CustomUnixFixPermissions(SubscriptionAction):
    """Fix permissions on document root directories for subscription, subdomains and addon domains"""
    def get_description(self):
        """
        :rtype: basestring
        """
        return messages.FIX_WEB_CONTENT_PERMISSIONS_ACTION_DESCRIPTION

    def get_failure_message(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        return messages.FIX_WEB_CONTENT_PERMISSIONS_ACTION_FAILURE

    def is_critical(self):
        """If action is critical or not

        If action is critical and it failed for a subscription, migration tool
        won't run the next operations for the subscription.

        :rtype: bool
        """
        return False

    def filter_subscription(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        # skip this action for subscriptions w/o web and for windows subscriptions
        if subscription.web_target_server is None or subscription.web_target_server.is_windows():
            return False

        return global_context.migrator.web_files.need_to_copy_files(global_context, subscription)

    def run(self, global_context, subscription):
        """Run action on given subscription

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.plesk.source.custom.migrated_subscription.CustomPanelMigratedSubscription
        """
        server = subscription.web_target_server
        system_user = subscription.converted_dump.phosting_sysuser_name
        with server.runner() as runner:
            target_path = UnixTargetWebPathConverter().expand(
                web_paths.WebspaceDocumentRoot(subscription.converted_dump), server
            )
            self._fix_path_document_root_permissions(runner, system_user, target_path)
            for site in subscription.converted_dump.iter_sites():
                if site.is_virtual_hosting:
                    target_path = UnixTargetWebPathConverter().expand(
                        web_paths.SiteDocumentRoot(subscription.converted_dump, site), server
                    )
                    self._fix_path_document_root_permissions(runner, system_user, target_path)

    @staticmethod
    def _fix_path_document_root_permissions(runner, system_user, target_path):
        """Fix permissions on document root located in specified path

        :type system_user: str | unicode
        :type target_path: str | unicode
        """
        runner.sh('chown -R {user}:{group} {path}', args=dict(
            user=system_user,
            group='psacln',
            path=target_path
        ))
        runner.sh('chgrp {group} {path}', args=dict(
            group='psaserv',
            path=target_path
        ))
