from xml.etree import ElementTree
import logging
import ntpath
import os

from parallels.core.actions.base.subscription_action import SubscriptionAction
from parallels.core.utils import subscription_filter
from parallels.core.utils.common import safe_format
from parallels.plesk.source.plesk import messages
from parallels.plesk.source.plesk.actions.content.web.utils import list_security_files

logger = logging.getLogger(__name__)


class CreateSecuredWebContent(SubscriptionAction):
    """Create files and directories that have customized Windows permissions.

    Operation of permissions update in Windows is too slow if there are a lot
    of folders. For example, update permissions of webspace with ~50000 folders
    takes ~1 hour. So we should create files and directories and apply customized
    permissions before web-content coping.
    """

    def get_description(self):
        return messages.ACTION_CREATE_SECURED_WEB_CONTENT_DESCRIPTION

    def get_failure_message(self, global_context, subscription):
        return messages.ACTION_CREATE_SECURED_WEB_CONTENT_FAILURE

    def is_critical(self):
        """If action is critical or not

        If action is critical and it failed for a subscription, migration tool
        won't run the next operations for the subscription.

        :rtype: bool
        """
        return False

    def filter_subscription(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :rtype: bool
        """
        return (
            subscription_filter.windows_with_virtual_hosting(subscription) and
            # Check that ResetPermissions action is already executed. We prevent repeat runs
            # of ResetPermissions action to improve performance and it is not sense to run
            # this action if ResetPermissions will be not executed.
            subscription_filter.reset_permissions(subscription, global_context)
        )

    def run(self, global_context, subscription):
        """
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        vhost_name = subscription.name_idn
        source_server = subscription.web_source_server
        target_server = subscription.web_target_server
        processed_entries = set()
        with source_server.runner() as source_runner, target_server.runner() as target_runner:
            source_vhosts_dir = source_server.vhosts_dir
            target_vhosts_dir = target_server.vhosts_dir
            for i, security_file in enumerate(list_security_files(subscription, include_subdomains=True)):
                fixed_security_file = global_context.migrator_server.get_session_file_path(
                    'windows-security-%s-%i.fixed.xml' % (vhost_name, i)
                )
                if not os.path.exists(fixed_security_file):
                    continue

                logger.debug(safe_format(messages.LOG_CREATE_SECURED_WEB_CONTENT, path=fixed_security_file))
                security_file_tree = ElementTree.parse(fixed_security_file)
                for entry_node in security_file_tree.findall('Entry'):
                    entry_sub_path = entry_node.get('SubPath')
                    if not entry_sub_path:
                        continue
                    entry_source_path = ntpath.join(source_vhosts_dir, entry_sub_path)
                    if entry_source_path in processed_entries:
                        continue
                    processed_entries.add(entry_source_path)
                    entry_is_dir = source_runner.is_dir(entry_source_path)
                    if entry_is_dir is None:
                        continue
                    entry_target_path = ntpath.join(target_vhosts_dir, entry_sub_path)
                    if entry_is_dir:
                        target_runner.mkdir(entry_target_path)
                        continue
                    if target_runner.file_exists(entry_target_path):
                        continue
                    target_runner.mkdir(ntpath.dirname(entry_target_path))
                    target_runner.upload_file_content(entry_target_path, '')
