import logging

from parallels.core import MigrationNoContextError
from parallels.core.utils.common import format_multiline_list
from parallels.core.actions.base.common_action import CommonAction
from parallels.plesk.hosting_description.validate.mode import ValidationMode
from parallels.plesk.hosting_description.converter_to_dump import read_hosting_description_file
from parallels.plesk.hosting_description.validate.overall_validator import HostingDescriptionValidator
from parallels.plesk.source.web import messages

logger = logging.getLogger(__name__)


class ValidateHostingDescriptionFiles(CommonAction):
    """Validate all hosting description files"""

    def get_failure_message(self, global_context):
        """Get message for situation when action failed

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :rtype: basestring
        """
        return messages.ACTION_VALIDATE_HOSTING_DESCRIPTION_FAILED

    def get_description(self):
        """Get short description of action as string

        :rtype: basestring
        """
        return messages.ACTION_VALIDATE_HOSTING_DESCRIPTION_DESCRIPTION

    def run(self, global_context):
        """Run action

        :type global_context: parallels.plesk.source.web.global_context.CustomPanelGlobalMigrationContext
        :rtype: None
        """
        is_config_has_hosting_description = False

        for hosting_description_config in global_context.conn.iter_hosting_description_configs():
            is_config_has_hosting_description = True
            if hosting_description_config.validation_mode == ValidationMode.SKIP:
                continue

            hosting_description_data = read_hosting_description_file(hosting_description_config)
            errors = HostingDescriptionValidator().validate(hosting_description_data)

            if len(errors) > 0:
                if hosting_description_config.validation_mode == ValidationMode.STOP:
                    error_message = messages.VALIDATION_FAILED_MIGRATION_STOPPED.format(
                        file=hosting_description_config.path, errors=format_multiline_list(errors)
                    )
                    raise MigrationNoContextError(error_message)
                elif hosting_description_config.validation_mode == ValidationMode.WARN:
                    error_message = messages.VALIDATION_FAILED.format(
                        file=hosting_description_config.path, errors=format_multiline_list(errors)
                    )
                    logger.warning(error_message)
            else:
                logger.info(messages.VALIDATION_PASSED.format(file=hosting_description_config.path))

        if not is_config_has_hosting_description:
            raise MigrationNoContextError(messages.ACTION_VALIDATE_HOSTING_DESCRIPTION_NOT_SPECIFIED)
